package ru.yandex.direct.grid.processing.service.aggregatedstatuses;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Streams;
import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatuses;
import ru.yandex.direct.core.entity.aggregatedstatuses.AggregatedStatusBaseData;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdEntityLevel;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.AggregatedStatusRetargetingData;
import ru.yandex.direct.core.entity.banner.BannerAdGroupRelation;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonRequest;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonWithDetails;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonText;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasons;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.dbutil.ShardByClient;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdModerationDiagData;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdNewStatusPopupInfo;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdPopupEntityEnum;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdStatusAdgroupPopupInfo;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdStatusCounter;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdStatusPopupInfo;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;

/**
 * Сервис отдельных graphQl ручек для работы с агрегированными статусами
 */

@GridGraphQLService
@ParametersAreNonnullByDefault
public class AggregatedStatusesGraphQlService {
    private final AggregatedStatuses aggregatedStatusesViewService;
    private final ShardByClient shardHelper;
    private final BannerAdGroupRelation bannerRelationsRepository;
    private final ModerationReasons moderationReasons;
    private final ModerationReasonText moderationReasonTextService;

    @Autowired
    public AggregatedStatusesGraphQlService(AggregatedStatuses aggregatedStatusesViewService,
                                            ShardByClient shardHelper,
                                            BannerAdGroupRelation bannerRelationsRepository,
                                            ModerationReasons moderationReasons,
                                            ModerationReasonText moderationReasonTextService) {
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.shardHelper = shardHelper;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.moderationReasons = moderationReasons;
        this.moderationReasonTextService = moderationReasonTextService;
    }

    /**
     * Информация для отображения попапа объсняющего статус кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "campaignStatusPopup")
    @PreAuthorizeRead
    public GdStatusPopupInfo campaignStatusPopup(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "id") Long campaignId) {
        ClientId clientId = context.getSubjectUser().getClientId();
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, AggregatedStatusAdGroupData> adgroupStatuses =
                aggregatedStatusesViewService.getAdGroupStatusesByCampaignId(shard, clientId, campaignId);
        return getGdStatusPopupInfo(GdPopupEntityEnum.CAMPAIGN, GdEntityLevel.CAMPAIGN, adgroupStatuses, Map.of());
    }

    /**
     * Информация для отображения попапа объсняющего статус группы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "adgroupStatusPopup")
    @PreAuthorizeRead
    public GdStatusAdgroupPopupInfo adgroupStatusPopup(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "id") Long adgroupId) {
        ClientId clientId = context.getSubjectUser().getClientId();
        int shard = shardHelper.getShardByClientId(clientId);

        Map<Long, AggregatedStatusAdData> adStatuses =
                aggregatedStatusesViewService.getAdStatusesByAdgroupIds(shard, clientId, List.of(adgroupId));
        GdStatusPopupInfo adsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.BANNER, GdEntityLevel.BANNER, adStatuses,
                Map.of());

        Map<Long, AggregatedStatusKeywordData> keywordStatuses =
                aggregatedStatusesViewService.getKeywordStatusesByAdgroupIds(shard, clientId, List.of(adgroupId));
        GdStatusPopupInfo keywordsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.KEYWORD, GdEntityLevel.KEYWORD,
                keywordStatuses, Map.of());

        Map<Long, AggregatedStatusRetargetingData> retargetingsStatuses =
                aggregatedStatusesViewService.getRetargetingStatusesByAdgroupIds(shard, clientId, List.of(adgroupId));
        GdStatusPopupInfo retargetingsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.RETARGETING,
                GdEntityLevel.RETARGETING, retargetingsStatuses, Map.of());

        return new GdStatusAdgroupPopupInfo(adsInfo, keywordsInfo, retargetingsInfo);
    }

    /**
     * Новая ручка для получени информации попапа кампаний. Содержит информацию о всех дочерних сущностях, а также
     * о причинах отклонения.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "campaignStatusPopupNew")
    @PreAuthorizeRead
    public GdNewStatusPopupInfo campaignStatusPopupNew(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "id") Long campaignId,
            @GraphQLArgument(name = "tld") String tld,
            @GraphQLArgument(name = "lang") GdLanguage lang) {
        ClientId clientId = context.getSubjectUser().getClientId();
        int shard = shardHelper.getShardByClientId(clientId);

        final var campaignStatuses = aggregatedStatusesViewService.getCampaignStatusesByIds(shard, Set.of(campaignId));
        final var adgroupStatuses = aggregatedStatusesViewService.getAdGroupStatusesByCampaignId(shard, clientId,
                campaignId);

        final var adGroupsPopupData = adgroupStatusPopupNew(context, adgroupStatuses.keySet(), tld, lang);
        final var allAdGroupStatuses = adGroupsPopupData.getStatuses();
        final var campaignStatusPopupInfo = getGdStatusPopupInfo(GdPopupEntityEnum.CAMPAIGN, GdEntityLevel.CAMPAIGN,
                campaignStatuses, Map.of());
        final var groupStatusPopupInfo = getGdStatusPopupInfo(GdPopupEntityEnum.GROUP, GdEntityLevel.GROUP,
                adgroupStatuses, Map.of());
        final var mergedStatuses = mergeStatuses(Streams.concat(
                campaignStatusPopupInfo.getStatuses().stream(),
                groupStatusPopupInfo.getStatuses().stream(),
                allAdGroupStatuses.stream()
        ).collect(Collectors.toList()));

        final var sortedStatuses = gdStatusCountersFromStream(mergedStatuses.stream(), GdEntityLevel.CAMPAIGN);
        final var totalCounters = new HashMap(adGroupsPopupData.getTotalCounters());
        totalCounters.put(GdEntityLevel.CAMPAIGN, 1);

        final var total = new HashMap(adGroupsPopupData.getTotal());
        total.put(GdPopupEntityEnum.CAMPAIGN, 1);

        return new GdNewStatusPopupInfo()
                .withTotal(total)
                .withTotalCounters(totalCounters)
                .withCurrentEntityStatus(sortedStatuses.getLeft())
                .withStatuses(sortedStatuses.getRight());
    }

    /**
     * Новая ручка для получени информации попапа групп. Кроме причин и статусов возвращает причины отклонения.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "adgroupStatusPopupNew")
    @PreAuthorizeRead
    public GdNewStatusPopupInfo adgroupStatusPopupNew(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "id") Long adgroupId,
            @GraphQLArgument(name = "tld") String tld,
            @GraphQLArgument(name = "lang") GdLanguage lang) {
        return adgroupStatusPopupNew(context, List.of(adgroupId), tld, lang);
    }

    public GdNewStatusPopupInfo adgroupStatusPopupNew(
            GridGraphQLContext context,
            Collection<Long> adgroupIds,
            @Nullable String tld,
            @Nullable GdLanguage lang) {
        final var clientId = context.getSubjectUser().getClientId();
        final var shard = shardHelper.getShardByClientId(clientId);

        final var adGroupStatuses = aggregatedStatusesViewService.getAdGroupStatusesByIds(shard, adgroupIds.stream().collect(Collectors.toSet()));
        final var adStatuses = aggregatedStatusesViewService.getAdStatusesByAdgroupIds(shard, clientId, adgroupIds);
        final var keywordStatuses = aggregatedStatusesViewService.getKeywordStatusesByAdgroupIds(shard, clientId,
                adgroupIds);
        final var retargetingsStatuses = aggregatedStatusesViewService.getRetargetingStatusesByAdgroupIds(shard,
                clientId, adgroupIds);

        final var bids = new HashSet<>(adStatuses.keySet());
        final var moderationReasonsWithDetails = this.moderationReasons.getReasonsWithDetails(shard, bids);
        final var moderationTextByDiagId = getModerationTextsByDiagId(moderationReasonsWithDetails, context, tld, lang);
        final var moderationDiags = new ModerationDiags(moderationReasonsWithDetails, moderationTextByDiagId,
                keywordStatuses);

        final var adGroupsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.GROUP, GdEntityLevel.GROUP, adGroupStatuses, Map.of());
        final var adsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.BANNER, GdEntityLevel.BANNER, adStatuses,
                moderationDiags.banners());
        final var keywordsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.KEYWORD, GdEntityLevel.KEYWORD,
                keywordStatuses, moderationDiags.keywords());
        final var retargetingsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.RETARGETING, GdEntityLevel.RETARGETING,
                retargetingsStatuses, Map.of());

        final var totalCounters = Map.of(GdEntityLevel.GROUP, adGroupsInfo.getTotal(),
                GdEntityLevel.BANNER, adsInfo.getTotal(),
                GdEntityLevel.KEYWORD, keywordsInfo.getTotal(),
                GdEntityLevel.RETARGETING, retargetingsInfo.getTotal());

        final var total = Map.of(GdPopupEntityEnum.GROUP, adGroupsInfo.getTotal(),
                GdPopupEntityEnum.BANNER, adsInfo.getTotal(),
                GdPopupEntityEnum.KEYWORD, keywordsInfo.getTotal(),
                GdPopupEntityEnum.RETARGETING, retargetingsInfo.getTotal());

        final var statuses = gdStatusCountersFromStream(Streams.concat(
                adGroupsInfo.getStatuses().stream(),
                adsInfo.getStatuses().stream(),
                keywordsInfo.getStatuses().stream(),
                retargetingsInfo.getStatuses().stream()
        ), GdEntityLevel.GROUP);

        return new GdNewStatusPopupInfo()
                .withTotal(total)
                .withTotalCounters(totalCounters)
                .withCurrentEntityStatus(statuses.getLeft())
                .withStatuses(statuses.getRight());
    }

    /**
     * Новая ручка для получени информации попапа баннера. Кроме информации о баннере возвращает информацию об
     * отклоненных фразах.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "bannerStatusPopupNew")
    @PreAuthorizeRead
    public GdNewStatusPopupInfo bannerStatusPopupNew(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "id") Long bannerId,
            @GraphQLArgument(name = "tld") String tld,
            @GraphQLArgument(name = "lang") GdLanguage lang) {
        final var clientId = context.getSubjectUser().getClientId();
        final var shard = shardHelper.getShardByClientId(clientId);
        final var adGroupIdByBannerId = bannerRelationsRepository.getAdGroupIdsByBannerIds(shard, List.of(bannerId));

        if (!adGroupIdByBannerId.containsKey(bannerId)) {
            return new GdNewStatusPopupInfo()
                    .withTotal(Map.of())
                    .withTotalCounters(Map.of())
                    .withStatuses(List.of());
        }

        final var adGroupId = adGroupIdByBannerId.get(bannerId);

        final var adStatuses = aggregatedStatusesViewService.getAdStatusesByIds(shard, Set.of(bannerId));
        final var keywordStatuses = aggregatedStatusesViewService.getKeywordStatusesByAdgroupIds(shard, clientId,
                List.of(adGroupId));

        final var moderationReasonsWithDetails = this.moderationReasons.getReasonsWithDetails(shard, List.of(bannerId));
        final var moderationTextByDiagId = getModerationTextsByDiagId(moderationReasonsWithDetails, context, tld, lang);
        final var moderationDiags = new ModerationDiags(moderationReasonsWithDetails, moderationTextByDiagId,
                keywordStatuses);

        final var adsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.BANNER, GdEntityLevel.BANNER, adStatuses,
                moderationDiags.banners());
        final var keywordsInfo = getGdStatusPopupInfo(GdPopupEntityEnum.KEYWORD, GdEntityLevel.KEYWORD,
                keywordStatuses, moderationDiags.keywords());

        final var totalCounters = Map.of(GdEntityLevel.BANNER, 1,
                GdEntityLevel.KEYWORD, keywordsInfo.getTotal());

        final var total = Map.of(GdPopupEntityEnum.BANNER, 1,
                GdPopupEntityEnum.KEYWORD, keywordsInfo.getTotal());

        final var statuses = gdStatusCountersFromStream(Stream.concat(
                adsInfo.getStatuses().stream(),
                keywordsInfo.getStatuses().stream()
        ), GdEntityLevel.BANNER);

        return new GdNewStatusPopupInfo()
                .withTotal(total)
                .withTotalCounters(totalCounters)
                .withCurrentEntityStatus(statuses.getLeft())
                .withStatuses(statuses.getRight());
    }

    private Pair<GdStatusCounter, List<GdStatusCounter>> gdStatusCountersFromStream(
            Stream<GdStatusCounter> stream, GdEntityLevel level
    ) {
        final var allStatusCounters = stream
                .map(counter -> this.filterCounterReasons(counter, level))
                .filter(counter -> counter.getModerationDiags().size() > 0
                        || counter.getReasons().size() > 0
                        || counter.getEntityLevel().value() > level.value())
                .sorted(new GdStatusCountersComparator())
                .collect(Collectors.toList());

        final var currentEntityStatus = allStatusCounters
                .stream()
                .filter(counter -> counter.getEntityLevel() == level)
                .findFirst();

        final var statusesOfOtherEntities = allStatusCounters
                .stream()
                .filter(counter -> counter.getEntityLevel() != level)
                .collect(Collectors.toList());

        return Pair.of(currentEntityStatus.orElse(null), statusesOfOtherEntities);
    }

    private GdStatusCounter filterCounterReasons(GdStatusCounter counter, GdEntityLevel level) {
        var reasons = (counter.getReasons() != null ? counter.getReasons() : new ArrayList<GdSelfStatusReason>())
                .stream()
                .filter(reason -> reason.level().value() >= counter.getEntityLevel().value())
                .collect(Collectors.toList());

        return new GdStatusCounter()
                .withEntityType(counter.getEntityType())
                .withEntityLevel(counter.getEntityLevel())
                .withStatus(counter.getStatus())
                .withReasons(reasons)
                .withModerationDiags(counter.getModerationDiags())
                .withCount(counter.getCount());
    }

    private Map<String, String> getModerationTextsByDiagId(
            List<ModerationReasonWithDetails> moderationReasonWithDetails,
            GridGraphQLContext context,
            @Nullable String tld,
            @Nullable GdLanguage inputLang
    ) {
        final var language = getLanguage(context, inputLang);
        final var diagIds = moderationReasonWithDetails
                .stream()
                .map(ModerationReasonWithDetails::getDiagId)
                .map(Object::toString)
                .distinct()
                .toArray(String[]::new);

        return this.moderationReasonTextService
                .showModReasons(new ModerationReasonRequest(diagIds, tld, GdLanguage.toSource(language)), null)
                .getReasons();
    }

    private Collection<GdStatusCounter> mergeStatuses(Collection<GdStatusCounter> statuses) {
        final var result = new HashMap<Integer, GdStatusCounter>();

        for (var status : statuses) {
            final var hash = Objects.hash(
                    status.getEntityType(),
                    status.getStatus(),
                    status.getReasons(),
                    status.getModerationDiags());

            final var defaultCounter = new GdStatusCounter()
                    .withEntityType(status.getEntityType())
                    .withEntityLevel(status.getEntityLevel())
                    .withStatus(status.getStatus())
                    .withReasons(status.getReasons())
                    .withModerationDiags(status.getModerationDiags())
                    .withCount(0);

            final var existCounter = result.getOrDefault(hash, defaultCounter);
            final var newCounter = new GdStatusCounter()
                    .withEntityType(status.getEntityType())
                    .withEntityLevel(status.getEntityLevel())
                    .withStatus(status.getStatus())
                    .withReasons(status.getReasons())
                    .withModerationDiags(status.getModerationDiags())
                    .withCount(status.getCount() + existCounter.getCount());

            result.put(hash, newCounter);
        }

        return result.values();
    }

    private GdStatusPopupInfo getGdStatusPopupInfo(
            GdPopupEntityEnum entityType,
            GdEntityLevel entityLevel,
            Map<Long, ? extends AggregatedStatusBaseData<?>> statuses,
            Map<Long, Set<GdModerationDiagData>> moderationReports) {
        int total = statuses.values().size();
        int needAttentionCnt = 0;
        HashMap<SelfStatusWithModerationDiags, GdStatusCounter> cntByStatus = new HashMap<>();

        for (var entry : statuses.entrySet()) {
            final var statusData = entry.getValue();
            final var id = entry.getKey();

            if (statusData.getSelfStatusObject().isEmpty()) {
                continue;
            }

            SelfStatus status = statusData.getSelfStatusObject().get();
            if (GdSelfStatusEnum.allOk().contains(status.getStatus())) {
                continue;
            }

            final var moderationDiags = moderationReports
                    .getOrDefault(id, new HashSet<>())
                    .stream()
                    .sorted((a, b) -> (int) (a.getDiagId() - b.getDiagId()))
                    .collect(Collectors.toList());

            final var selfStatusWithModerationDiags = new SelfStatusWithModerationDiags(status, moderationDiags);

            if (cntByStatus.containsKey(selfStatusWithModerationDiags)) {
                final var statusCounter = cntByStatus.get(selfStatusWithModerationDiags);
                statusCounter.withCount(statusCounter.getCount() + 1);
            } else {
                final var statusCounter = new GdStatusCounter()
                        .withEntityType(entityType)
                        .withEntityLevel(entityLevel)
                        .withStatus(status.getStatus())
                        .withReasons(status.getReasons())
                        .withModerationDiags(moderationDiags)
                        .withCount(1);

                cntByStatus.put(selfStatusWithModerationDiags, statusCounter);
            }
            needAttentionCnt++;
        }

        return new GdStatusPopupInfo(
                total,
                needAttentionCnt,
                countersMapToSortedList(cntByStatus)
        );
    }

    private List<GdStatusCounter> countersMapToSortedList(HashMap<SelfStatusWithModerationDiags, GdStatusCounter> counters) {
        return counters.values().stream().sorted((a, b) -> {
            int cmpStatuses = Integer.compare(b.getStatus().ordinal(), a.getStatus().ordinal());
            return cmpStatuses != 0
                    ? cmpStatuses :
                    Integer.compare(b.getReasons().size(), a.getReasons().size());
        }).collect(Collectors.toList());
    }

    private GdLanguage getLanguage(GridGraphQLContext context, @Nullable GdLanguage lang) {
        if (lang != null) {
            return lang;
        }

        final var subjectUser = context.getSubjectUser();
        if (subjectUser != null) {
            return GdLanguage.fromSource(subjectUser.getLang());
        }

        return GdLanguage.RU;
    }
}
