package ru.yandex.direct.grid.processing.service.aggregatedstatuses;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator;
import ru.yandex.direct.grid.model.campaign.GdCampaignPrimaryStatus;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignFilterStatus;

import static ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator.isActivePrimaryStatus;
import static ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator.isArchivedPrimaryStatus;
import static ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator.isDraftPrimaryStatus;
import static ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator.isModerationDeniedPrimaryStatus;
import static ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator.isModerationPrimaryStatus;
import static ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator.isStoppedPrimaryStatus;
import static ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator.isTemporarilyPausedPrimaryStatus;

@ParametersAreNonnullByDefault
public class CampaignPrimaryStatusCalculator {
    private static final Logger logger =
            LoggerFactory.getLogger(CampaignPrimaryStatusCalculator.class);

    private CampaignPrimaryStatusCalculator() {
    }

    /**
     * Получить статус для фильтрации кампании. Статус для фильтрации — надмножество primary статусов.
     *
     * @param campaign — кампания во внутреннем представлении с уже посчитанным агрегированным статусом
     * @return статус дял фильтрации
     */
    @SuppressWarnings("checkstyle:RightCurly")
    public static GdCampaignFilterStatus convertToFilterStatus(GdiCampaign campaign) {
        var filterStatus = CampaignStatusCalculator.convertToFilterStatus(campaign.getAggregatedStatus());
        return GdCampaignFilterStatus.fromSource(filterStatus);
    }

    public static GdCampaignPrimaryStatus convertToPrimaryStatus(GdiCampaign campaign) {
        if (campaign.getAggregatedStatus() == null || campaign.getAggregatedStatus().getStatus().isEmpty()) {
            return GdCampaignPrimaryStatus.DRAFT;
        }

        GdSelfStatusEnum status = campaign.getAggregatedStatus().getStatus().get();
        List<GdSelfStatusReason> reasons = ListUtils.emptyIfNull(campaign.getAggregatedStatus().getReasons());
        Collection<CampaignStatesEnum> states = CollectionUtils.emptyIfNull(campaign.getAggregatedStatus().getStates());

        // Архивные
        if (isArchivedPrimaryStatus(status, reasons, states)) {
            return GdCampaignPrimaryStatus.ARCHIVED;
        }
        // На модерации
        else if (isModerationPrimaryStatus(status, reasons, states)) {
            return GdCampaignPrimaryStatus.MODERATION;
        }
        // Отклоненные
        else if (isModerationDeniedPrimaryStatus(status, reasons, states)) {
            return GdCampaignPrimaryStatus.MODERATION_DENIED;
        }
        // Запущенные
        else if (isTemporarilyPausedPrimaryStatus(status, reasons, states)) {
            return GdCampaignPrimaryStatus.TEMPORARILY_PAUSED;
        }
        // Остановленные
        else if (isStoppedPrimaryStatus(status, reasons, states)) {
            return GdCampaignPrimaryStatus.STOPPED;
        }
        // Черновики
        else if (isDraftPrimaryStatus(status, reasons, states)) {
            return GdCampaignPrimaryStatus.DRAFT;
        }
        // Запущенные
        else if (isActivePrimaryStatus(status, reasons, states)) {
            return GdCampaignPrimaryStatus.ACTIVE;
        }

        logger.warn("Can't find primary status for {}", campaign.getAggregatedStatus());
        return GdCampaignPrimaryStatus.ACTIVE;
    }

}
