package ru.yandex.direct.grid.processing.service.aggregatedstatuses;

import java.util.Collection;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.KeywordStatesEnum;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionPrimaryStatus;

import static org.apache.commons.collections4.CollectionUtils.emptyIfNull;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.REJECTED_ON_MODERATION;

@ParametersAreNonnullByDefault
public class KeywordsPrimaryStatusCalculator {

    private static final Logger logger = LoggerFactory.getLogger(KeywordsPrimaryStatusCalculator.class);

    public static GdiShowConditionPrimaryStatus convertToPrimaryStatus(@Nullable  AggregatedStatusKeywordData statusData) {
        if (statusData == null || statusData.getStatus().isEmpty()) {
            return GdiShowConditionPrimaryStatus.DRAFT;
        }
        GdSelfStatusEnum status = statusData.getStatus().get();
        List<GdSelfStatusReason> reasons = ListUtils.emptyIfNull(statusData.getReasons());
        Collection<KeywordStatesEnum> states = emptyIfNull(statusData.getStates());

        // Архивные
        if (isArchivedPrimaryStatus(status, reasons, states)) {
            return GdiShowConditionPrimaryStatus.ARCHIVED;
        }
        // Черновики
        else if (isDraftPrimaryStatus(status, reasons, states)) {
            return GdiShowConditionPrimaryStatus.DRAFT;
        }
        // Отклонена на модерации
        else if (isModerationRejectedPrimaryStatus(status, reasons, states)) {
            return GdiShowConditionPrimaryStatus.MODERATION_REJECTED;
        }
        // Остановленные
        else if (isStoppedPrimaryStatus(status, reasons, states)) {
            return GdiShowConditionPrimaryStatus.STOPPED;
        }
        // Активные
        else if (isActivePrimaryStatus(status, reasons, states)) {
            return GdiShowConditionPrimaryStatus.ACTIVE;
        }

        logger.warn("Can't find primary status for {}", statusData);
        return GdiShowConditionPrimaryStatus.ACTIVE;
    }

    // Архивные
    private static boolean isArchivedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                   Collection<KeywordStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.ARCHIVED);
    }

    // Черновики
    private static boolean isDraftPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                Collection<KeywordStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.DRAFT);
    }

    // Отклонена на модерации
    private static boolean isModerationRejectedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                             Collection<KeywordStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.STOP_CRIT) && reasons.contains(REJECTED_ON_MODERATION);
    }

    // Остановленные
    private static boolean isStoppedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                  Collection<KeywordStatesEnum> states) {
        return GdSelfStatusEnum.allStop().contains(status) && status != GdSelfStatusEnum.STOP_PROCESSING;
    }

    // Активные
    private static boolean isActivePrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                 Collection<KeywordStatesEnum> states) {
        return GdSelfStatusEnum.allRun().contains(status) || status == GdSelfStatusEnum.STOP_PROCESSING;
    }
}
