package ru.yandex.direct.grid.processing.service.aggregatedstatuses;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.aggregatedstatuses.logic.RejectedReasons.EntitiesRejectedReasons;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonWithDetails;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdModerationDiagData;

@ParametersAreNonnullByDefault
public final class ModerationDiags {
    private final Map<Long, AggregatedStatusKeywordData> keywordStatuses;
    private final List<ModerationReasonWithDetails> moderationReasonsWithDetails;
    private final Map<String, String> moderationTextByDiagId;

    public ModerationDiags(List<ModerationReasonWithDetails> moderationReasonsWithDetails,
                           Map<String, String> moderationTextByDiagId,
                           Map<Long, AggregatedStatusKeywordData> keywordStatuses) {
        this.keywordStatuses = keywordStatuses;
        this.moderationReasonsWithDetails = moderationReasonsWithDetails;
        this.moderationTextByDiagId = moderationTextByDiagId;
    }

    public Map<Long, Set<GdModerationDiagData>> banners() {
        final var result = new HashMap<Long, Set<GdModerationDiagData>>();
        final var moderationReasonsWithoutPhraseReasons = moderationReasonsWithDetails
                .stream()
                .filter(moderationDetails -> moderationDetails.getType() != ModerationReasonObjectType.PHRASES)
                .collect(Collectors.toList());

        for (var modReasonDetails : moderationReasonsWithoutPhraseReasons) {
            final var set = result.getOrDefault(modReasonDetails.getBannerId(), new HashSet<>());
            set.add(convertModerationDiagData(modReasonDetails));
            result.put(modReasonDetails.getBannerId(), set);
        }

        return result;
    }

    public Map<Long, Set<GdModerationDiagData>> keywords() {
        final var rejectedKeywordsIdsSet = keywordStatuses.entrySet()
                .stream()
                .filter(entry -> EntitiesRejectedReasons.keywordRejectedReasons.hasRejectedReason(entry.getValue().getReasons()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());

        final var allPhrasesDiags = moderationReasonsWithDetails
                .stream()
                .filter(moderationDetails -> moderationDetails.getType() == ModerationReasonObjectType.PHRASES)
                .map(this::convertModerationDiagData)
                .collect(Collectors.toSet());

        return rejectedKeywordsIdsSet.stream().collect(Collectors.toMap(id -> id, id -> allPhrasesDiags));
    }

    private GdModerationDiagData convertModerationDiagData(ModerationReasonWithDetails modReasonDetails) {
        return new GdModerationDiagData()
                .withDiagId(modReasonDetails.getDiagId())
                .withDiagText(modReasonDetails.getDiagText())
                .withShortText(modReasonDetails.getShortText())
                .withJson(moderationTextByDiagId.get(modReasonDetails.getDiagId().toString()))
                .withComment(modReasonDetails.getComment())
                .withScreenshots(modReasonDetails.getScreenshots())
                .withToken(modReasonDetails.getToken());
    }
}
