package ru.yandex.direct.grid.processing.service.aggregatedstatuses;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.AggregatedStatusRetargetingData;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.RetargetingStatesEnum;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingBaseStatus;

import static org.apache.commons.collections4.CollectionUtils.emptyIfNull;

@ParametersAreNonnullByDefault
public class RetargetingBaseStatusCalculator {

    private static final Logger logger = LoggerFactory.getLogger(RetargetingBaseStatusCalculator.class);

    public static GdiRetargetingBaseStatus convertToRetargetingBaseStatus(@Nullable AggregatedStatusRetargetingData statusData) {
        if (statusData == null || statusData.getStatus().isEmpty()) {
            return GdiRetargetingBaseStatus.ACTIVE;
        }
        GdSelfStatusEnum status = statusData.getStatus().get();
        List<GdSelfStatusReason> reasons = ListUtils.emptyIfNull(statusData.getReasons());
        Collection<RetargetingStatesEnum> states = emptyIfNull(statusData.getStates());

        // Архивные: Статуса нет в списке фильтров, отдаем null чтобы не фильтровать по нему
        // При данном кейсе группа будет отображаться только при выборе всех
        if (isArchivedPrimaryStatus(status, reasons, states)) {
            return null;
        }
        // Остановленные
        else if (isStoppedPrimaryStatus(status, reasons, states)) {
            return GdiRetargetingBaseStatus.SUSPENDED;
        }
        // Активные
        else if (isActivePrimaryStatus(status, reasons, states)) {
            return GdiRetargetingBaseStatus.ACTIVE;
        }


        logger.warn("Can't find primary status for {}", statusData);
        return GdiRetargetingBaseStatus.ACTIVE;
    }

    // Архивные
    private static boolean isArchivedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                   Collection<RetargetingStatesEnum> states) {
        return status.equals(GdSelfStatusEnum.ARCHIVED);
    }

    // Остановленные
    private static boolean isStoppedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                  Collection<RetargetingStatesEnum> states) {
        return GdSelfStatusEnum.allStop().contains(status) && status != GdSelfStatusEnum.STOP_PROCESSING;
    }

    // Активные
    private static boolean isActivePrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                 Collection<RetargetingStatesEnum> states) {
        return GdSelfStatusEnum.allRun().contains(status) || status == GdSelfStatusEnum.STOP_PROCESSING;
    }
}
