package ru.yandex.direct.grid.processing.service.autooverdraft;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.model.client.GdAutoOverdraftDataContainer;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.currency.CurrencyCode.AUTO_OVERDRAFT_AVAILABLE_CURRENCIES;
import static ru.yandex.direct.currency.Money.MONEY_CENT_SCALE;
import static ru.yandex.direct.grid.core.util.money.GridMoneyUtils.applyNdsSubtruction;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class AutoOverdraftDataLoader extends GridBatchingDataLoader<GdAutoOverdraftRequestData, GdAutoOverdraftDataContainer> {
    private final AutoOverdraftDataService autoOverdraftDataService;
    private final ClientService clientService;
    private final FeatureService featureService;
    private final ClientNdsService clientNdsService;

    public AutoOverdraftDataLoader(AutoOverdraftDataService autoOverdraftDataService,
                                   ClientService clientService,
                                   FeatureService featureService,
                                   ClientNdsService clientNdsService,
                                   GridContextProvider gridContextProvider) {
        this.autoOverdraftDataService = autoOverdraftDataService;
        this.clientService = clientService;
        this.featureService = featureService;
        this.clientNdsService = clientNdsService;
        dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction());
    }

    private MappedBatchLoaderWithContext<GdAutoOverdraftRequestData, GdAutoOverdraftDataContainer> getBatchLoadFunction() {
        return (requestDataList, environment) -> {
            if (requestDataList.isEmpty()) {
                return CompletableFuture.completedFuture(emptyMap());
            }
            Map<Long, Integer> clientIdToShard = listToMap(requestDataList,
                    GdAutoOverdraftRequestData::getClientId, GdAutoOverdraftRequestData::getShard);

            List<ClientId> realClientIds = mapList(clientIdToShard.keySet(), ClientId::fromLong);
            Map<Long, Client> clients = listToMap(clientService.massGetClient(realClientIds), Client::getId);
            Map<ClientId, Set<String>> enabledFeaturesByClientId = featureService.getEnabled(//внутри кэш - можем брать повторно
                    mapSet(clientIdToShard.keySet(), ClientId::fromLong));
            Map<Long, ClientNds> clientNdsMap =
                    listToMap(clientNdsService.massGetEffectiveClientNds(clients.values()), ClientNds::getClientId);
            Map<GdAutoOverdraftRequestData, GdAutoOverdraftDataContainer> result = listToMap(requestDataList, Function.identity(),
                    singleRequestData -> retrieveAutoOverdraftDataContainer(
                            singleRequestData.getClientId(),
                            enabledFeaturesByClientId.get(ClientId.fromLong(singleRequestData.getClientId())),
                            clients.get(singleRequestData.getClientId()),
                            clientNdsMap.get(singleRequestData.getClientId())
                    )
            );

            return CompletableFuture.completedFuture(result);
        };
    }

    private GdAutoOverdraftDataContainer retrieveAutoOverdraftDataContainer(
            Long clientId,
            Set<String> availableFeatures,
            Client client,
            @Nullable ClientNds clientNds
    ) {
        WalletRestMoney personalWallet = autoOverdraftDataService.getPersonalWallet(
                client.getWorkCurrency(),
                ClientId.fromLong(clientId)
        );
        GdAutoOverdraftDataContainer result = new GdAutoOverdraftDataContainer();
        Boolean autoOverdraftEnabled = AUTO_OVERDRAFT_AVAILABLE_CURRENCIES.contains(client.getWorkCurrency()) &&
                personalWallet != null &&
                availableFeatures.contains(FeatureName.PAYMENT_BEFORE_MODERATION.getName()) &&
                (
                        client.getOverdraftLimit().compareTo(BigDecimal.ZERO) > 0 ||
                                client.getAutoOverdraftLimit().compareTo(BigDecimal.ZERO) > 0
                ) &&
                !client.getIsBrand() &&
                !AutoOverdraftUtils.CLIENT_IDS_WITH_TWO_ACTIVE_WALLETS.contains(client.getId());

        result.setAutoOverdraftEnabled(autoOverdraftEnabled);

        if (autoOverdraftEnabled) {
            BigDecimal debt = client.getDebt();
            BigDecimal overdraftLimit = client.getOverdraftLimit();
            BigDecimal autoOverdraftLimit = client.getAutoOverdraftLimit();

            BigDecimal overspending = personalWallet.getRest().bigDecimalValue().negate();
            BigDecimal autoOverdraftUpperLimit = debt
                    .max(overspending.add(debt))
                    .max(overdraftLimit)
                    .setScale(MONEY_CENT_SCALE, RoundingMode.HALF_UP);
            // округление до копеек, чтобы избежать редкой ошибки валидации,
            // когда округленный порог больше округленной верхней границы
            BigDecimal autoOverdraftLowerLimit = debt
                    .max(overspending.add(debt))
                    .max(AutoOverdraftUtils.AUTOOVERDRAFT_LIMIT_DEFAULT_MIN_VALUE)
                    .setScale(MONEY_CENT_SCALE, RoundingMode.HALF_UP);
            Boolean autoOverdraftResetEnabled = overspending.compareTo(BigDecimal.ZERO) <= 0;
            Boolean isBanned = client.getStatusBalanceBanned().equals("Yes") ||
                    overdraftLimit.compareTo(BigDecimal.ZERO) == 0;
            Boolean autoOverdraftEditable = !isBanned && autoOverdraftUpperLimit.compareTo(BigDecimal.ZERO) != 0;

            result.setAutoOverdraftUpperLimit(autoOverdraftUpperLimit);
            result.setAutoOverdraftLowerLimit(autoOverdraftLowerLimit);
            result.setAutoOverdraftResetEnabled(autoOverdraftResetEnabled);
            result.setAutoOverdraftEditable(autoOverdraftEditable);

            if (clientNds != null) {
                applyNdsSubtruction(result::getAutoOverdraftUpperLimit,
                        result::setAutoOverdraftUpperLimitWithoutNds,
                        clientNds);
                applyNdsSubtruction(result::getAutoOverdraftLowerLimit,
                        result::setAutoOverdraftLowerLimitWithoutNds,
                        clientNds);
            }
        }
        return result;
    }
}
