package ru.yandex.direct.grid.processing.service.autooverdraft;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.jooq.TransactionalRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.balance.container.PaymentMethodInfo;
import ru.yandex.direct.core.entity.balance.container.PersonPaymentMethodInfo;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.repository.ClientOptionsRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.autooverdraft.GdPersonPaymentMethodInfo;
import ru.yandex.direct.grid.processing.model.autooverdraft.GdSetAutoOverdraftParams;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.service.autooverdraft.converter.AutoOverdraftDataConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static org.apache.commons.lang3.BooleanUtils.toBoolean;
import static ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils.AVAILABLE_CURRENCY_CODES;
import static ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils.CLIENT_IDS_WITH_TWO_ACTIVE_WALLETS;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.UNDER_WALLET;
import static ru.yandex.direct.currency.Money.MONEY_CENT_SCALE;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;

@Service
@ParametersAreNonnullByDefault
public class AutoOverdraftDataService {
    private static final Logger logger = LoggerFactory.getLogger(AutoOverdraftDataService.class);

    private static final Set<CampaignType> CAMPAIGN_TYPES = ImmutableSet.<CampaignType>builder()
            .addAll(UNDER_WALLET)
            .add(CampaignType.WALLET)
            .build();

    private static final BigDecimal AUTO_OVERDRAFT_MIN_DEFAULT_VALUE = BigDecimal.ONE;

    private final BalanceService balanceService;
    private final ClientService clientService;
    private final ClientOptionsRepository clientOptionsRepository;
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final GridValidationService gridValidationService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final CampaignRepository campaignRepository;
    private final CampaignService campaignService;

    @Autowired
    public AutoOverdraftDataService(
            FeatureService featureService,
            BalanceService balanceService, ClientService clientService,
            ClientOptionsRepository clientOptionsRepository,
            DslContextProvider dslContextProvider, ShardHelper shardHelper,
            GridValidationService gridValidationService,
            PpcPropertiesSupport ppcPropertiesSupport,
            CampaignRepository campaignRepository,
            CampaignService campaignService) {
        this.balanceService = balanceService;
        this.clientService = clientService;
        this.clientOptionsRepository = clientOptionsRepository;
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.gridValidationService = gridValidationService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.campaignRepository = campaignRepository;
        this.campaignService = campaignService;
    }

    List<GdPersonPaymentMethodInfo> getPersonPaymentMethods(GridGraphQLContext context, GdClient client) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        Client clientInfo = clientService.getClient(clientId);
        checkClient(clientInfo);

        WalletRestMoney personalWallet = getPersonalWallet(clientInfo.getWorkCurrency(), clientId);
        checkPersonalWallet(personalWallet);

        List<PersonPaymentMethodInfo> paymentOptions = balanceService
                .getPaymentOptions(context.getSubjectUser().getUid(), clientId, personalWallet.getWalletId());
        return mapList(paymentOptions, AutoOverdraftDataConverter::toGdPersonPaymentMethod);
    }

    private void checkClient(@Nullable Client clientInfo) {
        Preconditions.checkState(clientInfo != null, "Client cannot be found");
        Preconditions.checkState(!Boolean.TRUE.equals(clientInfo.getIsBrand()), "Client must not be brand");
        Preconditions.checkState(!CLIENT_IDS_WITH_TWO_ACTIVE_WALLETS.contains(clientInfo.getClientId()),
                "Client is listed in bad clients list");
        Preconditions.checkState(clientInfo.getOverdraftLimit() != null, "Client must have overdraft limit");
        Preconditions.checkState(
                clientInfo.getOverdraftLimit().compareTo(BigDecimal.ZERO) > 0
                        || clientInfo.getAutoOverdraftLimit() != null
                        && clientInfo.getAutoOverdraftLimit().compareTo(BigDecimal.ZERO) > 0,
                String.format("One of overdraft/auto overdraft limit must be greater than zero. "
                                + "Overdraft limit = %s, Auto overdraft limit = %s",
                        clientInfo.getOverdraftLimit(), clientInfo.getAutoOverdraftLimit()));
    }

    private void checkPersonalWallet(@Nullable WalletRestMoney personalWallet) {
        Preconditions.checkState(personalWallet != null, "Client must have personal wallet");
        Preconditions.checkState(AVAILABLE_CURRENCY_CODES.contains(personalWallet.getRest().getCurrencyCode()),
                "Personal wallet currency must be RUB, BYN or KZT ");
    }

    @Nullable
    public WalletRestMoney getPersonalWallet(
            CurrencyCode workCurrency,
            ClientId clientId
    ) {
        List<Campaign> allCampaigns = campaignService.searchNotEmptyCampaignsByClientIdAndTypes(clientId,
                CAMPAIGN_TYPES);
        Map<Long, WalletRestMoney> walletsRestMoney = campaignService.getWalletsRestMoney(clientId, allCampaigns);

        return allCampaigns.stream()
                .filter(c -> this.isActivePersonalWallet(workCurrency, c))
                .findFirst()
                .map(Campaign::getId)
                .map(walletId -> walletsRestMoney.values().stream()
                        .distinct()
                        .filter(wrm -> walletId.equals(wrm.getWalletId()))
                        .findFirst()
                        .orElse(null)
                )
                .orElse(null);
    }

    private boolean isActivePersonalWallet(CurrencyCode workCurrency, Campaign campaign) {
        boolean isWallet = campaign.getType() == CampaignType.WALLET;
        boolean isNotAgencyWallet = campaign.getAgencyId() == null || campaign.getAgencyId() == 0;
        boolean isNotArchived = !toBoolean(campaign.getStatusArchived());
        boolean hasClientWorkCurrency = campaign.getCurrency() == workCurrency;
        return isWallet && isNotAgencyWallet && isNotArchived && hasClientWorkCurrency;
    }

    private void validateAutoOverdraftLimit(GdSetAutoOverdraftParams request, BigDecimal moneyOnWallet, Client client,
                                            Map<Long, Map<String, Optional<BigDecimal>>> personIdToCodeLimitMap) {
        BigDecimal overdraftLimit = client.getOverdraftLimit();
        BigDecimal debt = nvl(client.getDebt(), BigDecimal.ZERO);
        BigDecimal overspending = moneyOnWallet.negate();
        BigDecimal autoOverdraftLowerLimit = overspending.add(debt).max(getAutoOverdraftLimitMinValue()).max(debt)
                .setScale(MONEY_CENT_SCALE, RoundingMode.HALF_UP);
        BigDecimal autoOverdraftUpperLimit = overspending.add(debt).max(overdraftLimit).max(debt)
                .setScale(MONEY_CENT_SCALE, RoundingMode.HALF_UP);

        logger.info(JsonUtils.toJson(ImmutableMap.of(
                "overdraftLimit", overdraftLimit,
                "debt", debt,
                "overspending", overspending,
                "autoOverdraftLowerLimit", autoOverdraftLowerLimit,
                "autoOverdraftUpperLimit", autoOverdraftUpperLimit
        )));
        ModelItemValidationBuilder<GdSetAutoOverdraftParams> divb = ModelItemValidationBuilder.of(request);
        divb.item(GdSetAutoOverdraftParams.AUTO_OVERDRAFT_LIMIT)
                .check(inRange(autoOverdraftLowerLimit, autoOverdraftUpperLimit));
        gridValidationService.throwGridValidationExceptionIfHasErrors(divb.getResult());

        divb.item(GdSetAutoOverdraftParams.PERSON_ID).check(inSet(personIdToCodeLimitMap.keySet()));
        gridValidationService.throwGridValidationExceptionIfHasErrors(divb.getResult());

        Map<String, Optional<BigDecimal>>
                codeToLimitMap = personIdToCodeLimitMap.getOrDefault(request.getPersonId(), emptyMap());
        Set<String> allowedPaymentMethodCodes = codeToLimitMap.keySet();
        divb.item(GdSetAutoOverdraftParams.PAYMENT_METHOD_CODE).check(inSet(allowedPaymentMethodCodes));
        gridValidationService.throwGridValidationExceptionIfHasErrors(divb.getResult());

        BigDecimal paymentMethodUpperLimit = codeToLimitMap.get(request.getPaymentMethodCode())
                .orElse(autoOverdraftUpperLimit);
        divb.item(GdSetAutoOverdraftParams.AUTO_OVERDRAFT_LIMIT).check(notGreaterThan(paymentMethodUpperLimit));
        gridValidationService.throwGridValidationExceptionIfHasErrors(divb.getResult());
    }

    private BigDecimal getAutoOverdraftLimitMinValue() {
        return ppcPropertiesSupport
                .find(PpcPropertyEnum.AUTO_OVERDRAFT_MIN_VALUE.getName())
                .filter(StringUtils::isNumeric)
                .map(BigDecimal::new)
                .orElse(AUTO_OVERDRAFT_MIN_DEFAULT_VALUE);
    }

    void setAutoOverdraftLimit(GridGraphQLContext context, GdSetAutoOverdraftParams request) {
        ClientId clientId = context.getSubjectUser().getClientId();
        Client clientInfo = clientService.getClient(clientId);
        int shard = shardHelper.getShardByClientId(clientId);
        checkClient(clientInfo);

        WalletRestMoney personalWallet = getPersonalWallet(clientInfo.getWorkCurrency(), clientId);
        checkPersonalWallet(personalWallet);

        CurrencyCode currencyCode = personalWallet.getRest().getCurrencyCode();

        List<PersonPaymentMethodInfo> paymentOptions = balanceService
                .getPaymentOptions(context.getSubjectUser().getUid(), clientId, personalWallet.getWalletId());

        Map<Long, Map<String, Optional<BigDecimal>>> personIdToCodeLimitMap =
                listToMap(paymentOptions,
                        ppmi -> ppmi.getPersonInfo().getId(),
                        ppmi -> listToMap(ppmi.getPaymentMethods(),
                                PaymentMethodInfo::getCode, pmi -> Optional.ofNullable(pmi.getLimit())));

        logger.info(JsonUtils.toJson(ImmutableMap.of("request", request, "personalWallet", personalWallet,
                "client", clientInfo, "limits", personIdToCodeLimitMap)));
        validateAutoOverdraftLimit(request, personalWallet.getRest().bigDecimalValue(), clientInfo,
                personIdToCodeLimitMap);

        TransactionalRunnable tr = conf -> {
            BigDecimal autoOverdraftLimit = request.getAutoOverdraftLimit().setScale(MONEY_CENT_SCALE,
                    RoundingMode.HALF_UP);
            boolean changed = clientOptionsRepository.updateAutoOverdraftLimit(conf, clientId, autoOverdraftLimit);
            if (changed) {
                campaignRepository.resetBannerSystemSyncStatus(conf, singleton(personalWallet.getWalletId()));
            }
            // balance call goes last intentionally
            // to rollback client overdraft limit update
            // in case of exceptions during the balance call
            balanceService.setOverdraftParams(request.getPersonId(), request.getPaymentMethodCode(),
                    currencyCode.name(), autoOverdraftLimit);
        };
        dslContextProvider.ppcTransaction(shard, tr);
    }

    void resetAutoOverdraftLimit(GridGraphQLContext context) {
        ClientId clientId = context.getSubjectUser().getClientId();
        Client clientInfo = clientService.getClient(clientId);
        int shard = shardHelper.getShardByClientId(clientId);
        checkClient(clientInfo);

        WalletRestMoney personalWallet = getPersonalWallet(clientInfo.getWorkCurrency(), clientId);
        checkPersonalWallet(personalWallet);

        Preconditions.checkState(personalWallet.getRest().bigDecimalValue().compareTo(BigDecimal.ZERO) >= 0,
                "Money on wallet must be greater than or equal to zero. Current value = " + personalWallet.getRest().bigDecimalValue());

        TransactionalRunnable tr = conf -> {
            clientOptionsRepository.updateAutoOverdraftLimit(conf, clientId, BigDecimal.ZERO);
            campaignRepository.resetBannerSystemSyncStatus(conf, singleton(personalWallet.getWalletId()));
            // balance call goes last intentionally
            // to rollback client overdraft limit update
            // in case of exceptions during the balance call
            balanceService.resetOverdraftParams(clientId, personalWallet.getRest().getCurrencyCode().name());
        };
        dslContextProvider.ppcTransaction(shard, tr);
    }
}
