package ru.yandex.direct.grid.processing.service.autooverdraft;

import java.math.BigDecimal;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.autooverdraft.GdPersonPaymentMethodInfo;
import ru.yandex.direct.grid.processing.model.autooverdraft.GdSetAutoOverdraftParams;
import ru.yandex.direct.grid.processing.model.autooverdraft.mutation.GdResetAutoOverdraftParamsPayload;
import ru.yandex.direct.grid.processing.model.autooverdraft.mutation.GdSetAutoOverdraftParamsPayload;
import ru.yandex.direct.grid.processing.model.client.GdAutoOverdraftDataContainer;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientAutoOverdraftInfo;

/**
 * Сервис про автоовердрафты
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class AutoOverdraftGraphQlService {

    private final AutoOverdraftDataService autoOverdraftDataService;
    private final AutoOverdraftDataLoader autoOverdraftDataLoader;

    @Autowired
    public AutoOverdraftGraphQlService(AutoOverdraftDataService autoOverdraftDataService,
                                       AutoOverdraftDataLoader autoOverdraftDataLoader) {
        this.autoOverdraftDataService = autoOverdraftDataService;
        this.autoOverdraftDataLoader = autoOverdraftDataLoader;
    }

    @GraphQLQuery(name = "autoOverdraftUpperLimit")
    public CompletableFuture<BigDecimal> getAutoOverdraftUpperLimit(
            @GraphQLContext GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        var requestData = toOverdraftRequestData(clientAutoOverdraftInfo);
        return autoOverdraftDataLoader.get().load(requestData)
                .thenApply(GdAutoOverdraftDataContainer::getAutoOverdraftUpperLimit);
    }

    @GraphQLQuery(name = "autoOverdraftLowerLimit")
    public CompletableFuture<BigDecimal> getAutoOverdraftLowerLimit(
            @GraphQLContext GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        var requestData = toOverdraftRequestData(clientAutoOverdraftInfo);
        return autoOverdraftDataLoader.get().load(requestData)
                .thenApply(GdAutoOverdraftDataContainer::getAutoOverdraftLowerLimit);
    }

    @GraphQLQuery(name = "autoOverdraftUpperLimitWithoutNds")
    public CompletableFuture<BigDecimal> getAutoOverdraftUpperLimitWithoutNds(
            @GraphQLContext GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        var requestData = toOverdraftRequestData(clientAutoOverdraftInfo);
        return autoOverdraftDataLoader.get().load(requestData)
                .thenApply(GdAutoOverdraftDataContainer::getAutoOverdraftUpperLimitWithoutNds);
    }

    @GraphQLQuery(name = "autoOverdraftLowerLimitWithoutNds")
    public CompletableFuture<BigDecimal> getAutoOverdraftLowerLimitWithoutNds(
            @GraphQLContext GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        var requestData = toOverdraftRequestData(clientAutoOverdraftInfo);
        return autoOverdraftDataLoader.get().load(requestData)
                .thenApply(GdAutoOverdraftDataContainer::getAutoOverdraftLowerLimitWithoutNds);
    }

    @GraphQLQuery(name = "autoOverdraftResetEnabled")
    public CompletableFuture<Boolean> getAutoOverdraftResetEnabled(
            @GraphQLContext GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        var requestData = toOverdraftRequestData(clientAutoOverdraftInfo);
        return autoOverdraftDataLoader.get().load(requestData)
                .thenApply(GdAutoOverdraftDataContainer::getAutoOverdraftResetEnabled);
    }

    @GraphQLQuery(name = "autoOverdraftEditable")
    public CompletableFuture<Boolean> getAutoOverdraftEditable(
            @GraphQLContext GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        var requestData = toOverdraftRequestData(clientAutoOverdraftInfo);
        return autoOverdraftDataLoader.get().load(requestData)
                .thenApply(GdAutoOverdraftDataContainer::getAutoOverdraftEditable);
    }

    @GraphQLQuery(name = "autoOverdraftEnabled")
    public CompletableFuture<Boolean> getAutoOverdraftEnabled(
            @GraphQLContext GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        var requestData = toOverdraftRequestData(clientAutoOverdraftInfo);
        return autoOverdraftDataLoader.get().load(requestData)
                .thenApply(GdAutoOverdraftDataContainer::getAutoOverdraftEnabled);
    }

    private static GdAutoOverdraftRequestData toOverdraftRequestData(
            GdClientAutoOverdraftInfo clientAutoOverdraftInfo) {
        return new GdAutoOverdraftRequestData(clientAutoOverdraftInfo.getClientId(), clientAutoOverdraftInfo.getShard());
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможных вариантах оплаты автоовердрафта
     *
     * @param client клиент
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "autoOverdraftPaymentOptions")
    public List<@GraphQLNonNull GdPersonPaymentMethodInfo> getAutoOverdraftPaymentOptions(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        return autoOverdraftDataService.getPersonPaymentMethods(context, client);
    }

    /**
     * GraphQL подзапрос. Устанавливает плательщика и способ оплаты автоовердрафта
     */
    @EnableLoggingOnValidationIssues
    @PreAuthorizeWrite
    @GraphQLNonNull
    @GraphQLMutation(name = "setAutoOverdraftLimit")
    public GdSetAutoOverdraftParamsPayload setAutoOverdraftLimit(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSetAutoOverdraftParams request) {
        autoOverdraftDataService.setAutoOverdraftLimit(context, request);
        return new GdSetAutoOverdraftParamsPayload();
    }

    /**
     * GraphQL подзапрос. Обнуление лимита автоовердрафта
     *
     * @return новый лимит автоовердрафта
     */
    @EnableLoggingOnValidationIssues
    @PreAuthorizeWrite
    @GraphQLNonNull
    @GraphQLMutation(name = "resetAutoOverdraftLimit")
    public GdResetAutoOverdraftParamsPayload resetAutoOverdraftLimit(
            @GraphQLRootContext GridGraphQLContext context) {
        autoOverdraftDataService.resetAutoOverdraftLimit(context);
        return new GdResetAutoOverdraftParamsPayload().withAutoOverdraftLimit(BigDecimal.ZERO);
    }

}
