package ru.yandex.direct.grid.processing.service.banner;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerHrefItem;
import ru.yandex.direct.grid.core.entity.banner.service.GridFindAndReplaceBannerHrefService;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefBase;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefTargetType;
import ru.yandex.direct.grid.processing.model.common.GdCachedResult;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.model.common.GdiOperationResult;
import ru.yandex.direct.grid.processing.service.cache.CacheFilterData;
import ru.yandex.direct.grid.processing.service.cache.CacheRecordInfo;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerHrefConverter.convertToPreviewItems;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.convertEmptyToNull;
import static ru.yandex.direct.grid.processing.util.OperationResultSorter.sortByValidationResult;

/**
 * Общая часть сервиса массового поиска и изменения ссылок баннеров
 */
@ParametersAreNonnullByDefault
public abstract class AbstractFindAndReplaceBannerHrefService {

    private final GridCacheService cacheService;
    private final GridFindAndReplaceBannerHrefService hrefService;
    private final GridValidationResultConversionService validationResultConverter;

    @Autowired
    public AbstractFindAndReplaceBannerHrefService(
            GridCacheService cacheService,
            GridFindAndReplaceBannerHrefService hrefService,
            GridValidationResultConversionService validationResultConverter) {
        this.cacheService = cacheService;
        this.hrefService = hrefService;
        this.validationResultConverter = validationResultConverter;
    }

    /**
     * Получить превью для поиска и изменения ссылок баннера
     */
    protected GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem> preview(GdFindAndReplaceAdsHrefBase input,
                                                                         Long operatorUid, ClientId clientId,
                                                                         ReplaceRule replaceRule,
                                                                         ReplaceRule turbolandingReplaceRule,
                                                                         CacheRecordInfo<GdFindAndReplaceAdsHrefPayloadItem, ? extends CacheFilterData, GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem>> recordInfo) {
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        Optional<GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem>> res = cacheService.getFromCache(recordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        List<GdiFindAndReplaceBannerHrefItem> bannersInfo = getBannersForUpdate(clientId, input, replaceRule, turbolandingReplaceRule);

        GridBannerUpdateInfo updateInfo =
                hrefService.updateBannersHrefAndSitelinksPreview(bannersInfo, operatorUid, clientId);

        GdResult<GdFindAndReplaceAdsHrefPayloadItem> gdResult = buildGdResult(bannersInfo, updateInfo);
        GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem> payload =
                new GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem>()
                        .withValidationResult(gdResult.getValidationResult())
                        .withTotalCount(gdResult.getTotalCount())
                        .withSuccessCount(gdResult.getSuccessCount());

        return cacheService.getResultAndSaveToCache(recordInfo, payload, gdResult.getRowset(), range);
    }

    /**
     * Поиск и изменение ссылок баннера
     */
    protected GdResult<GdFindAndReplaceAdsHrefPayloadItem> replace(GdFindAndReplaceAdsHrefBase input, Long operatorUid,
                                                                   ClientId clientId, ReplaceRule replaceRule,
                                                                   ReplaceRule turbolandingReplaceRule) {
        List<GdiFindAndReplaceBannerHrefItem> bannersInfo = getBannersForUpdate(clientId, input, replaceRule, turbolandingReplaceRule);
        GridBannerUpdateInfo updateInfo = hrefService.updateBannersHrefAndSitelinks(bannersInfo, operatorUid, clientId);
        return buildGdResult(bannersInfo, updateInfo);
    }

    /**
     * Подготавливает данные и извлекает информацию о баннерах и сайтлинках через
     * {@link GridFindAndReplaceBannerHrefService}
     */
    private List<GdiFindAndReplaceBannerHrefItem> getBannersForUpdate(
            ClientId clientId, GdFindAndReplaceAdsHrefBase input, ReplaceRule replaceRule, ReplaceRule turbolandingReplaceRule) {
        boolean needReplaceBannerHref = input.getTargetTypes().contains(GdFindAndReplaceAdsHrefTargetType.AD_HREF);
        boolean needReplaceSitelinkHref =
                input.getTargetTypes().contains(GdFindAndReplaceAdsHrefTargetType.SITELINK_HREF);
        boolean needReplaceTurbolandingParams =
                input.getTargetTypes().contains(GdFindAndReplaceAdsHrefTargetType.TURBOLANDING_PARAMS);

        return hrefService
                .getFindAndReplaceBannersHrefItems(clientId, input.getAdIds(), input.getAdIdsHrefExceptions(),
                        input.getSitelinkIdsHrefExceptions(), needReplaceBannerHref, needReplaceSitelinkHref,
                        needReplaceTurbolandingParams, replaceRule, turbolandingReplaceRule);
    }

    /**
     * Собирает результат выполнения поиска и замены. Происходит сортировка по результату валидации: сначала ошибки,
     * затем warning'и, затем остальное. Из итогового результата удаляются все warning'и.
     * TotalCount = количество баннеров, которые должны были обновиться (href и/или сайтлинки).
     * SuccessCount = количество успешно обновленных баннеров. Баннер считается успешно обновленным, если обновилось
     * все,
     * что требуется. То есть если был обновлен href, но не были обновлены сайтлинки (но должны были) - баннер не
     * считается
     * успешно обновленным.
     *
     * @param bannersInfo информация о баннерах
     * @param updateInfo  информация об обновлениии баннеров
     * @return результат изменений
     */
    private GdResult<GdFindAndReplaceAdsHrefPayloadItem> buildGdResult(
            List<GdiFindAndReplaceBannerHrefItem> bannersInfo, GridBannerUpdateInfo updateInfo) {
        GdiOperationResult<GdiFindAndReplaceBannerHrefItem, BannerWithSystemFields> sortedOperationResult =
                sortByValidationResult(new GdiOperationResult<>(bannersInfo, updateInfo.getValidationResult()));

        GdValidationResult validationResult =
                validationResultConverter.buildGridValidationResult(sortedOperationResult.getValidationResult());
        removeWarnings(validationResult);

        List<GdFindAndReplaceAdsHrefPayloadItem> rowset = convertToPreviewItems(sortedOperationResult.getRowset());

        return new GdResult<GdFindAndReplaceAdsHrefPayloadItem>()
                .withTotalCount(rowset.size())
                .withSuccessCount(updateInfo.getUpdatedBannerCount())
                .withRowset(rowset)
                .withValidationResult(convertEmptyToNull(validationResult));
    }

    /**
     * Удалить все warnings из GdValidationResult
     *
     * @param validationResult - плоский результат валидации
     */
    private void removeWarnings(@Nullable GdValidationResult validationResult) {
        if (validationResult == null) {
            return;
        }
        validationResult.setWarnings(Collections.emptyList());
    }
}
