package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.GdAdAccess;
import ru.yandex.direct.grid.processing.model.banner.GdAdFilter;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrice;
import ru.yandex.direct.grid.processing.model.banner.GdAdWithTotals;
import ru.yandex.direct.grid.processing.model.banner.GdAdsContainer;
import ru.yandex.direct.grid.processing.model.banner.GdAdsContext;
import ru.yandex.direct.grid.processing.model.banner.GdBannerButton;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAdContainer;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAds;
import ru.yandex.direct.grid.processing.model.banner.GdMulticard;
import ru.yandex.direct.grid.processing.model.banner.GdMulticardStatus;
import ru.yandex.direct.grid.processing.model.banner.GdTurboGalleryHrefPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdCreative;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdImage;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddContentPromotionAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddCpmAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddDynamicAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddInternalAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMcBannerAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMobileContentAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMulticardsToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSetToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSets;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSetsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSmartAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddTurboLandingToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAdsMassAction;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAdsMassActionPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdCopyAdsInput;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdAgeFlags;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdPrice;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdTurboGalleryHrefsInput;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateBannerButton;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateContentPromotionAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateCpmAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateDynamicAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateInternalAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMcBannerAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMobileContentAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateOrganization;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartCenters;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartCentersPayload;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImage;
import ru.yandex.direct.grid.processing.service.banner.container.BannersCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.banner.loader.BannerImageDataLoader;
import ru.yandex.direct.grid.processing.service.banner.loader.CanBeDeletedAdsDataLoader;
import ru.yandex.direct.grid.processing.service.banner.loader.MulticardReasonDataLoader;
import ru.yandex.direct.grid.processing.service.banner.mutation.AddInternalAdsMutationService;
import ru.yandex.direct.grid.processing.service.banner.mutation.UpdateInternalAdsMutationService;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toBannersCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toGdAdsContext;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;

/**
 * Сервис, возвращающий данные о баннерах клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class AdGraphQlService {
    public static final String ADS_RESOLVER_NAME = "ads";

    private static final String GET_DOMAINS_ARG_NAME = "adIds";
    private static final String GET_DISPLAY_HREFS_ARG_NAME = "adIds";
    private final GridCacheService gridCacheService;
    private final BannerDataService bannerDataService;
    private final GridValidationService gridValidationService;
    private final GridShortenerService gridShortenerService;
    private final FeatureService featureService;

    private final FindAndReplaceBannerHrefDomainService findAndReplaceBannerHrefDomainService;
    private final FindAndReplaceBannersDisplayHrefService findAndReplaceBannersDisplayHrefService;
    private final AdMassActionsService adMassActionsService;
    private final AddInternalAdsMutationService addInternalAdsMutationService;
    private final UpdateInternalAdsMutationService updateInternalAdsMutationService;
    private final BannerImageDataLoader bannerImageDataLoader;
    private final MulticardReasonDataLoader multicardReasonDataLoader;
    private final CanBeDeletedAdsDataLoader canBeDeletedAdsDataLoader;
    private final BannerButtonService bannerButtonService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public AdGraphQlService(GridCacheService gridCacheService, BannerDataService bannerDataService,
                            GridValidationService gridValidationService,
                            GridShortenerService gridShortenerService,
                            FindAndReplaceBannerHrefDomainService findAndReplaceBannerHrefDomainService,
                            AdMassActionsService bannerMassActionsService,
                            FindAndReplaceBannersDisplayHrefService findAndReplaceBannersDisplayHrefService,
                            AddInternalAdsMutationService addInternalAdsMutationService,
                            UpdateInternalAdsMutationService updateInternalAdsMutationService,
                            BannerImageDataLoader bannerImageDataLoader,
                            MulticardReasonDataLoader multicardReasonDataLoader,
                            FeatureService featureService,
                            CanBeDeletedAdsDataLoader canBeDeletedAdsDataLoader,
                            BannerButtonService bannerButtonService) {
        this.gridCacheService = gridCacheService;
        this.bannerDataService = bannerDataService;
        this.gridValidationService = gridValidationService;
        this.gridShortenerService = gridShortenerService;
        this.findAndReplaceBannerHrefDomainService = findAndReplaceBannerHrefDomainService;
        this.findAndReplaceBannersDisplayHrefService = findAndReplaceBannersDisplayHrefService;
        this.adMassActionsService = bannerMassActionsService;
        this.addInternalAdsMutationService = addInternalAdsMutationService;
        this.updateInternalAdsMutationService = updateInternalAdsMutationService;
        this.bannerImageDataLoader = bannerImageDataLoader;
        this.multicardReasonDataLoader = multicardReasonDataLoader;
        this.featureService = featureService;
        this.canBeDeletedAdsDataLoader = canBeDeletedAdsDataLoader;
        this.bannerButtonService = bannerButtonService;
    }

    /**
     * GraphQL подзапрос. Получает информацию о баннерах клиента, полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = ADS_RESOLVER_NAME)
    public GdAdsContext getAds(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdsContainer input) {
        gridValidationService.validateGdAdsContainer(input);
        GdClientInfo client = context.getQueriedClient();

        if (input.getFilterKey() != null) {
            GdAdFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    ClientId.fromLong(client.getId()),
                    GdAdFilter.class,
                    () -> new GdAdFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }

        GdStatRequirements statRequirements = normalizeStatRequirements(input.getStatRequirements(),
                context.getInstant(), input.getFilter().getRecommendations());
        input.setStatRequirements(statRequirements);

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());


        // пытаемся прочитать из кеша нужный диапазон строк
        BannersCacheRecordInfo recordInfo = toBannersCacheRecordInfo(client.getId(), input);
        Optional<GdAdsContext> res = gridCacheService.getFromCache(recordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        boolean cpcAndCpmOnOneGridEnabled =
                featureService.isEnabledForClientId(ClientId.fromLong(client.getId()), CPC_AND_CPM_ON_ONE_GRID_ENABLED);

        // в кеше данные не нашлись, читаем из mysql/YT
        GdAdWithTotals adsWithTotals = bannerDataService.getBanners(client, input, context);
        GdAdsContext adsContext = toGdAdsContext(adsWithTotals, input.getFilter(), cpcAndCpmOnOneGridEnabled);
        adsContext.setFilter(input.getFilter());

        // сохраняем в кеш, если надо, и возвращаем нужный диапазон строк в результате
        return gridCacheService.getResultAndSaveToCacheIfRequested(recordInfo, adsContext,
                adsWithTotals.getGdAds(), range,
                context.getFetchedFieldsReslover().getAd().getCacheKey());
    }

    @GraphQLQuery(name = "canBeDeleted")
    public CompletableFuture<Boolean> getAds(
            @SuppressWarnings("unused") @GraphQLContext GdAdAccess adAccess) {
        if (!adAccess.getCanEdit()) {
            // По аналогии с GroupDataService.getCanBeDeletedAdGroup делаем canBeDeleted зависимым от canEdit
            // todo maxlog: подумать о том, как бы безопасно разделить операторские флаги от собственных атрибутов
            //  объектов
            return CompletableFuture.completedFuture(false);
        }
        return canBeDeletedAdsDataLoader.get().load(adAccess.getAdId());
    }

    /**
     * GraphQL подзапрос. Отдает для выбранной пары (номер кампании-тип баннера) последний измененный баннер выбранного
     * типа в кампании, или null
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "lastChangedAds")
    public GdLastChangedAds getLastChangedAd(
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdLastChangedAdContainer input) {
        return bannerDataService.getLastChangedAds(input);
    }

    /**
     * Мутация для массового добавления баннеров
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addAds")
    public GdAddAdsPayload addAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового добавления dynamic баннеров
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addDynamicAds")
    public GdAddAdsPayload addDynamicAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddDynamicAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addDynamicAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового добавления smart баннеров
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addSmartAds")
    public GdAddAdsPayload addSmartAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddSmartAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addSmartAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового добавления баннеров внутренней рекламы
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addInternalAds")
    public GdAddAdsPayload addInternalAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddInternalAds input) {
        if (input.getAdAddItems().isEmpty()) {
            return new GdAddAdsPayload().withAddedAds(emptyList());
        }

        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        return addInternalAdsMutationService.addInternalAds(clientId, context.getOperator().getUid(), input);
    }

    /**
     * Мутация для массового обновления баннеров внутренней рекламы
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateInternalAds")
    public GdUpdateAdsPayload updateInternalAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateInternalAds input) {
        if (input.getAdUpdateItems().isEmpty()) {
            return new GdUpdateAdsPayload().withUpdatedAds(emptyList());
        }

        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        return updateInternalAdsMutationService.updateInternalAds(clientId, context.getOperator().getUid(), input);
    }

    /**
     * Мутация для массового добавления сайтлинков
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addSitelinkSets")
    public GdAddSitelinkSetsPayload addSitelinkSets(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddSitelinkSets input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addSitelinkSets(clientId, input);
    }

    /**
     * Мутация для массового обновления баннеров
     *
     * @param context     контекст
     * @param gdUpdateAds запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateAds")
    public GdUpdateAdsPayload updateAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateAds gdUpdateAds) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateAds(clientId, context.getOperator(), gdUpdateAds);
    }

    /**
     * Мутация для массового обновления dynamic баннеров
     *
     * @param context     контекст
     * @param gdUpdateAds запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateDynamicAds")
    public GdUpdateAdsPayload updateDynamicAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateDynamicAds gdUpdateAds) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateDynamicAds(clientId, context.getOperator(), gdUpdateAds);
    }

    /**
     * Мутация для массового обновления ссылок на турбо-галерею
     *
     * @param context       контекст
     * @param gdUpdateHrefs запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateTurboGalleryHrefs")
    public GdUpdateAdsPayload updateTurboGalleryHrefs(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateAdTurboGalleryHrefsInput gdUpdateHrefs) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateTurboGalleryHrefs(clientId, context.getOperator(),
                gdUpdateHrefs.getHrefUpdateItems());
    }

    /**
     * Мутация для массового обновления smart баннеров
     *
     * @param context          контекст
     * @param gdUpdateSmartAds запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateSmartAds")
    public GdUpdateAdsPayload updateSmartAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateSmartAds gdUpdateSmartAds) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateSmartAds(clientId, context.getOperator(), gdUpdateSmartAds);
    }

    /**
     * Мутация для массового добавления cpm баннеров
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addCpmAds")
    public GdAddAdsPayload addCpmAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddCpmAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addCpmAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового обновления cpm баннеров
     *
     * @param context контекст
     * @param input   запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateCpmAds")
    public GdUpdateAdsPayload updateCpmAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCpmAds input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateCpmAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового добавления mcbanner баннеров
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addMcBannerAds")
    public GdAddAdsPayload addMcBannerAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddMcBannerAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addMcBannerAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового обновления mcbanner баннеров
     *
     * @param context контекст
     * @param input   запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидаци
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateMcBannerAds")
    public GdUpdateAdsPayload updateMcBannerAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateMcBannerAds input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateMcBannerAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового добавления mobile_content баннеров
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addMobileContentAds")
    public GdAddAdsPayload addMobileContentAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddMobileContentAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addMobileContentAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового обновления mobile_content баннеров
     *
     * @param context контекст
     * @param input   запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидаци
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateMobileContentAds")
    public GdUpdateAdsPayload updateMobileContentAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateMobileContentAds input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateMobileContentAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового добавления баннеров продвижения контента
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addContentPromotionAds")
    public GdAddAdsPayload addContentPromotionAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddContentPromotionAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addContentPromotionAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового обновления баннеров продвижения контента
     *
     * @param context контекст
     * @param input   запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидаци
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateContentPromotionAds")
    public GdUpdateAdsPayload updateContentPromotionAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateContentPromotionAds input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateContentPromotionAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для обновления смарт-центров изображений
     *
     * @param context              контекст
     * @param gdUpdateSmartCenters запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateSmartCenters")
    public GdUpdateSmartCentersPayload updateSmartCenters(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateSmartCenters gdUpdateSmartCenters) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateSmartCenters(clientId, gdUpdateSmartCenters.getImageHash(),
                gdUpdateSmartCenters.getSmartCenters());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "checkTurboGalleryHref")
    public GdTurboGalleryHrefPayload checkTurboGalleryHref(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "originalHref") String originalHref
    ) {
        String turboHref = bannerDataService.checkTurboGalleryHref(originalHref);
        return new GdTurboGalleryHrefPayload()
                .withTurboGalleryHref(turboHref);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "getAdsDomains")
    public Set<String> getAdsAndSitelinksDomains(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = GET_DOMAINS_ARG_NAME) List<@GraphQLNonNull Long> adIds) {
        return findAndReplaceBannerHrefDomainService
                .getDomains(adIds, context.getSubjectUser().getClientId(), GET_DOMAINS_ARG_NAME);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "getAdsDisplayHrefs")
    public Set<String> getAdsDisplayHrefs(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = GET_DISPLAY_HREFS_ARG_NAME) List<@GraphQLNonNull Long> adIds) {
        return findAndReplaceBannersDisplayHrefService
                .getDisplayHrefs(adIds, context.getSubjectUser().getClientId(), GET_DOMAINS_ARG_NAME);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "archiveAds")
    public GdAdsMassActionPayload archiveBanners(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdsMassAction input) {
        return adMassActionsService.archiveUnarchiveBanners(context.getSubjectUser().getClientId(),
                context.getOperator().getUid(), input, Boolean.TRUE);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "unarchiveAds")
    public GdAdsMassActionPayload unarchiveBanners(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdsMassAction input) {
        return adMassActionsService.archiveUnarchiveBanners(context.getSubjectUser().getClientId(),
                context.getOperator().getUid(), input, Boolean.FALSE);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "suspendAds")
    public GdAdsMassActionPayload suspendBanners(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdsMassAction input) {
        return adMassActionsService.suspendResumeBanners(context.getSubjectUser().getClientId(),
                context.getOperator().getUid(), input, Boolean.FALSE);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "resumeAds")
    public GdAdsMassActionPayload resumeBanners(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdsMassAction input) {
        return adMassActionsService.suspendResumeBanners(context.getSubjectUser().getClientId(),
                context.getOperator().getUid(), input, Boolean.TRUE);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "deleteAds")
    public GdAdsMassActionPayload deleteBanners(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdsMassAction input) {
        return adMassActionsService.deleteBanners(context.getSubjectUser().getClientId(),
                context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "remoderateAds")
    public GdAdsMassActionPayload remoderateBanners(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdsMassAction input) {
        return adMassActionsService.remoderateBanners(context.getSubjectUser().getClientId(),
                context.getOperator().getUid(), input);
    }

    @GraphQLQuery(name = "validateAdPrice")
    public GdValidationResult validateAdPrice(@GraphQLNonNull @GraphQLArgument(name = "input") List<GdAdPrice> prices) {
        return bannerDataService.validateAdPrice(prices);
    }

    /**
     * Мутация для массового обновления цен на товары на баннерах.
     * Логика работы ручки близка к {@link AdGraphQlService#updateAds(GridGraphQLContext, GdUpdateAds)},
     * но эта ручка принимает только текстовые баннеры (другие баннеры цены не поддерживают),
     * а изменения любых полей кроме AdPrice игнорируются
     *
     * @param context          контекст
     * @param gdUpdateAdPrices запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateAdPrices")
    public GdUpdateAdsPayload updateAdPrices(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") List<@GraphQLNonNull GdUpdateAdPrice> gdUpdateAdPrices) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateAdPrices(clientId, context.getOperator(), gdUpdateAdPrices);
    }


    /**
     * Мутация для массового обновления организаций на баннерах.
     * Ручка принимает только текстовые баннеры (другие баннеры организации не поддерживают).
     *
     * @param context               контекст
     * @param gdUpdateOrganizations запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateOrganizations",
            description = "Мутация для массового обновления организаций на баннерах")
    public GdUpdateAdsPayload updateOrganizations(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input")
                    List<@GraphQLNonNull GdUpdateOrganization> gdUpdateOrganizations) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateOrganizations(clientId, context.getOperator(), gdUpdateOrganizations);
    }

    /**
     * Мутация для отклонения привязки автоматически предложенной организации.
     * <p>
     * Помечает переданные организации как отклонённые для того, чтобы в дальнейшем снова не предлагать
     * пользователю привязать их вручную, см.
     * {@link ru.yandex.direct.grid.core.entity.banner.service.GridBannerService#enrichBannerWithOrganizationData}
     * Перед отклонением производится проверка на то, что баннер принадлежит клиенту.
     * <p>
     * Ручка принимает только текстовые баннеры (другие баннеры организации не поддерживают).
     *
     * @param context               контекст
     * @param gdUpdateOrganizations запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "rejectAutoOrganizations",
            description = "Мутация для отклонения привязки автоматически предложенной организации")
    public GdUpdateAdsPayload rejectAutoOrganizations(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input")
                    List<@GraphQLNonNull GdUpdateOrganization> gdUpdateOrganizations) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.rejectAutoOrganizations(clientId, gdUpdateOrganizations);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addSitelinkSetToAds")
    public GdUpdateAdsPayload addSitelinkSetsToAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddSitelinkSetToAds input) {
        @SuppressWarnings("ConstantConditions") ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addSitelinkSetsToAds(clientId, context.getOperator(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addTurboLandingToAds")
    public GdUpdateAdsPayload addTurboLandingToAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddTurboLandingToAds input) {
        @SuppressWarnings("ConstantConditions") ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addTurboLandingToAds(clientId, context.getOperator(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addAdImage")
    public GdUpdateAdsPayload addAdImage(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddAdImage input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addBannerImage(clientId, context.getOperator(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addAdCreative")
    public GdUpdateAdsPayload addAdCreative(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddAdCreative input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addCreative(clientId, context.getOperator(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addMulticardsToAds")
    public GdUpdateAdsPayload addMulticardsToAds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddMulticardsToAds input) {
        @SuppressWarnings("ConstantConditions") ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.addMulticardsToAds(clientId, context.getOperator(), input);
    }

    /**
     * Мутация для массового обновления возрастных флагов баннеров.
     * Дает возможность установить только флаг age, только baby_food,
     * установить оба флага и удалить (не для всех ролей).
     *
     * @param context            контекст
     * @param gdUpdateAdAgeFlags запрос на обновление флагов (adIds, adAgeValue, adBabyFoodValue)
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateAdAgeFlags")
    public GdUpdateAdsPayload updateAdAgeFlags(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateAdAgeFlags gdUpdateAdAgeFlags) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerDataService.updateAdAgeFlags(clientId, context.getOperator(), gdUpdateAdAgeFlags);
    }

    /**
     * Мутация для массового копирования баннеров.
     * Дает возможность скопировать множество баннеров одной группы в произвольную группу.
     *
     * @param context контекст
     * @param input   данные для копирования
     * @return результат копирования, содержит id успешно скопированных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "copyAds")
    public GdAdsMassActionPayload copyBanners(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCopyAdsInput input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return adMassActionsService.copyAds(clientId, context.getOperator().getUid(), input);
    }

    /**
     * Подзапрос для получения картинок мультибаннера
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "image")
    public CompletableFuture<GdImage> getImage(@GraphQLContext GdMulticard multicard) {
        return bannerImageDataLoader.get().load(multicard.getImageHash());
    }

    /**
     * Подзапрос для получения причин отклонения карточек мультибаннера
     */
    @GraphQLQuery(name = "rejectReasonIds")
    public CompletableFuture<Set<? extends Long>> getRejectedReasonIds(@GraphQLContext GdMulticard multicard) {
        return multicardReasonDataLoader.get().load(multicard.getId());
    }

    /**
     * Позапрос для получения статуса модерации отдельной карточки мультибаннера.
     * Карточка считается отклоненной, если мультибаннер отклонен, и есть хотя бы одна причина отклонения.
     */
    @GraphQLQuery(name = "status")
    @GraphQLNonNull
    public CompletableFuture<GdMulticardStatus> getMulticardStatus(@GraphQLContext GdMulticard multicard) {
        return multicardReasonDataLoader.get().load(multicard.getId()).thenApply(reasons -> {
            if (multicard.getMulticardSetStatus() == GdMulticardStatus.REJECTED) {
                if (CollectionUtils.isNotEmpty(reasons)) {
                    return GdMulticardStatus.REJECTED;
                } else {
                    return GdMulticardStatus.ACCEPTED;
                }
            }
            return multicard.getMulticardSetStatus();
        });
    }

    /**
     * Запрос для проверки корректности кнопки
     * @param context контекст
     * @param button запрос на обновление кнопки
     * @return результат валидации
     */
    @GraphQLQuery(name = "validateBannerButton")
    public GdValidationResult validateBannerButton(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdBannerButton button) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerButtonService.validateBannerButton(clientId, button);
    }

    /**
     * Мутация для массового обновления кнопок на баннерах
     * Логика работы ручки близка к {@link AdGraphQlService#updateAds(GridGraphQLContext, GdUpdateAds)},
     * но эта ручка работает только для ТГО и игнорирует обновления любых полей за исключением полей кнопки
     * Также эта ручка не поддерживает кастомный текст кнопки баннера
     * @param context          контекст
     * @param gdUpdateBannerButtons запрос на обновление
     * @return результат обновления, содержит id успешно обновленных баннеров и результат валидации
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateBannerButtons")
    public GdUpdateAdsPayload updateBannerButtons(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateBannerButton gdUpdateBannerButtons) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return bannerButtonService.updateBannerButtons(clientId, context.getOperator(), gdUpdateBannerButtons);
    }
}
