package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;
import java.util.function.Function;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannerArchiveUnarchiveService;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.banner.service.BannerSuspendResumeService;
import ru.yandex.direct.core.entity.banner.service.CopyBannerService;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAdsMassAction;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAdsMassActionPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdCopyAdsInput;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Path;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.grid.processing.service.banner.converter.MassActionsDataConverter.getGdBannerMassActionPayload;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
public class AdMassActionsService {

    private final GridValidationService gridValidationService;
    private final AdValidationService adValidationService;
    private final BannerArchiveUnarchiveService bannerArchiveUnarchiveService;
    private final BannerSuspendResumeService bannerSuspendResumeService;
    private final BannerService bannerService;
    private final BannerModerateService bannerModerateService;
    private final CopyBannerService copyBannerService;

    @Autowired
    public AdMassActionsService(GridValidationService gridValidationService,
                                AdValidationService adValidationService,
                                BannerArchiveUnarchiveService bannerArchiveUnarchiveService,
                                BannerSuspendResumeService bannerSuspendResumeService,
                                BannerService bannerService, BannerModerateService bannerModerateService,
                                CopyBannerService copyBannerService) {
        this.gridValidationService = gridValidationService;
        this.adValidationService = adValidationService;
        this.bannerArchiveUnarchiveService = bannerArchiveUnarchiveService;
        this.bannerSuspendResumeService = bannerSuspendResumeService;
        this.bannerService = bannerService;
        this.bannerModerateService = bannerModerateService;
        this.copyBannerService = copyBannerService;
    }

    /**
     * Архивация/разархивация баннеров
     */
    public GdAdsMassActionPayload archiveUnarchiveBanners(ClientId clientId, Long operatorUid, GdAdsMassAction input,
                                                          Boolean archive) {
        return massAction(
                input,
                argInput -> {
                    List<ModelChanges<BannerWithSystemFields>> bannersToArchiveUnarchive =
                            mapList(argInput.getAdIds(), id -> new ModelChanges<>(id, BannerWithSystemFields.class)
                                    .process(archive, BannerWithSystemFields.STATUS_ARCHIVED));
                    return bannerArchiveUnarchiveService
                            .archiveUnarchiveBanners(clientId, operatorUid, bannersToArchiveUnarchive, archive);
                }
        );
    }

    /**
     * Остановить/запустить баннеры
     */
    public GdAdsMassActionPayload suspendResumeBanners(ClientId clientId, Long operatorUid, GdAdsMassAction input,
                                                       Boolean resume) {
        return massAction(
                input,
                argInput -> {
                    List<ModelChanges<BannerWithSystemFields>> bannersToStopResume =
                            mapList(argInput.getAdIds(), id -> new ModelChanges<>(id, BannerWithSystemFields.class)
                                    .process(resume, BannerWithSystemFields.STATUS_SHOW));
                    return bannerSuspendResumeService.suspendResumeBanners(clientId, operatorUid, bannersToStopResume,
                            resume);
                }
        );
    }

    /**
     * Удаление баннеров
     */
    public GdAdsMassActionPayload deleteBanners(ClientId clientId, Long operatorUid, GdAdsMassAction input) {
        return massAction(
                input,
                argInput -> bannerService.deleteBannersPartial(operatorUid, clientId, argInput.getAdIds())
        );
    }

    public GdAdsMassActionPayload remoderateBanners(ClientId clientId, Long operatorUid, GdAdsMassAction input) {
        return massAction(
                input,
                argInput -> bannerModerateService.remoderateBanners(clientId, operatorUid, argInput.getAdIds())
        );
    }

    public GdAdsMassActionPayload copyAds(ClientId clientFromId, Long operatorUid, GdCopyAdsInput input) {
        return massAction(
                input,
                arg -> {
                    adValidationService.validateAdsCopyInput(input);
                    if (arg.getDestinationAdGroupId().isPresent() && arg.getDestinationClientId().isPresent()) {
                        return copyBannerService.copyBanners(
                                operatorUid,
                                arg.getAdIds(),
                                clientFromId,
                                ClientId.fromLong(arg.getDestinationClientId().get()),
                                arg.getDestinationAdGroupId().get());

                    } else {
                        return copyBannerService.copySameAdGroup(operatorUid, arg.getAdIds(), clientFromId);
                    }
                },
                true
        );
    }

    private <T extends GdAdsMassAction> GdAdsMassActionPayload massAction(T input,
                                                                          Function<T, MassResult<Long>> action) {
        return massAction(input, action, false);
    }

    private <T extends GdAdsMassAction> GdAdsMassActionPayload massAction(T input,
                                                                          Function<T, MassResult<Long>> action,
                                                                          boolean hasOutput) {
        List<Long> adIds = input.getAdIds();
        if (isEmpty(input.getAdIds())) {
            return EMPTY_MASS_ACTION_PAYLOAD;
        }
        Path path = path(field(GdAdsMassAction.AD_IDS));

        gridValidationService.validateAdIds(adIds, path);

        MassResult<Long> result = action.apply(input);

        GdValidationResult validationResult = gridValidationService.getValidationResult(result, path);

        return getGdBannerMassActionPayload(result, adIds, validationResult, hasOutput);
    }

    private final static GdAdsMassActionPayload EMPTY_MASS_ACTION_PAYLOAD =
            new GdAdsMassActionPayload()
                    .withProcessedAdIds(emptyList())
                    .withSkippedAdIds(emptyList())
                    .withSuccessCount(0)
                    .withTotalCount(0)
                    .withValidationResult(null);

}
