package ru.yandex.direct.grid.processing.service.banner;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.type.pixels.PixelProvider;
import ru.yandex.direct.core.entity.banner.type.price.BannerWithPriceConstants;
import ru.yandex.direct.core.entity.banner.type.tns.BannerWithTnsConstants;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrice;
import ru.yandex.direct.grid.processing.model.banner.GdAdType;
import ru.yandex.direct.grid.processing.model.banner.GdAgeValue;
import ru.yandex.direct.grid.processing.model.banner.GdBabyFoodValue;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAdContainer;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAdContainerItem;
import ru.yandex.direct.grid.processing.model.banner.GdTurboGalleryParams;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdCreative;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdImage;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddContentPromotionAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddContentPromotionAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddCpmAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddCpmAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddDynamicAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddDynamicAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddHrefToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMcBannerAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMcBannerAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMobileContentAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMobileContentAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMulticard;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMulticardsToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelink;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSet;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSetToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSets;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddTurboLandingToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdContentPromotion;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdCopyAdsInput;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdAgeFlags;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdPrice;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateContentPromotionAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateContentPromotionAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateCpmAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateCpmAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateDynamicAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateDynamicAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMcBannerAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMcBannerAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMobileContentAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMobileContentAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateOrganization;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartCentersItem;
import ru.yandex.direct.grid.processing.model.cliententity.GdPixel;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageSmartCenter;
import ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter;
import ru.yandex.direct.grid.processing.service.validation.GridDefectIds;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.stringIsNotBlank;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.absentValueInField;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.bannerCannotHaveTurboGalleryHref;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.imageSizeNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.insufficientRights;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.priceGreaterThanOld;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.restrictedCharsInField;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.unsupportedBannerType;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_REGULAR_FORMATS;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_SMALL_FORMATS;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_WIDE_FORMATS;
import static ru.yandex.direct.grid.processing.model.cliententity.GdPixelKind.AUDIENCE;
import static ru.yandex.direct.grid.processing.model.cliententity.GdPixelKind.AUDIT;
import static ru.yandex.direct.grid.processing.service.banner.AdTypeUtils.EDITABLE_AD_FEATURE_BY_TYPE;
import static ru.yandex.direct.grid.processing.service.banner.AdTypeUtils.getEditableAdTypes;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataService.TYPES_WITH_TURBO_GALLERY_HREF;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.hrefIsValid;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.StringUtils.ifNotBlank;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
@ParametersAreNonnullByDefault
public class AdValidationService {
    private final GridValidationService gridValidationService;
    static final Set<RbacRole> ROLES_ALLOWED_TO_CHANGE_FLAGS = Set.of();

    private static final Validator<GdAdPrice, Defect> AD_PRICE_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAdPrice> vb = ModelItemValidationBuilder.of(req);
                BigDecimal price = validateAndGetAdPrice(req, vb);
                validateAdPriceOld(req, vb, price);
                return vb.getResult();
            };

    private static final Validator<GdContentPromotion, Defect> CONTENT_PROMOTION_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdContentPromotion> vb = ModelItemValidationBuilder.of(req);

                return vb.getResult();
            };

    private static final Validator<GdUpdateAd, Defect> AD_ITEM_UPDATE_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateAd> vb = ModelItemValidationBuilder.of(req);
                GdAdType adType = req.getAdType();
                vb.item(GdUpdateAd.ID)
                        .check(validId());
                vb.item(GdUpdateAd.AD_PRICE)
                        .checkBy(AD_PRICE_VALIDATOR, When.notNull());
                vb.item(GdUpdateAd.TURBO_GALLERY_PARAMS)
                        .check(validTypeForAddTurboGalleryHref(adType), When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdUpdateDynamicAd, Defect> UPDATE_DYNAMIC_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateDynamicAd> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdUpdateDynamicAd.ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<List<GdAdPrice>, Defect> AD_PRICE_MASS_VALIDATOR =
            req -> {
                ListValidationBuilder<GdAdPrice, Defect> lvb = ListValidationBuilder.of(req);
                lvb.checkEachBy(AD_PRICE_VALIDATOR, When.notNull());
                return lvb.getResult();
            };

    private static final Validator<List<Long>, Defect> ORGANIZATIONS_MASS_VALIDATOR =
            req -> {
                ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(req);
                lvb.checkEach(validId(), When.notNull());
                return lvb.getResult();
            };

    private static final Validator<GdAddSitelinkSetToAds, Defect> SITELINK_SET_ID_WITH_AD_IDS_MASS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddSitelinkSetToAds> vb = ModelItemValidationBuilder.of(req);
                vb.list(GdAddSitelinkSetToAds.AD_IDS)
                        .checkEach(validId());
                vb.item(GdAddSitelinkSetToAds.SITELINK_SET_ID)
                        .check(validId(), When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdAddTurboLandingToAds, Defect> TURBO_LANDIND_ID_WITH_AD_IDS_MASS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddTurboLandingToAds> vb = ModelItemValidationBuilder.of(req);
                vb.list(GdAddTurboLandingToAds.AD_IDS)
                        .checkEach(validId());
                vb.item(GdAddTurboLandingToAds.TURBO_LANDING_ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<GdAddHrefToAds, Defect> HREF_WITH_AD_IDS_MASS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddHrefToAds> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdAddHrefToAds.HREF)
                        .check(validHref());
                vb.list(GdAddHrefToAds.AD_IDS)
                        .checkEach(validId());
                return vb.getResult();
            };

    private static final Validator<GdAddAdImage, Defect> IMAGE_HASH_WITH_AD_IDS_MASS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddAdImage> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdAddAdImage.AD_IMAGE_HASH)
                        .check(stringIsNotBlank(), absentValueInField());
                vb.list(GdAddAdImage.AD_IDS)
                        .checkEach(validId());
                return vb.getResult();
            };

    private static final Validator<GdAddAdCreative, Defect> CREATIVE_ID_WITH_AD_IDS_MASS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddAdCreative> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdAddAdCreative.CREATIVE_ID)
                        .check(validId());
                vb.list(GdAddAdCreative.AD_IDS)
                        .checkEach(validId());
                return vb.getResult();
            };

    private static final Validator<GdAddMulticard, Defect> MULTICARD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddMulticard> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdAddMulticard.HREF)
                        .check(validHref());
                vb.item(GdAddMulticard.IMAGE_HASH)
                        .check(stringIsNotBlank(), absentValueInField());
                return vb.getResult();
            };

    private static final Validator<GdAddMulticardsToAds, Defect> MULTICARDS_WITH_AD_IDS_MASS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddMulticardsToAds> vb = ModelItemValidationBuilder.of(req);
                vb.list(GdAddMulticardsToAds.MULTICARDS)
                        .checkEachBy(MULTICARD_VALIDATOR);
                vb.list(GdAddMulticardsToAds.AD_IDS)
                        .checkEach(validId());
                return vb.getResult();
            };

    private static final Validator<GdUpdateAds, Defect> ADS_UPDATE_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdUpdateAds.AD_UPDATE_ITEMS)
                        .checkEachBy(AD_ITEM_UPDATE_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdUpdateDynamicAds, Defect> UPDATE_DYNAMIC_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateDynamicAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdUpdateDynamicAds.AD_UPDATE_ITEMS)
                        .checkEachBy(UPDATE_DYNAMIC_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdAddAd, Defect> ADD_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAddAd.AD_GROUP_ID)
                        .check(validId());
                vb.item(GdAddAd.VCARD_ID)
                        .check(validId(), When.notNull());
                vb.item(GdAddAd.SITELINKS_SET_ID)
                        .check(validId(), When.notNull());
                vb.item(GdAddAd.CREATIVE_ID)
                        .check(validId(), When.notNull());
                vb.list(GdAddAd.CALLOUT_IDS)
                        .checkEach(validId(), When.notNull());
                vb.item(GdAddAd.AD_PRICE)
                        .checkBy(AD_PRICE_VALIDATOR, When.notNull());
                vb.item(GdAddAd.TURBO_GALLERY_HREF)
                        .check(Constraint.fromPredicate(
                                r -> TYPES_WITH_TURBO_GALLERY_HREF.contains(req.getAdType()),
                                bannerCannotHaveTurboGalleryHref()), When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdAddDynamicAd, Defect> ADD_DYNAMIC_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddDynamicAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAddDynamicAd.AD_GROUP_ID)
                        .check(validId());
                vb.item(GdAddDynamicAd.VCARD_ID)
                        .check(validId());
                vb.item(GdAddDynamicAd.SITELINKS_SET_ID)
                        .check(validId());
                vb.list(GdAddDynamicAd.CALLOUT_IDS)
                        .checkEach(validId());
                return vb.getResult();
            };

    private static final Validator<GdUpdateMcBannerAd, Defect> UPDATE_MCBANNER_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateMcBannerAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdUpdateMcBannerAd.ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<GdUpdateContentPromotionAd, Defect> UPDATE_CONTENT_PROMOTION_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateContentPromotionAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdUpdateContentPromotionAd.ID)
                        .check(validId());
                vb.item(GdUpdateContentPromotionAd.CONTENT_PROMOTION)
                        .checkBy(CONTENT_PROMOTION_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdUpdateMcBannerAds, Defect> UPDATE_MCBANNER_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateMcBannerAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdUpdateMcBannerAds.AD_UPDATE_ITEMS)
                        .checkEachBy(UPDATE_MCBANNER_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdUpdateContentPromotionAds, Defect> UPDATE_CONTENT_PROMOTION_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateContentPromotionAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdUpdateContentPromotionAds.AD_UPDATE_ITEMS)
                        .checkEachBy(UPDATE_CONTENT_PROMOTION_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdAddMcBannerAd, Defect> ADD_MCBANNER_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddMcBannerAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAddMcBannerAd.AD_GROUP_ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<GdAddContentPromotionAd, Defect> ADD_CONTENT_PROMOTION_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddContentPromotionAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAddContentPromotionAd.AD_GROUP_ID)
                        .check(validId());
                vb.item(GdAddContentPromotionAd.CONTENT_PROMOTION)
                        .checkBy(CONTENT_PROMOTION_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdAddMcBannerAds, Defect> ADD_MCBANNER_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddMcBannerAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddMcBannerAds.AD_ADD_ITEMS)
                        .checkEachBy(ADD_MCBANNER_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdUpdateMobileContentAd, Defect> UPDATE_MOBILE_CONTENT_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateMobileContentAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdUpdateMobileContentAd.ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<GdUpdateMobileContentAds, Defect> UPDATE_MOBILE_CONTENT_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateMobileContentAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdUpdateMobileContentAds.AD_UPDATE_ITEMS)
                        .checkEachBy(UPDATE_MOBILE_CONTENT_AD_VALIDATOR);

                return vb.getResult();
            };


    private static final Validator<GdAddMobileContentAd, Defect> ADD_MOBILE_CONTENT_AD_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddMobileContentAd> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAddMobileContentAd.AD_GROUP_ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<GdAddMobileContentAds, Defect> ADD_MOBILE_CONTENT_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddMobileContentAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddMobileContentAds.AD_ADD_ITEMS)
                        .checkEachBy(ADD_MOBILE_CONTENT_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdAddContentPromotionAds, Defect> ADD_CONTENT_PROMOTION_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddContentPromotionAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddContentPromotionAds.AD_ADD_ITEMS)
                        .checkEachBy(ADD_CONTENT_PROMOTION_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdAddAds, Defect> ADD_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddAds.AD_ADD_ITEMS)
                        .checkEachBy(ADD_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdAddDynamicAds, Defect> ADD_DYNAMIC_ADS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddDynamicAds> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddDynamicAds.AD_ADD_ITEMS)
                        .checkEachBy(ADD_DYNAMIC_AD_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdAddSitelink, Defect> ADD_SITELINK_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddSitelink> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAddSitelink.TURBOLANDING_ID)
                        .check(validId(), When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdAddSitelinkSet, Defect> ADD_SITELINK_SET_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddSitelinkSet> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddSitelinkSet.SITELINKS)
                        .check(notEmptyCollection())
                        .checkEachBy(ADD_SITELINK_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdAddSitelinkSets, Defect> ADD_SITELINK_SETS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddSitelinkSets> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddSitelinkSets.SITELINK_SETS_ADD_ITEMS)
                        .check(notEmptyCollection())
                        .checkEachBy(ADD_SITELINK_SET_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdImageSmartCenter, Defect> IMAGE_SMART_CENTER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdImageSmartCenter> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdImageSmartCenter.X)
                        .check(CommonConstraints.notNull());
                vb.item(GdImageSmartCenter.Y)
                        .check(CommonConstraints.notNull());
                vb.item(GdImageSmartCenter.RATIO)
                        .check(CommonConstraints.notNull());

                return vb.getResult();
            };

    private static final Validator<GdUpdateSmartCentersItem, Defect> UPDATE_SMART_CENTERS_ITEM_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateSmartCentersItem> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdUpdateSmartCentersItem.SIZE)
                        .check(validateImageSize());
                vb.item(GdUpdateSmartCentersItem.SMART_CENTER)
                        .checkBy(IMAGE_SMART_CENTER_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<List<GdUpdateSmartCentersItem>, Defect> UPDATE_SMART_CENTERS_VALIDATOR =
            req -> {
                ListValidationBuilder<GdUpdateSmartCentersItem, Defect> lvb = ListValidationBuilder.of(req);

                lvb.checkEachBy(UPDATE_SMART_CENTERS_ITEM_VALIDATOR, When.notNull());
                return lvb.getResult();
            };


    private static final Validator<GdLastChangedAdContainerItem, Defect> LAST_CHANGED_AD_CONTAINER_ITEM_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdLastChangedAdContainerItem> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdLastChangedAdContainerItem.CAMPAIGN_ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<GdLastChangedAdContainer, Defect> LAST_CHANGED_AD_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdLastChangedAdContainer> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdLastChangedAdContainer.ITEMS)
                        .checkEachBy(LAST_CHANGED_AD_CONTAINER_ITEM_VALIDATOR);
                return vb.getResult();
            };

    private static Constraint<Optional<GdAgeValue>, Defect> rightsForDeleteAgeFlagAreCorrect(User operator) {
        return Constraint.fromPredicate(
                gdAgeValue -> {
                    if (gdAgeValue.isEmpty()) {
                        return ROLES_ALLOWED_TO_CHANGE_FLAGS.contains(operator.getRole());
                    }
                    return true;
                },
                insufficientRights());
    }

    private static Constraint<Optional<GdBabyFoodValue>, Defect> rightsForDeleteBabyFoodFlagAreCorrect(User operator) {
        return Constraint.fromPredicate(
                gdBabyFoodValue -> {
                    if (gdBabyFoodValue.isEmpty()) {
                        return ROLES_ALLOWED_TO_CHANGE_FLAGS.contains(operator.getRole());
                    }
                    return true;
                },
                insufficientRights());
    }

    private static Validator<GdUpdateAdAgeFlags, Defect> updateAdAgeFlagsValidator(User operator) {
        return req -> {
            ModelItemValidationBuilder<GdUpdateAdAgeFlags> vb = ModelItemValidationBuilder.of(req);
            vb.list(GdUpdateAdAgeFlags.AD_IDS)
                    .checkEach(validId());

            vb.item(GdUpdateAdAgeFlags.AD_AGE_VALUE)
                    .check(rightsForDeleteAgeFlagAreCorrect(operator));

            vb.item(GdUpdateAdAgeFlags.AD_BABY_FOOD_VALUE)
                    .check(rightsForDeleteBabyFoodFlagAreCorrect(operator));
            return vb.getResult();
        };
    }

    public AdValidationService(GridValidationService gridValidationService) {
        this.gridValidationService = gridValidationService;
    }

    private static Constraint<GdTurboGalleryParams, Defect> validTypeForAddTurboGalleryHref(GdAdType adType) {
        return Constraint.fromPredicate(params -> {
            if (params != null) {
                return TYPES_WITH_TURBO_GALLERY_HREF.contains(adType);
            }
            return true;
        }, bannerCannotHaveTurboGalleryHref());
    }

    /**
     * Проверяет старую цену на товар на баннере и возвращает ее если ее можно распарсить, иначе возвращает null
     */
    private static void validateAdPriceOld(GdAdPrice req, ModelItemValidationBuilder<GdAdPrice> vb,
                                           @Nullable BigDecimal price) {
        ItemValidationBuilder<String, Defect> ivb = vb.item(GdAdPrice.PRICE_OLD);
        try {
            BigDecimal priceOld = ifNotBlank(req.getPriceOld(), AdMutationDataConverter::webPriceToCorePrice);
            ivb
                    .check(s -> greaterThan(BigDecimal.ZERO).apply(priceOld), When.isValidAnd(When.notNull()))
                    .check(s -> notGreaterThan(BannerWithPriceConstants.MAX_BANNER_PRICE).apply(priceOld),
                            When.isValidAnd(When.notNull()))
                    .check(s -> greaterThan(price).apply(priceOld), priceGreaterThanOld(),
                            When.isValidAnd(When.isTrue(price != null && priceOld != null)));
        } catch (NumberFormatException e) {
            ivb.getResult().addError(invalidValue());
        }
    }

    /**
     * Проверяет новую цену на товар на баннере и возвращает ее если ее можно распарсить, иначе возвращает null
     */
    @Nullable
    private static BigDecimal validateAndGetAdPrice(GdAdPrice req, ModelItemValidationBuilder<GdAdPrice> vb) {
        ItemValidationBuilder<String, Defect> ivb = vb.item(GdAdPrice.PRICE);
        try {
            BigDecimal price = AdMutationDataConverter.webPriceToCorePrice(req.getPrice());
            ivb
                    .check(s -> greaterThan(BigDecimal.ZERO).apply(price), When.isValid())
                    .check(s -> notGreaterThan(BannerWithPriceConstants.MAX_BANNER_PRICE).apply(price),
                            When.isValid());
            return price;
        } catch (NumberFormatException e) {
            ivb.getResult().addError(invalidValue());
            return null;
        }
    }

    private static Constraint<GdPixel, Defect> audiencePixelUrlIsCorrect() {
        return Constraint.fromPredicate(pixel -> {
            if (pixel.getKind() == AUDIENCE) {
                return PixelProvider.fromUrl(pixel.getUrl()) == PixelProvider.YANDEXAUDIENCE;
            }
            return true;
        }, invalidAudiencePixelFormat());
    }

    private static Constraint<GdPixel, Defect> auditPixelUrlIsCorrect() {
        return Constraint.fromPredicate(pixel -> {
            if (pixel.getKind() == AUDIT) {
                PixelProvider pixelProvider = PixelProvider.fromUrl(pixel.getUrl());
                return pixelProvider != null && pixelProvider != PixelProvider.YANDEXAUDIENCE;
            }
            return true;
        }, invalidAuditPixelFormat());
    }

    static Defect<Void> invalidAudiencePixelFormat() {
        return new Defect<>(GridDefectIds.Pixels.INVALID_AUDIENCE_PIXEL_FORMAT);
    }

    static Defect<Void> invalidAuditPixelFormat() {
        return new Defect<>(GridDefectIds.Pixels.INVALID_AUDIT_PIXEL_FORMAT);
    }


    private static Constraint<String, Defect> validateImageSize() {
        return fromPredicate(size -> SUPPORTED_SMALL_FORMATS.contains(size) ||
                        SUPPORTED_REGULAR_FORMATS.contains(size) ||
                        SUPPORTED_WIDE_FORMATS.contains(size),
                imageSizeNotFound());
    }

    private static Constraint<GdAdType, Defect> editableAdType(Set<GdAdType> editableAdTypes) {
        return Constraint.fromPredicate(
                gdAdType -> !EDITABLE_AD_FEATURE_BY_TYPE.containsKey(gdAdType) || editableAdTypes.contains(gdAdType),
                unsupportedBannerType());
    }

    private static final Validator<GdCopyAdsInput, Defect> ADS_COPY_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdCopyAdsInput> vb = ModelItemValidationBuilder.of(req);

        vb.item(GdCopyAdsInput.DESTINATION_CLIENT_ID)
                .check(notNull())
                .check(CommonConstraints.notEmpty(), When.isTrue(req.getDestinationAdGroupId().isPresent()));

        vb.item(GdCopyAdsInput.DESTINATION_AD_GROUP_ID)
                .check(notNull())
                .check(CommonConstraints.notEmpty(), When.isTrue(req.getDestinationClientId().isPresent()));

        return vb.getResult();
    };


    /**
     * Валидация запроса на обновления баннеров
     */
    void validateUpdateAdsRequest(GdUpdateAds gdUpdateAds) {
        gridValidationService.applyValidator(ADS_UPDATE_VALIDATOR, gdUpdateAds, false);
    }

    void validateUpdateDynamicAdsRequest(GdUpdateDynamicAds gdUpdateAds) {
        gridValidationService.applyValidator(UPDATE_DYNAMIC_ADS_VALIDATOR, gdUpdateAds, false);
    }

    void validateAddAdsRequest(GdAddAds input) {
        gridValidationService.applyValidator(ADD_ADS_VALIDATOR, input, false);
    }

    void validateAddDynamicAdsRequest(GdAddDynamicAds input) {
        gridValidationService.applyValidator(ADD_DYNAMIC_ADS_VALIDATOR, input, false);
    }

    ValidationResult<GdAddCpmAds, Defect> validateAddCpmAdsRequest(GdAddCpmAds input, Set<String> enabledFeatures) {
        return new AddCpmAdsValidator(enabledFeatures).apply(input);
    }

    ValidationResult<GdUpdateCpmAds, Defect> validateUpdateCpmAdsRequest(GdUpdateCpmAds input,
                                                                         Set<String> enabledFeatures) {
        return new UpdateCpmAdsValidator(enabledFeatures).apply(input);
    }

    void validateUpdateMcBannerAdsRequest(GdUpdateMcBannerAds input) {
        gridValidationService.applyValidator(UPDATE_MCBANNER_ADS_VALIDATOR, input, false);
    }

    void validateUpdateContentPromotionAdsRequest(GdUpdateContentPromotionAds input) {
        gridValidationService.applyValidator(UPDATE_CONTENT_PROMOTION_ADS_VALIDATOR, input, false);
    }

    void validateHref(String input) {
        gridValidationService.applyValidator(hrefIsValid(), input, false);
    }

    void validateAddMcBannerAdsRequest(GdAddMcBannerAds input) {
        gridValidationService.applyValidator(ADD_MCBANNER_ADS_VALIDATOR, input, false);
    }

    void validateAddContentPromotionAdsRequest(GdAddContentPromotionAds input) {
        gridValidationService.applyValidator(ADD_CONTENT_PROMOTION_ADS_VALIDATOR, input, false);
    }

    void validateUpdateMobileContentAdsRequest(GdUpdateMobileContentAds input) {
        gridValidationService.applyValidator(UPDATE_MOBILE_CONTENT_ADS_VALIDATOR, input, false);
    }

    void validateAddMobileContentAdsRequest(GdAddMobileContentAds input) {
        gridValidationService.applyValidator(ADD_MOBILE_CONTENT_ADS_VALIDATOR, input, false);
    }

    void validateAddSitelinkSetsRequest(GdAddSitelinkSets input) {
        gridValidationService.applyValidator(ADD_SITELINK_SETS_VALIDATOR, input, false);
    }

    void validateUpdateAdPricesRequest(List<GdUpdateAdPrice> pricesByAdId) {
        gridValidationService.applyValidator(AD_PRICE_MASS_VALIDATOR,
                mapList(pricesByAdId, GdUpdateAdPrice::getPrice), false);
    }

    void validateUpdateOrganizationsRequest(List<GdUpdateOrganization> request) {
        gridValidationService.applyValidator(ORGANIZATIONS_MASS_VALIDATOR,
                mapList(request, GdUpdateOrganization::getPermalinkId), false);
    }

    void validateRejectAutoOrganizationsRequest(List<GdUpdateOrganization> organizations, Set<Long> existingBannerIds) {
        gridValidationService.applyValidator(new RejectAutoOrganizationsMassValidator(existingBannerIds),
                organizations, false);
    }

    void validateAddSitelinkSetsToAdsRequest(GdAddSitelinkSetToAds input) {
        gridValidationService.applyValidator(SITELINK_SET_ID_WITH_AD_IDS_MASS_VALIDATOR, input, false);
    }

    void validateAddTurboLandingToAdsRequest(GdAddTurboLandingToAds input) {
        gridValidationService.applyValidator(TURBO_LANDIND_ID_WITH_AD_IDS_MASS_VALIDATOR, input, false);
    }

    void validateAddHrefToAdsRequest(GdAddHrefToAds input) {
        gridValidationService.applyValidator(HREF_WITH_AD_IDS_MASS_VALIDATOR, input, false);
    }

    void validateAddBannerImageRequest(GdAddAdImage input) {
        gridValidationService.applyValidator(IMAGE_HASH_WITH_AD_IDS_MASS_VALIDATOR, input, false);
    }

    public void validateAddCreativeRequest(GdAddAdCreative input) {
        gridValidationService.applyValidator(CREATIVE_ID_WITH_AD_IDS_MASS_VALIDATOR, input, false);
    }

    public void validateAddMulticardsRequest(GdAddMulticardsToAds input) {
        gridValidationService.applyValidator(MULTICARDS_WITH_AD_IDS_MASS_VALIDATOR, input, false);
    }

    ValidationResult<List<GdAdPrice>, Defect> validateAdPrices(List<GdAdPrice> prices) {
        return AD_PRICE_MASS_VALIDATOR.apply(prices);
    }

    void validateUpdateSmartCenters(List<GdUpdateSmartCentersItem> smartCenters) {
        gridValidationService.applyValidator(UPDATE_SMART_CENTERS_VALIDATOR, smartCenters, false);
    }

    void validateGetLastChangedAdRequest(GdLastChangedAdContainer input) {
        gridValidationService.applyValidator(LAST_CHANGED_AD_CONTAINER_VALIDATOR, input, false);
    }

    void validateUpdateAdAgeFlags(User operator, GdUpdateAdAgeFlags input) {
        gridValidationService.applyValidator(updateAdAgeFlagsValidator(operator), input, false);
    }

    public void validateAdsCopyInput(GdCopyAdsInput input) {
        gridValidationService.applyValidator(ADS_COPY_VALIDATOR, input, false);
    }

    private static class AddCpmAdsValidator implements Validator<GdAddCpmAds, Defect> {

        private final Set<String> enabledFeatures;

        private AddCpmAdsValidator(Set<String> enabledFeatures) {
            this.enabledFeatures = enabledFeatures;
        }

        @Override
        public ValidationResult<GdAddCpmAds, Defect> apply(GdAddCpmAds request) {
            ModelItemValidationBuilder<GdAddCpmAds> vb = ModelItemValidationBuilder.of(request);

            Set<GdAdType> editableAdTypes = getEditableAdTypes(enabledFeatures);

            vb.list(GdAddCpmAds.AD_ADD_ITEMS)
                    .checkEachBy(requestItem -> {
                        ModelItemValidationBuilder<GdAddCpmAd> itemValidationBuilder =
                                ModelItemValidationBuilder.of(requestItem);

                        itemValidationBuilder.list(GdAddCpmAd.PIXELS)
                                .checkEach(audiencePixelUrlIsCorrect())
                                .checkEach(auditPixelUrlIsCorrect());

                        itemValidationBuilder.item(GdAddCpmAd.AD_TYPE)
                                .check(editableAdType(editableAdTypes));

                        itemValidationBuilder.item(GdAddCpmAd.TNS_ID)
                                .check(notBlank(), When.notNull())
                                .check(fromPredicate(tnsId -> tnsId.matches("^[a-zA-Z0-9]+\\z"),
                                                restrictedCharsInField()),
                                        When.isValidAnd(When.notNull()))
                                .check(StringConstraints.maxStringLength(BannerWithTnsConstants.TNS_ID_MAX_LENGTH),
                                        When.isValidAnd(When.notNull()));

                        return itemValidationBuilder.getResult();
                    });

            return vb.getResult();
        }
    }

    private static class UpdateCpmAdsValidator implements Validator<GdUpdateCpmAds, Defect> {

        private final Set<String> enabledFeatures;

        private UpdateCpmAdsValidator(Set<String> enabledFeatures) {
            this.enabledFeatures = enabledFeatures;
        }

        @Override
        public ValidationResult<GdUpdateCpmAds, Defect> apply(GdUpdateCpmAds request) {
            ModelItemValidationBuilder<GdUpdateCpmAds> vb = ModelItemValidationBuilder.of(request);

            Set<GdAdType> editableAdTypes = getEditableAdTypes(enabledFeatures);

            vb.list(GdUpdateCpmAds.AD_UPDATE_ITEMS)
                    .checkEachBy(requestItem -> {
                        ModelItemValidationBuilder<GdUpdateCpmAd> itemValidationBuilder =
                                ModelItemValidationBuilder.of(requestItem);

                        itemValidationBuilder.list(GdUpdateCpmAd.PIXELS)
                                .checkEach(audiencePixelUrlIsCorrect())
                                .checkEach(auditPixelUrlIsCorrect());

                        itemValidationBuilder.item(GdUpdateCpmAd.AD_TYPE)
                                .check(editableAdType(editableAdTypes));

                        itemValidationBuilder.item(GdUpdateCpmAd.TNS_ID)
                                .check(notBlank(), When.notNull())
                                .check(fromPredicate(tnsId -> tnsId.matches("^[a-zA-Z0-9]+\\z"),
                                                restrictedCharsInField()),
                                        When.isValidAnd(When.notNull()))
                                .check(StringConstraints.maxStringLength(BannerWithTnsConstants.TNS_ID_MAX_LENGTH),
                                        When.isValidAnd(When.notNull()));

                        return itemValidationBuilder.getResult();
                    });

            return vb.getResult();
        }
    }

    private static class RejectAutoOrganizationsMassValidator implements Validator<List<GdUpdateOrganization>, Defect> {

        private final Set<Long> existingBannerIds;

        private RejectAutoOrganizationsMassValidator(Set<Long> existingBannerIds) {
            this.existingBannerIds = existingBannerIds;
        }

        @Override
        public ValidationResult<List<GdUpdateOrganization>, Defect> apply(List<GdUpdateOrganization> request) {
            ListValidationBuilder<GdUpdateOrganization, Defect> lvb = ListValidationBuilder.of(request);
            lvb.checkEachBy(req -> {
                ModelItemValidationBuilder<GdUpdateOrganization> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdUpdateOrganization.BANNER_ID)
                        .check(notNull())
                        .check(validId())
                        .check(fromPredicate(existingBannerIds::contains, objectNotFound()), When.isValid());
                vb.item(GdUpdateOrganization.PERMALINK_ID)
                        .check(notNull())
                        .check(validId());
                return vb.getResult();
            }, When.notNull());
            return lvb.getResult();
        }
    }
}
