package ru.yandex.direct.grid.processing.service.banner;

import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.bannersystem.BannerSystemClient;
import ru.yandex.direct.bannersystem.BsUriFactory;
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportDimension;
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportGroupByDate;
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportMetric;
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRequest;
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportResponse;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.model.client.GdClient;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class BannerAimingGraphQlService {
    private static final Logger logger = LoggerFactory.getLogger(BannerAimingGraphQlService.class);
    private static final DateTimeFormatter BS_MASTER_REPORT_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final Map<String, String> correctPageName = Map.of("yandexmain", "yandex.ru");
    private final BannerService bannerService;
    private final CampaignRepository campaignService;
    private final ShardHelper shardHelper;
    private final BannerSystemClient bannerSystemClient;

    @Autowired
    public BannerAimingGraphQlService(
            BannerService bannerService, CampaignRepository campaignService,
            ShardHelper shardHelper,
            BannerSystemClient bannerSystemClient) {
        this.bannerService = bannerService;
        this.campaignService = campaignService;
        this.shardHelper = shardHelper;
        this.bannerSystemClient = bannerSystemClient;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "topShowedBannerDomains")
    public List<@GraphQLNonNull String> getTopShowedBannerDomains(
            @GraphQLArgument(name = "bannerId") @GraphQLNonNull Long bid,
            @GraphQLContext GdClient client) {

        Long clientId = client.getInfo().getId();
        List<BannerWithSystemFields> banners = bannerService.getBannersByIds(
                Collections.singleton(bid));
        if (banners.size() == 1) {
            BannerWithSystemFields banner = banners.get(0);
            Long campaignId = banner.getCampaignId();
            int shard = shardHelper.getShardByCampaignId(campaignId);
            Map<Long, CampaignSimple>
                    campaigns = campaignService.getCampaignsSimple(shard, Collections.singleton(
                    campaignId));
            CampaignSimple campaign = campaigns.get(campaignId);
            LocalDate today = LocalDate.now();
            LocalDate weekAgo = today.minusDays(7);
            if (campaign.getClientId().equals(clientId)) {
                try {
                    MasterReportRequest request = new MasterReportRequest()
                            .setWithVat(0)
                            .setWithPerformanceCoverage(0)
                            .setWithDiscount(1)
                            .setLang("ru")
                            .setGroupByDate(MasterReportGroupByDate.NONE)
                            .setDontGroupAndFilterZerosForTotals(1)
                            .setCountableFieldsByTargettype(Collections.emptyList())
                            .setCurrency(campaign.getCurrency().name())
                            .setCountableFields(Set.of(MasterReportMetric.SHOWS))
                            .setGroupBy(Set.of(MasterReportDimension.PAGE_NAME))
                            .setDateFrom(BS_MASTER_REPORT_DATE_FORMATTER.format(weekAgo))
                            .setDateTo(BS_MASTER_REPORT_DATE_FORMATTER.format(today))
                            .setFiltersPre(
                                    Map.of(
                                            MasterReportRequest.ReportColumn.BID,
                                            Map.of(MasterReportRequest.FilterOperator.EQ, List.of(String.valueOf(bid)))
                                    )
                            )
                            .setOrderIds(Collections.singletonList(campaign.getOrderId().toString()))
                            .setOrderBy(Collections.singletonList(
                                    new MasterReportRequest.OrderBy().setField("Shows").setDir("desc")))
                            .setLimits(
                                    new MasterReportRequest.Limits().setLimit(10 + 1).setOffset(0));
                    MasterReportResponse response = bannerSystemClient
                            .doRequest(BsUriFactory.MASTER_REPORT, request, UUID.randomUUID(),
                                    Duration.ofSeconds(5));
                    if (Integer.valueOf(0).equals(response.getStatus()) && response.getData() != null) {
                        return mapList(
                                response.getData(),
                                row -> correctPageName.getOrDefault(row.getPageName(), row.getPageName())
                        );
                    }
                } catch (RuntimeException e) {
                    logger.error("Failed to get top showed banner pages, bid = " + bid, e);
                }
            }
        }
        return Collections.emptyList();
    }

}
