package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.banner.model.BannerWithButton;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.type.button.BannerWithButtonHelper;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.GdBannerButton;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateBannerButton;
import ru.yandex.direct.grid.processing.model.constants.GdButtonAction;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.toCoreButtonAction;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;

@Service
@ParametersAreNonnullByDefault
public class BannerButtonService {

    private final PathNodeConverterProvider bannerButtonPathNodeConverterProvider;

    private final BannerWithButtonHelper bannerWithButtonHelper;

    private final FeatureService featureService;

    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;

    private final BannerDataService bannerDataService;

    private final GridValidationService gridValidationService;

    private static final PathNodeConverter VALIDATE_BANNER_BUTTON_PATH_CONVERTER =
            SkipByDefaultMappingPathNodeConverter.builder()
                    .replace(BannerWithButton.BUTTON_ACTION.name(), GdBannerButton.ACTION.name())
                    .replace(BannerWithButton.BUTTON_HREF.name(), GdBannerButton.HREF.name())
                    .build();

    @Autowired
    public BannerButtonService(BannerWithButtonHelper bannerWithButtonHelper, FeatureService featureService,
                               BannersUpdateOperationFactory bannersUpdateOperationFactory,
                               BannerDataService bannerDataService,
                               GridValidationService gridValidationService) {
        this.bannerWithButtonHelper = bannerWithButtonHelper;
        this.featureService = featureService;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.bannerDataService = bannerDataService;
        this.gridValidationService = gridValidationService;
        this.bannerButtonPathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, VALIDATE_BANNER_BUTTON_PATH_CONVERTER)
                .build();
    }

    @Nullable
    public GdValidationResult validateBannerButton(ClientId clientId, @Nonnull GdBannerButton button) {
        var validator = createBannerButtonValidator(clientId);
        ValidationResult<GdBannerButton, Defect> vr = validator.apply(button);

        if (hasValidationIssues(vr)) {
            return GridValidationResultConversionService
                    .buildGridValidationResult(vr, emptyPath(), bannerButtonPathNodeConverterProvider);
        }
        return null;
    }

    @Nonnull
    public GdUpdateAdsPayload updateBannerButtons(ClientId clientId, User operator,
                                           @Nonnull @GraphQLNonNull GdUpdateBannerButton request) {
        if (isEmpty(request.getBannerIds())) {
            return new GdUpdateAdsPayload().withUpdatedAds(emptyList());
        }

        gridValidationService.applyValidator(createMassBannerButtonValidator(clientId), request, false);
        GdBannerButton gdUpdateBannerButton = request.getButton();

        List<ModelChanges<TextBanner>> modelChanges = StreamEx.of(request.getBannerIds())
                .map(bannerId -> ModelChanges.build(bannerId, TextBanner.class,
                                BannerWithButton.BUTTON_ACTION, toCoreButtonAction(gdUpdateBannerButton))
                        .process(gdUpdateBannerButton != null ? gdUpdateBannerButton.getHref() : null,
                                BannerWithButton.BUTTON_HREF)
                        .process(null, BannerWithButton.BUTTON_CAPTION))
                .toList();
        MassResult<Long> result = bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, operator.getUid(), clientId, TextBanner.class)
                .prepareAndApply();

        return bannerDataService.toGdUpdateAdsPayload(result);
    }

    private Validator<GdBannerButton, Defect> createBannerButtonValidator(ClientId clientId){
        return req -> {
            ModelItemValidationBuilder<GdBannerButton> vb = ModelItemValidationBuilder.of(req);
            vb.item(GdBannerButton.HREF).check(StringConstraints.notBlank());
            vb.item(GdBannerButton.HREF).check(validHref());
            vb.item(GdBannerButton.ACTION).check(CommonConstraints.notNull());
            vb.item(GdBannerButton.ACTION).check(Constraint.fromPredicate(isValidButtonAction(clientId),
                    CommonDefects.invalidValue()));

            return vb.getResult();
        };
    }

    private Predicate<GdButtonAction> isValidButtonAction(ClientId clientId) {
        var clientFeatures = featureService.getEnabledForClientId(clientId);
        var allowedButtonActions = bannerWithButtonHelper.getAllowedButtonActions(clientFeatures)
                .stream().map(GdButtonAction::fromSource).collect(Collectors.toList());
        return allowedButtonActions::contains;
    }

    private Validator<GdUpdateBannerButton, Defect> createMassBannerButtonValidator(ClientId clientId){
        return req -> {
            var buttonValidator = createBannerButtonValidator(clientId);

            ModelItemValidationBuilder<GdUpdateBannerButton> vb = ModelItemValidationBuilder.of(req);
            vb.list(GdUpdateBannerButton.BANNER_IDS).checkEach(validId());
            vb.item(GdUpdateBannerButton.BUTTON).checkBy(buttonValidator, When.notNull());
            return vb.getResult();
        };
    }
}
