package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.Age;
import ru.yandex.direct.core.entity.banner.model.BabyFood;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurer;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurerSystem;
import ru.yandex.direct.core.entity.banner.model.BannerMulticard;
import ru.yandex.direct.core.entity.banner.model.BannerMulticardSetStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerTurboLandingStatusModerate;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.banner.model.InternalModerationInfo;
import ru.yandex.direct.core.entity.banner.model.ModerateBannerPage;
import ru.yandex.direct.core.entity.banner.model.NewMobileContentPrimaryAction;
import ru.yandex.direct.core.entity.banner.model.NewReflectedAttribute;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.model.TurboAppMetaContent;
import ru.yandex.direct.core.entity.banner.type.pixels.PixelProvider;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.image.converter.BannerImageConverter;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService;
import ru.yandex.direct.core.entity.organization.model.PermalinkAssignType;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.banner.model.GdiAbstractBannerImage;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerButton;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerContentPromotion;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerCreative;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerCreativeType;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerFilter;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageAvatarsHost;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageNamespace;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerOrderBy;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerPrimaryStatus;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerShowType;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatus;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusVCardModeration;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerTurboLanding;
import ru.yandex.direct.grid.core.entity.banner.model.GdiButtonAction;
import ru.yandex.direct.grid.core.entity.banner.model.GdiInternalBannerExtraInfo;
import ru.yandex.direct.grid.core.entity.banner.model.GdiInternalBannerImageResource;
import ru.yandex.direct.grid.core.entity.banner.model.GdiSitelink;
import ru.yandex.direct.grid.core.entity.banner.service.GridBannerConstants;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.campaign.GdCampaignPrimaryStatus;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.banner.GdAd;
import ru.yandex.direct.grid.processing.model.banner.GdAdAccess;
import ru.yandex.direct.grid.processing.model.banner.GdAdFeatures;
import ru.yandex.direct.grid.processing.model.banner.GdAdFilter;
import ru.yandex.direct.grid.processing.model.banner.GdAdOrderBy;
import ru.yandex.direct.grid.processing.model.banner.GdAdOrderByField;
import ru.yandex.direct.grid.processing.model.banner.GdAdPermalinkAssignType;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrice;
import ru.yandex.direct.grid.processing.model.banner.GdAdPriceCurrency;
import ru.yandex.direct.grid.processing.model.banner.GdAdPricePrefix;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrimaryStatus;
import ru.yandex.direct.grid.processing.model.banner.GdAdResources;
import ru.yandex.direct.grid.processing.model.banner.GdAdStatus;
import ru.yandex.direct.grid.processing.model.banner.GdAdType;
import ru.yandex.direct.grid.processing.model.banner.GdAdWithTotals;
import ru.yandex.direct.grid.processing.model.banner.GdAdsContainer;
import ru.yandex.direct.grid.processing.model.banner.GdAdsContext;
import ru.yandex.direct.grid.processing.model.banner.GdAgeValue;
import ru.yandex.direct.grid.processing.model.banner.GdBabyFoodValue;
import ru.yandex.direct.grid.processing.model.banner.GdBannerAdditionalHref;
import ru.yandex.direct.grid.processing.model.banner.GdBannerButton;
import ru.yandex.direct.grid.processing.model.banner.GdBannerMeasurer;
import ru.yandex.direct.grid.processing.model.banner.GdBannerMeasurerSystem;
import ru.yandex.direct.grid.processing.model.banner.GdBannerTurbolanding;
import ru.yandex.direct.grid.processing.model.banner.GdBannerTurbolandingStatusModerate;
import ru.yandex.direct.grid.processing.model.banner.GdCPCVideoAd;
import ru.yandex.direct.grid.processing.model.banner.GdCPMAudioBannerAd;
import ru.yandex.direct.grid.processing.model.banner.GdCPMBannerAd;
import ru.yandex.direct.grid.processing.model.banner.GdCPMGeoPinBannerAd;
import ru.yandex.direct.grid.processing.model.banner.GdCPMOutdoorBannerAd;
import ru.yandex.direct.grid.processing.model.banner.GdContentPromotionAd;
import ru.yandex.direct.grid.processing.model.banner.GdDynamicAd;
import ru.yandex.direct.grid.processing.model.banner.GdFlags;
import ru.yandex.direct.grid.processing.model.banner.GdImageAd;
import ru.yandex.direct.grid.processing.model.banner.GdInternalAd;
import ru.yandex.direct.grid.processing.model.banner.GdInternalAdImageResource;
import ru.yandex.direct.grid.processing.model.banner.GdInternalModerationInfo;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAd;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAds;
import ru.yandex.direct.grid.processing.model.banner.GdMcBannerAd;
import ru.yandex.direct.grid.processing.model.banner.GdMobileContentAd;
import ru.yandex.direct.grid.processing.model.banner.GdMobileContentAdAction;
import ru.yandex.direct.grid.processing.model.banner.GdMobileContentAdFeature;
import ru.yandex.direct.grid.processing.model.banner.GdMulticard;
import ru.yandex.direct.grid.processing.model.banner.GdMulticardStatus;
import ru.yandex.direct.grid.processing.model.banner.GdPlacementPageModerationResult;
import ru.yandex.direct.grid.processing.model.banner.GdSitelink;
import ru.yandex.direct.grid.processing.model.banner.GdSitelinkTurbolanding;
import ru.yandex.direct.grid.processing.model.banner.GdSmartAd;
import ru.yandex.direct.grid.processing.model.banner.GdTemplateVariable;
import ru.yandex.direct.grid.processing.model.banner.GdTextAd;
import ru.yandex.direct.grid.processing.model.banner.GdTurboApp;
import ru.yandex.direct.grid.processing.model.cliententity.GdAdCreativeType;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdPixel;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdAdImageAvatarsHost;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdAdImageNamespace;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdAdImageType;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImage;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageSize;
import ru.yandex.direct.grid.processing.model.constants.GdButtonAction;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.service.banner.container.BannersCacheFilterData;
import ru.yandex.direct.grid.processing.service.banner.container.BannersCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter;
import ru.yandex.direct.grid.processing.util.StatHelper;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static java.math.RoundingMode.HALF_UP;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Map.entry;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.banner.service.validation.ModerateBannerValidationService.remoderationAllowed;
import static ru.yandex.direct.core.entity.banner.type.pixels.PixelProvider.YANDEXAUDIENCE;
import static ru.yandex.direct.core.entity.creative.model.CreativeType.BANNERSTORAGE;
import static ru.yandex.direct.core.entity.creative.model.CreativeType.CPM_OVERLAY;
import static ru.yandex.direct.core.entity.creative.model.CreativeType.CPM_VIDEO_CREATIVE;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.VIDEO_TYPES;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.grid.core.entity.banner.service.GridBannerStatusUtils.extractBannerStatus;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CONTENT_PROMOTION;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_GEOPRODUCT;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_PRICE;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_PRICE_AUDIO;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_PRICE_BANNER;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_PRICE_FRONTPAGE_VIDEO;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_PRICE_VIDEO;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_VIDEO;
import static ru.yandex.direct.grid.processing.model.banner.GdAdType.CPM_YNDX_FRONTPAGE;
import static ru.yandex.direct.grid.processing.model.cliententity.GdPixelKind.AUDIENCE;
import static ru.yandex.direct.grid.processing.model.cliententity.GdPixelKind.AUDIT;
import static ru.yandex.direct.grid.processing.model.contentpromotion.GdContentPromotionType.fromSource;
import static ru.yandex.direct.grid.processing.service.banner.GridBannerAggregationFieldsUtils.isBannerAimingAllowed;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toGdCreativeImplementation;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toSupportedGdImageFormats;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalGoalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.internalStatsToOuter;
import static ru.yandex.direct.grid.processing.util.StatHelper.recalcTotalStatsForUnitedGrid;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.Predicates.not;

@ParametersAreNonnullByDefault
public class BannerDataConverter {
    private static final Logger logger = LoggerFactory.getLogger(BannerDataConverter.class);

    private static final Set<GdAdType> CONTENT_PROMOTION_AD_TYPES = Set.of(GdAdType.CONTENT_PROMOTION_VIDEO,
            GdAdType.CONTENT_PROMOTION_COLLECTION);
    private static final BiMap<BannersBannerType, GdAdType> TO_GD_AD_TYPE_MAP = ImmutableBiMap.<BannersBannerType,
                    GdAdType>builder()
            .put(BannersBannerType.text, GdAdType.TEXT)
            .put(BannersBannerType.dynamic, GdAdType.DYNAMIC)
            .put(BannersBannerType.mobile_content, GdAdType.MOBILE_CONTENT)
            .put(BannersBannerType.performance, GdAdType.PERFORMANCE)
            .put(BannersBannerType.image_ad, GdAdType.IMAGE_AD)
            .put(BannersBannerType.mcbanner, GdAdType.MCBANNER)
            .put(BannersBannerType.cpm_outdoor, GdAdType.CPM_OUTDOOR)
            .put(BannersBannerType.cpm_banner, GdAdType.CPM_BANNER)
            .put(BannersBannerType.cpm_indoor, GdAdType.CPM_INDOOR)
            .put(BannersBannerType.cpm_audio, GdAdType.CPM_AUDIO)
            .put(BannersBannerType.cpm_geo_pin, GdAdType.CPM_GEO_PIN)
            .put(BannersBannerType.cpc_video, GdAdType.CPC_VIDEO)
            .put(BannersBannerType.internal, GdAdType.INTERNAL)
            .put(BannersBannerType.content_promotion, GdAdType.CONTENT_PROMOTION)
            .build();
    private static final BiMap<GdAdType, BannersBannerType> GD_AD_TYPE_TO_INTERNAL_TYPE_MAP =
            TO_GD_AD_TYPE_MAP.inverse();

    private static final Map<GdAdType, Set<AdGroupType>> GD_AD_TYPE_TO_AD_GROUP_TYPE_MAP = Map.ofEntries(
            entry(GdAdType.TEXT, Set.of(AdGroupType.BASE)),
            entry(GdAdType.IMAGE_AD, Set.of(AdGroupType.BASE, AdGroupType.MOBILE_CONTENT)),
            entry(GdAdType.CPC_VIDEO, Set.of(AdGroupType.BASE, AdGroupType.MOBILE_CONTENT)),
            entry(GdAdType.DYNAMIC, Set.of(AdGroupType.DYNAMIC)),
            entry(GdAdType.MOBILE_CONTENT, Set.of(AdGroupType.MOBILE_CONTENT)),
            entry(GdAdType.PERFORMANCE, Set.of(AdGroupType.PERFORMANCE)),
            entry(GdAdType.MCBANNER, Set.of(AdGroupType.MCBANNER)),
            entry(GdAdType.INTERNAL, Set.of(AdGroupType.INTERNAL)),
            entry(GdAdType.CONTENT_PROMOTION, Set.of(AdGroupType.CONTENT_PROMOTION)),
            entry(GdAdType.CONTENT_PROMOTION_VIDEO, Set.of(AdGroupType.CONTENT_PROMOTION_VIDEO)),
            entry(GdAdType.CONTENT_PROMOTION_SERVICE, Set.of(AdGroupType.CONTENT_PROMOTION)),
            entry(GdAdType.CONTENT_PROMOTION_COLLECTION, Set.of(AdGroupType.CONTENT_PROMOTION)),
            entry(GdAdType.CPM_AUDIO, Set.of(AdGroupType.CPM_AUDIO)),
            entry(GdAdType.CPM_BANNER, Set.of(AdGroupType.CPM_BANNER)),
            entry(GdAdType.CPM_DEALS, Set.of(AdGroupType.CPM_BANNER)),
            entry(GdAdType.CPM_GEO_PIN, Set.of(AdGroupType.CPM_GEO_PIN)),
            entry(GdAdType.CPM_GEOPRODUCT, Set.of(AdGroupType.CPM_GEOPRODUCT)),
            entry(GdAdType.CPM_INDOOR, Set.of(AdGroupType.CPM_INDOOR)),
            entry(GdAdType.CPM_OUTDOOR, Set.of(AdGroupType.CPM_OUTDOOR)),
            entry(GdAdType.CPM_VIDEO, Set.of(AdGroupType.CPM_VIDEO)),
            entry(GdAdType.CPM_YNDX_FRONTPAGE, Set.of(AdGroupType.CPM_YNDX_FRONTPAGE))
    );

    private static final Set<GdAdType> CPM_BANNER_TYPES = Set.of(
            GdAdType.CPM_AUDIO,
            GdAdType.CPM_BANNER,
            GdAdType.CPM_DEALS,
            GdAdType.CPM_GEOPRODUCT,
            GdAdType.CPM_INDOOR,
            GdAdType.CPM_OUTDOOR,
            GdAdType.CPM_YNDX_FRONTPAGE,
            GdAdType.CPM_VIDEO
    );

    private static final BiMap<GdiBannerCreativeType, GdAdCreativeType> TO_GD_AD_CREATIVE_TYPE_MAP =
            ImmutableBiMap.<GdiBannerCreativeType, GdAdCreativeType>builder()
                    .put(GdiBannerCreativeType.BANNERSTORAGE, GdAdCreativeType.BANNERSTORAGE)
                    .put(GdiBannerCreativeType.PERFORMANCE, GdAdCreativeType.PERFORMANCE)
                    .put(GdiBannerCreativeType.CANVAS, GdAdCreativeType.CANVAS)
                    .put(GdiBannerCreativeType.HTML5_CREATIVE, GdAdCreativeType.HTML5_CREATIVE)
                    .put(GdiBannerCreativeType.VIDEO_ADDITION, GdAdCreativeType.VIDEO_ADDITION)
                    .build();

    private static final Map<StatusModerate, GdiBannerStatusModerate> TO_GDI_BANNER_STATUS_MODERATE =
            ImmutableMap.<StatusModerate, GdiBannerStatusModerate>builder()
                    .put(StatusModerate.NEW, GdiBannerStatusModerate.NEW)
                    .put(StatusModerate.ERROR, GdiBannerStatusModerate.NEW)
                    .put(StatusModerate.NO, GdiBannerStatusModerate.NO)
                    .put(StatusModerate.READY, GdiBannerStatusModerate.READY)
                    .put(StatusModerate.SENDING, GdiBannerStatusModerate.SENDING)
                    .put(StatusModerate.SENT, GdiBannerStatusModerate.SENT)
                    .put(StatusModerate.YES, GdiBannerStatusModerate.YES)
                    .build();

    static GdAdType getGdAdType(GdiBanner banner, GdAdGroupTruncated adGroup) {
        if (isCpmVideo(banner)) {
            if (adGroup.getType() == GdAdGroupType.CPM_PRICE_VIDEO) {
                return CPM_PRICE_VIDEO;
            }
            if (adGroup.getType() == GdAdGroupType.CPM_PRICE_FRONTPAGE_VIDEO) {
                return CPM_PRICE_FRONTPAGE_VIDEO;
            }
            return CPM_VIDEO;
        }
        if (adGroup.getType() == GdAdGroupType.CPM_PRICE_AUDIO) {
            return CPM_PRICE_AUDIO;
        }
        if (adGroup.getType() == GdAdGroupType.CPM_PRICE_BANNER) {
            return CPM_PRICE_BANNER;
        }
        if (adGroup.getType() == GdAdGroupType.CPM_GEOPRODUCT) {
            return CPM_GEOPRODUCT;
        }
        if (adGroup.getType() == GdAdGroupType.CPM_PRICE) {
            return CPM_PRICE;
        }
        if (adGroup.getType() == GdAdGroupType.CPM_YNDX_FRONTPAGE) {
            return CPM_YNDX_FRONTPAGE;
        }
        return toGdAdType(banner.getBannerType());
    }

    public static GdAdType toGdAdType(BannersBannerType bannerType) {
        return TO_GD_AD_TYPE_MAP.get(bannerType);
    }

    public static BannersBannerType toBannerType(GdAdType adType) {
        if (TO_GD_AD_TYPE_MAP.containsValue(adType)) {
            return GD_AD_TYPE_TO_INTERNAL_TYPE_MAP.get(adType);
        }
        if (CPM_BANNER_TYPES.contains(adType)) {
            return BannersBannerType.cpm_banner;
        }
        if (CONTENT_PROMOTION_AD_TYPES.contains(adType)) {
            return BannersBannerType.cpm_banner;
        }
        throw new IllegalArgumentException("Unsupported ad type " + adType.name());
    }

    public static Set<AdGroupType> getAdGroupType(GdAdType adType) {
        if (GD_AD_TYPE_TO_AD_GROUP_TYPE_MAP.containsKey(adType)) {
            return GD_AD_TYPE_TO_AD_GROUP_TYPE_MAP.get(adType);
        }
        throw new IllegalArgumentException("Unsupported ad type " + adType.name());
    }

    public static BannersBannerType toInternalAdType(@Nullable GdAdType gdAdType) {
        List<GdAdType> cpmBannerTypes = List.of(
                CPM_VIDEO,
                CPM_GEOPRODUCT,
                CPM_PRICE,
                CPM_YNDX_FRONTPAGE,
                CPM_PRICE_VIDEO,
                CPM_PRICE_FRONTPAGE_VIDEO,
                CPM_PRICE_BANNER);
        if (cpmBannerTypes.contains(gdAdType)) {
            return BannersBannerType.cpm_banner;
        }
        if (CONTENT_PROMOTION_AD_TYPES.contains(gdAdType)) {
            return BannersBannerType.content_promotion;
        }
        if (CPM_PRICE_AUDIO == gdAdType) {
            return BannersBannerType.cpm_audio;
        }

        return GD_AD_TYPE_TO_INTERNAL_TYPE_MAP.get(gdAdType);
    }

    private static GdAdCreativeType toGdAdCreativeType(GdiBannerCreativeType gdiBannerCreativeType) {
        return TO_GD_AD_CREATIVE_TYPE_MAP.get(gdiBannerCreativeType);
    }

    @Nullable
    static GdAd toGdAdImplementation(int index, GridGraphQLContext context, GdAdGroupTruncated group,
                                     Set<String> enabledFeatures,
                                     Set<Long> allowedReasonsToSelfRemoderate, GdiBanner internal) {
        GdiBannerStatus status = extractBannerStatus(internal);
        GdAdStatus statusInformation = toGdAdStatus(status);

        boolean cpcAndCpmOnOneGridEnabled = enabledFeatures.contains(CPC_AND_CPM_ON_ONE_GRID_ENABLED.getName());

        GdAd typedAd;
        switch (internal.getBannerType()) {
            case text:
                typedAd = new GdTextAd()
                        .withBannerPrice(toGdBannerPrice(internal.getBannerPrice()))
                        .withTurboGalleryHref(internal.getTurboGalleryHref())
                        .withPermalinkId(internal.getPermalinkId())
                        .withPhoneId(internal.getPhoneId())
                        .withPermalinkAssignType(toGdAdPermalinkAssignType(internal.getPermalinkAssignType()))
                        .withTurboApp(toGdTurboApp(internal.getTurboApp()))
                        .withModFlags(toGdFlags(internal.getModFlags()))
                        .withMulticards(toGdMulticards(internal.getMulticards(),
                                internal.getMulticardSetStatusModerate()))
                        .withPreferVCardOverPermalink(internal.getPreferVCardOverPermalink())
                        .withShowTitleAndBody(internal.getShowTitleAndBody());
                break;
            case image_ad:
                typedAd = new GdImageAd()
                        .withPermalinkId(internal.getPermalinkId());
                break;
            case dynamic:
                typedAd = new GdDynamicAd();
                break;
            case mobile_content:
                GdFlags modFlags = nvl(toGdFlags(internal.getModFlags()), new GdFlags());
                modFlags.withAge(nvl(modFlags.getAge(), GdAgeValue.AGE_18));
                typedAd = new GdMobileContentAd()
                        .withModFlags(modFlags)
                        .withPrimaryAction(toGdMobileContentAdAction(internal.getPrimaryAction()))
                        .withReflectedAttrs(toGdMobileContentAdFeature(internal.getReflectedAttrs()))
                        .withImpressionUrl(internal.getImpressionUrl());
                break;
            case performance:
                typedAd = new GdSmartAd()
                        .withTurboApp(toGdTurboApp(internal.getTurboApp()));
                break;
            case mcbanner:
                typedAd = new GdMcBannerAd();
                break;
            case cpm_outdoor:
                typedAd = new GdCPMOutdoorBannerAd()
                        .withPlacementPages(mapList(nvl(internal.getPlacementPages(), emptyList()),
                                BannerDataConverter::toGdPlacementPage))
                        .withPixels(mapList(nvl(internal.getPixels(), emptyList()), BannerDataConverter::toGdPixel));
                break;
            case cpm_banner:
                typedAd = new GdCPMBannerAd()
                        .withPixels(mapList(nvl(internal.getPixels(), emptyList()), BannerDataConverter::toGdPixel))
                        .withTnsId(internal.getTnsId())
                        .withAdditionalHrefs(toGdBannerAdditionalHrefs(internal.getAdditionalHrefs()))
                        .withMulticards(toGdMulticards(internal.getMulticards(),
                                internal.getMulticardSetStatusModerate()))
                        .withBigKingImage(toGdBannerImage(internal.getBigKingImage()));
                break;
            case cpm_indoor:
                typedAd = new GdCPMBannerAd()
                        .withPixels(mapList(nvl(internal.getPixels(), emptyList()), BannerDataConverter::toGdPixel))
                        .withTnsId(null)
                        .withAdditionalHrefs(toGdBannerAdditionalHrefs(internal.getAdditionalHrefs()));
                break;
            case cpc_video:
                typedAd = new GdCPCVideoAd();
                break;
            case cpm_audio:
                typedAd = new GdCPMAudioBannerAd()
                        .withPixels(mapList(nvl(internal.getPixels(), emptyList()), BannerDataConverter::toGdPixel));
                break;
            case cpm_geo_pin:
                typedAd = new GdCPMGeoPinBannerAd()
                        .withPixels(mapList(nvl(internal.getPixels(), emptyList()), BannerDataConverter::toGdPixel))
                        .withTnsId(internal.getTnsId())
                        .withPermalinkId(internal.getPermalinkId());
                break;
            case internal:
                GdiInternalBannerExtraInfo internalAdExtraInfo = checkNotNull(internal.getInternalAdExtraInfo());

                // В баннерах внутренней рекламы может быть несколько картинок, их хэши хранятся
                // среди переменных GdInternalAd.templateVariables
                // При отображении формы редактирования или просмотра баннера, фронтенду нужен не только хэш,
                // а дополнительная информация о картинке, чтобы собрать урл картинки и компонент загрузки/управления
                // изображением.
                // Чтобы передать фронту эту информацию, мы отдельно собираем картиночные переменные
                // и передаём их в виде списка GdInternalAd.images, с нужной фронтенду информацией о картинке.
                // Связывать переменные с картинками следует по ID ресурса GdInternalAdImageResource.templateResourceId
                List<GdInternalAdImageResource> imageResources = mapList(internalAdExtraInfo.getImages(),
                        BannerDataConverter::toGdInternalAdImageResource);

                // Первую картинку из переменных будем считать основной, и запишем её в GdInternalAd.image,
                // откуда фронтенд берёт картинки для отображения в гридах.
                GdImage mainImageOrNull = StreamEx.of(imageResources)
                        .map(GdInternalAdImageResource::getImage)
                        .findFirst()
                        .orElse(null);

                typedAd = new GdInternalAd()
                        .withDescription(internalAdExtraInfo.getDescription())
                        .withTemplateId(internalAdExtraInfo.getTemplateId())
                        .withImage(mainImageOrNull)
                        .withImages(imageResources)
                        .withTemplateVars(
                                mapList(internalAdExtraInfo.getTemplateVariables(),
                                        BannerDataConverter::toGdTemplateVariable))
                        .withModerationInfo(toGdInternalModerationInfo(internal));
                break;
            case content_promotion:
                GdiBannerContentPromotion bannerContentPromotion = checkNotNull(internal.getBannerContentPromotion());
                ContentPromotionContent contentPromotion = checkNotNull(internal.getContentPromotion());

                GdAdType gdAdType = GdAdType.valueOf(format("%s_%s", CONTENT_PROMOTION.name(),
                        contentPromotion.getType().name()));

                typedAd = new GdContentPromotionAd()
                        .withType(gdAdType)
                        .withContentPromotionId(contentPromotion.getId())
                        .withContentPromotionType(fromSource(contentPromotion.getType()))
                        .withContentPromotionPreviewUrl(contentPromotion.getPreviewUrl())
                        .withContentPromotionUrl(contentPromotion.getUrl())
                        .withContentPromotionVisitUrl(bannerContentPromotion.getVisitUrl());
                break;
            default:
                logger.error("Unsupported ad type: {}", internal.getBannerType());
                return null;
        }

        return typedAd.withIndex(index)
                .withId(internal.getId())
                .withExportId(internal.getBsBannerId())
                .withAdGroupId(internal.getGroupId())
                .withAdGroup(group)
                .withCampaignId(internal.getCampaignId())
                // Бывает удобно заполнять тип прямо в свиче выше при работе с синтетическими типами баннеров.
                // В этом случае затирать тип не нужно.
                .withType(nvl(typedAd.getType(), getGdAdType(internal, group)))
                .withIsMobile(internal.getType() == GdiBannerShowType.MOBILE)
                .withTitle(internal.getTitle())
                .withTitleExtension(internal.getTitleExtension())
                .withBody(internal.getBody())
                .withHref(internal.getHref())
                .withDomainId(internal.getDomainId())

                .withLastChange(internal.getLastChange())
                .withDomain(internal.getDomain())
                .withReverseDomain(internal.getReverseDomain())
                .withPhoneFlag(getPhoneFlag(internal))
                .withVcardId(internal.getVcardId())
                .withFlags(internal.getFlags())
                .withSitelinksSetId(internal.getSitelinksSetId())
                .withOptsGeoFlag(internal.getOptsGeoFlag())
                .withOptsNoDisplayHref(internal.getOptsNoDisplayHref())

                .withAccess(toGdAdAccess(context.getOperator(), context.getSubjectUser(), internal, group,
                        enabledFeatures, allowedReasonsToSelfRemoderate))
                .withStats(internalStatsToOuter(internal.getStat(), group.getCampaign().getType(), cpcAndCpmOnOneGridEnabled))
                .withStatsByDays(mapList(internal.getStatsByDays(),
                        s -> internalStatsToOuter(s, group.getCampaign().getType(), cpcAndCpmOnOneGridEnabled)))
                .withGoalStats(mapList(internal.getGoalStats(), StatHelper::internalGoalStatToOuter))
                // для GdInternalAd картинка заполняется выше, при создании экземпляра GdInternalAd
                .withImage(typedAd.getImage() == null ? toGdBannerImage(internal.getImage()) : typedAd.getImage())
                .withBannerImage(toGdBannerImage(internal.getBannerImage()))
                .withLogoImage(toGdBannerImage(internal.getBannerLogo()))
                .withButton(toGdBannerButton(internal.getButton()))
                .withCreative(toGdBannerCreative(internal.getCreative()))
                .withTypedCreative(toGdCreativeImplementation(internal.getTypedCreative()))
                .withCallouts(mapList(internal.getCallouts(), ClientDataConverter::toGdCallout))
                .withTurbolanding(toGdBannerTurbolanding(internal.getTurbolanding(), internal.getTurbolandingParams()))
                .withCallouts(mapList(internal.getCallouts(), ClientDataConverter::toGdCallout))
                .withDynamicDisclaimer(internal.getDynamicDisclaimer())
                .withLinkTail(internal.getLinkTail())
                .withSitelinks(mapList(internal.getSitelinks(), BannerDataConverter::toGdSitelink))
                .withStatus(statusInformation)
                .withAggregatedStatus(internal.getAggregatedStatus())
                .withAggregatedStatusInfo(internal.getAggregatedStatusInfo())
                .withIsAimingAllowed(isBannerAimingAllowed(internal, group))
                .withMeasurers(mapList(nvl(internal.getMeasurers(), emptyList()),
                        BannerDataConverter::toGdBannerMeasurer))
                .withName(internal.getName())
                .withHasCallouts(isNotEmpty(internal.getCallouts()))
                .withHasSitelink(isNotEmpty(internal.getSitelinks()))
                .withHasTurboGallery(StringUtils.isNotEmpty(internal.getTurboGalleryHref()))
                .withHasTurbolanding(internal.getTurbolanding() != null)
                .withHasVideo(hasVideo(internal))
                .withHasLogo(internal.getBannerLogo() != null)
                .withHasButton(internal.getButton() != null);
    }

    @Nullable
    private static List<GdMulticard> toGdMulticards(List<BannerMulticard> multicards,
                                                    BannerMulticardSetStatusModerate statusModerate) {
        return mapList(multicards, multicard -> new GdMulticard()
                .withId(multicard.getMulticardId())
                .withText(multicard.getText())
                .withImageHash(multicard.getImageHash())
                .withHref(multicard.getHref())
                .withMulticardSetStatus(toGdMulticardStatus(statusModerate)));
    }

    private static GdMulticardStatus toGdMulticardStatus(BannerMulticardSetStatusModerate statusModerate) {
        if (statusModerate == BannerMulticardSetStatusModerate.NEW) {
            return GdMulticardStatus.DRAFT;
        } else if (statusModerate == BannerMulticardSetStatusModerate.NO) {
            return GdMulticardStatus.REJECTED;
        } else if (statusModerate == BannerMulticardSetStatusModerate.YES) {
            return GdMulticardStatus.ACCEPTED;
        }
        return GdMulticardStatus.ON_MODERATION;
    }

    private static GdTurboApp toGdTurboApp(@Nullable TurboAppMetaContent metaContent) {
        if (metaContent == null ||
                metaContent.getName() == null ||
                metaContent.getDescription() == null ||
                metaContent.getIconUrl() == null) {
            return null;
        }

        return new GdTurboApp()
                .withName(metaContent.getName())
                .withDescription(metaContent.getDescription())
                .withIconUrl(metaContent.getIconUrl());
    }

    @Nullable
    private static Boolean getPhoneFlag(GdiBanner internal) {
        return ifNotNull(internal.getVcardId(), v -> internal.getPhoneFlag() == GdiBannerStatusVCardModeration.YES);
    }

    private static boolean isCpmVideo(GdiBanner internal) {
        return internal.getTypedCreative() != null &&
                (internal.getTypedCreative().getType() == CPM_VIDEO_CREATIVE ||
                        internal.getTypedCreative().getType() == BANNERSTORAGE ||
                        internal.getTypedCreative().getType() == CPM_OVERLAY);
    }

    private static boolean hasVideo(GdiBanner internal) {

        return internal.getTypedCreative() != null
                && VIDEO_TYPES.contains(internal.getTypedCreative().getType());
    }

    /**
     * На {@link GdAdAccess} заполняем параметры, которые можем вычислить в текущем контексте.
     * И проставляем ID для вычисления дополнительных параметров через DataLoader'ы
     *
     * @see ru.yandex.direct.grid.processing.service.banner.loader.CanBeDeletedAdsDataLoader
     */
    static GdAdAccess toGdAdAccess(User operator, User subjectClient, GdiBanner internal, GdAdGroupTruncated group,
                                   Set<String> enabledFeatures, Set<Long> allowedReasonsToSelfRemoderate) {
        boolean allowEdit = group.getAccess().getCanEdit() && !internal.getStatusArchived();
        boolean canBeSelfRemoderated = allowEdit && canBeSelfRemoderated(operator, subjectClient, enabledFeatures,
                allowedReasonsToSelfRemoderate, internal, group);

        return new GdAdAccess()
                .withAdId(internal.getId())
                .withCanEditOrganizationPhone(internal.getCanEditOrganizationPhone())
                .withCanBeSelfRemoderated(canBeSelfRemoderated)
                .withCanEdit(allowEdit)
                .withCanEditButtons(BannersBannerType.text == internal.getBannerType());
    }

    private static GdTemplateVariable toGdTemplateVariable(TemplateVariable templateVariable) {
        return new GdTemplateVariable()
                .withTemplateResourceId(templateVariable.getTemplateResourceId())
                .withValue(templateVariable.getInternalValue());
    }


    private static boolean canBeSelfRemoderated(User operator, User subjectClient, Set<String> enabledFeatures,
                                                Set<Long> allowedReasonsToSelfRemoderate, GdiBanner internal,
                                                GdAdGroupTruncated group) {
        var isFeatureEnabled = enabledFeatures.contains(FeatureName.CLIENT_ALLOWED_TO_REMODERATE.getName());
        var remoderationAllowed = remoderationAllowed(operator.getRole(), subjectClient.getRole(),
                isFeatureEnabled);
        var mightHaveRejectReasons = internal.getAggregatedStatusInfo() == null ||
                internal.getAggregatedStatusInfo().getMightHaveRejectReasons() == null ||
                internal.getAggregatedStatusInfo().getMightHaveRejectReasons();
        return remoderationAllowed
                && mightHaveRejectReasons
                && !internal.getStatusArchived()
                && internal.getStatusModerate() != GdiBannerStatusModerate.NEW
                && group.getCampaign().getStatus().getPrimaryStatus() != GdCampaignPrimaryStatus.ARCHIVED
                && group.getCampaign().getStatus().getPrimaryStatus() != GdCampaignPrimaryStatus.DRAFT
                && ModerationReasonService.allReasonsAllowedSelfRemoderate(allowedReasonsToSelfRemoderate)
                .test(nvl(internal.getModerationReasonIds(), emptySet()));
    }

    @Nullable
    private static GdInternalModerationInfo toGdInternalModerationInfo(GdiBanner gdiBanner) {
        InternalModerationInfo moderationInfo = gdiBanner.getInternalAdExtraInfo().getModerationInfo();
        if (moderationInfo == null) {
            return null;
        }

        return new GdInternalModerationInfo()
                .withTicketUrl(moderationInfo.getTicketUrl())
                .withCustomComment(moderationInfo.getCustomComment())
                .withIsSecretAd(moderationInfo.getIsSecretAd())
                .withStatusShowAfterModeration(
                        nvl(moderationInfo.getStatusShowAfterModeration(), gdiBanner.getStatusShow()))
                .withSendToModeration(nvl(moderationInfo.getSendToModeration(), false))
                .withStatusShow(gdiBanner.getStatusShow());
    }

    @Nullable
    public static InternalModerationInfo toCoreInternalModerationInfo(
            @Nullable GdInternalModerationInfo moderationInfo) {
        if (moderationInfo == null) {
            return null;
        }

        return new InternalModerationInfo()
                .withTicketUrl(moderationInfo.getTicketUrl())
                .withCustomComment(moderationInfo.getCustomComment())
                .withIsSecretAd(moderationInfo.getIsSecretAd())
                .withSendToModeration(moderationInfo.getSendToModeration())
                .withStatusShowAfterModeration(moderationInfo.getStatusShowAfterModeration());
    }

    @Nullable
    static GdAdPrice toGdBannerPrice(@Nullable BannerPrice bannerPrice) {
        return ifNotNull(bannerPrice, price -> new GdAdPrice()
                .withPrice(price.getPrice().setScale(2, HALF_UP).toString())
                .withPriceOld(ifNotNull(price.getPriceOld(), p -> p.setScale(2, HALF_UP).toString()))
                .withPrefix(ifNotNull(price.getPrefix(), p -> GdAdPricePrefix.valueOf(p.name())))
                .withCurrency(GdAdPriceCurrency.valueOf(price.getCurrency().name())));
    }

    @Nullable
    private static GdPlacementPageModerationResult toGdPlacementPage(@Nullable ModerateBannerPage moderateBannerPage) {
        return ifNotNull(moderateBannerPage, page -> new GdPlacementPageModerationResult()
                .withBannerId(page.getBannerId())
                .withId(page.getId())
                .withBannerId(page.getBannerId())
                .withPageId(page.getPageId())
                .withVersion(page.getVersion())
                .withStatusModerate(page.getStatusModerate())
                .withStatusModerateOperator(page.getStatusModerateOperator())
                .withCreateTime(page.getCreateTime())
                .withComment(page.getComment()));
    }

    @Nullable
    public static GdImage toGdBannerImage(@Nullable GdiAbstractBannerImage imageData) {
        return ifNotNull(imageData, data -> new GdImage()
                .withImageHash(data.getImageHash())
                .withAvatarsHost(toGdAdImageAvatarsHost(data.getAvatarsHost()))
                .withImageSize(new GdImageSize()
                        .withWidth(data.getWidth().intValue())
                        .withHeight(data.getHeight().intValue()))
                .withFormats(toSupportedGdImageFormats(data.getImageType(),
                        BannerImageConverter.mergeImageMdsMeta(data.getMdsMeta(), data.getMdsMetaUserOverride())))
                .withMdsGroupId(data.getMdsGroupId())
                .withName(data.getName())
                .withNamespace(toGdAdImageNamespace(data.getNamespace()))
                .withType(toGdAdImageType(data.getImageType())));
    }

    @Nullable
    private static GdCreative toGdBannerCreative(@Nullable GdiBannerCreative creativeData) {
        return ifNotNull(creativeData, data -> new GdCreative()
                .withCreativeId(data.getCreativeId())
                .withPreviewUrl(data.getPreviewUrl())
                .withName(data.getName())
                .withCreativeType(toGdAdCreativeType(data.getCreativeType()))
                .withWidth(data.getWidth())
                .withHeight(data.getHeight())
                .withHasPackshot(data.getHasPackshot()));
    }

    @Nullable
    private static GdBannerButton toGdBannerButton(@Nullable GdiBannerButton button) {
        return ifNotNull(button, data -> new GdBannerButton()
                .withAction(GdButtonAction.fromSource(GdiButtonAction.toSource(data.getAction())))
                .withCustomText(data.getCustomText())
                .withHref(data.getHref()));
    }

    @Nullable
    private static GdSitelink toGdSitelink(@Nullable GdiSitelink sitelink) {
        return ifNotNull(sitelink, data -> new GdSitelink()
                .withTitle(data.getTitle())
                .withHref(data.getHref())
                .withDescription(data.getDescription())
                .withTurbolanding(toGdSitelinkTurbolanding(data.getTurbolanding())));
    }

    @Nullable
    public static GdSitelink toGdSitelink(@Nullable Sitelink sitelink, @Nullable TurboLanding turboLanding) {
        return ifNotNull(sitelink, data -> new GdSitelink()
                .withTitle(data.getTitle())
                .withHref(data.getHref())
                .withDescription(data.getDescription())
                .withTurbolanding(toGdSitelinkTurbolanding(turboLanding)));
    }

    private static GdAdStatus toGdAdStatus(GdiBannerStatus status) {
        return new GdAdStatus()
                .withPrimaryStatus(toGdAdPrimaryStatus(status.getPrimaryStatus()))
                .withHasInactiveResources(listToSet(status.getHasInactiveResources(), GdAdResources::fromSource))
                .withPreviousVersionShown(status.getPreviousVersionShown())
                .withPlacementPagesRequired(status.getPlacementPagesRequired())
                .withRejectedOnModeration(status.getRejectedOnModeration());
    }

    private static GdAdPrimaryStatus toGdAdPrimaryStatus(GdiBannerPrimaryStatus value) {
        return ifNotNull(value, data -> GdAdPrimaryStatus.valueOf(data.name()));
    }

    private static GdiBannerPrimaryStatus toGdiBannerPrimaryStatus(GdAdPrimaryStatus value) {
        return GdiBannerPrimaryStatus.valueOf(value.name());
    }

    private static GdAdImageAvatarsHost toGdAdImageAvatarsHost(GdiBannerImageAvatarsHost gdiBannerImageAvatarsHost) {
        return GdAdImageAvatarsHost.valueOf(gdiBannerImageAvatarsHost.name());
    }

    private static GdAdImageNamespace toGdAdImageNamespace(GdiBannerImageNamespace gdiBannerImageNamespace) {
        return GdAdImageNamespace.valueOf(gdiBannerImageNamespace.name());
    }

    static GdiBannerStatusModerate toGdiBannerStatusModerate(StatusModerate statusModerate) {
        return TO_GDI_BANNER_STATUS_MODERATE.get(statusModerate);
    }

    static List<GdiBannerOrderBy> toBannerOrderByYtFields(List<GdAdOrderBy> orderBy) {
        return StreamEx.of(orderBy)
                .filter(not(BannerOrderHelper::isOrderingOccursAfterDataFetching))
                .map(ob -> new GdiBannerOrderBy()
                        .withOrder(ob.getOrder())
                        .withField(GdAdOrderByField.toSource(ob.getField()))
                        .withGoalId(ob.getParams() == null ? null : ob.getParams().getGoalId())
                )
                .toList();
    }

    static GdiBannerFilter toInternalFilter(GdAdFilter filter) {
        return new GdiBannerFilter()
                .withBannerIdIn(filter.getAdIdIn())
                .withBannerIdNotIn(filter.getAdIdNotIn())
                .withBannerIdContainsAny(filter.getAdIdContainsAny())
                .withCampaignIdIn(filter.getCampaignIdIn())
                .withAdGroupIdIn(filter.getAdGroupIdIn())

                .withTypeIn(mapSet(filter.getTypeIn(), BannerDataConverter::toInternalAdType))
                .withInternalAdTemplateIdIn(filter.getInternalAdTemplateIdIn())
                .withPrimaryStatusContains(
                        mapSet(filter.getPrimaryStatusContains(), BannerDataConverter::toGdiBannerPrimaryStatus))
                .withStats(StatHelper.toInternalStatsFilter(filter.getStats()))
                .withGoalStats(mapList(filter.getGoalStats(), StatHelper::toInternalGoalStatsFilter))
                .withArchived(filter.getArchived())
                .withExportIdIn(filter.getExportIdIn())
                .withExportIdNotIn(filter.getExportIdNotIn())
                .withExportIdContainsAny(filter.getExportIdContainsAny())
                .withVcardExists(filter.getVcardExists())
                .withImageExists(filter.getImageExists())
                .withSitelinksExists(filter.getSitelinksExists())
                .withTurbolandingsExist(filter.getTurbolandingsExist())

                .withTitleIn(filter.getTitleIn())
                .withTitleNotIn(filter.getTitleNotIn())
                .withTitleContains(filter.getTitleContains())
                .withTitleOrBodyContains(filter.getTitleOrBodyContains())
                .withTitleNotContains(filter.getTitleNotContains())

                .withInternalAdTitleIn(filter.getInternalAdTitleIn())
                .withInternalAdTitleNotIn(filter.getInternalAdTitleNotIn())
                .withInternalAdTitleContains(filter.getInternalAdTitleContains())
                .withInternalAdTitleNotContains(filter.getInternalAdTitleNotContains())

                .withTitleExtensionIn(filter.getTitleExtensionIn())
                .withTitleExtensionNotIn(filter.getTitleExtensionNotIn())
                .withTitleExtensionContains(filter.getTitleExtensionContains())
                .withTitleExtensionNotContains(filter.getTitleExtensionNotContains())

                .withBodyIn(filter.getBodyIn())
                .withBodyNotIn(filter.getBodyNotIn())
                .withBodyContains(filter.getBodyContains())
                .withBodyNotContains(filter.getBodyNotContains())

                .withHrefIn(filter.getHrefIn())
                .withHrefNotIn(filter.getHrefNotIn())
                .withHrefContains(filter.getHrefContains())
                .withHrefNotContains(filter.getHrefNotContains())

                .withReasonsContainSome(filter.getReasonsContainSome());
    }

    static BannersCacheRecordInfo toBannersCacheRecordInfo(long clientId, GdAdsContainer input) {
        return new BannersCacheRecordInfo(clientId, input.getCacheKey(),
                new BannersCacheFilterData()
                        .withFilter(input.getFilter())
                        .withOrderBy(input.getOrderBy())
                        .withStatRequirements(input.getStatRequirements()));
    }

    static GdAdsContext toGdAdsContext(GdAdWithTotals adWithTotals,
                                       GdAdFilter inputFilter,
                                       boolean cpcAndCpmOnOneGridEnabled) {
        List<GdAd> rowsetFull = adWithTotals.getGdAds();
        var totalStats = nvl(adWithTotals.getTotalStats(), calcTotalStats(mapList(rowsetFull, GdAd::getStats)));

        if (cpcAndCpmOnOneGridEnabled) {
            Map<GdCampaignType, List<GdEntityStats>> campaignTypeToStats = StreamEx.of(rowsetFull)
                    .mapToEntry(gdAd -> gdAd.getAdGroup().getCampaign().getType(), GdAd::getStats)
                    .grouping();
            recalcTotalStatsForUnitedGrid(totalStats, campaignTypeToStats);
        }

        // показываем предупреждение если есть тоталы из БД и был фильтр по коду (либо мог быть)
        var totalStatsWithoutFiltersWarn = adWithTotals.getTotalStats() != null
                && (rowsetFull.size() < GridBannerConstants.getMaxBannerRows() || hasAnyCodeFilter(inputFilter));
        return new GdAdsContext()
                .withTotalCount(rowsetFull.size())
                .withHasObjectsOverLimit(rowsetFull.size() >= GridBannerConstants.getMaxBannerRows())
                .withAdIds(listToSet(rowsetFull, GdAd::getId))
                .withFeatures(toGdAdFeatures(rowsetFull))
                .withTotalStats(totalStats)
                .withTotalStatsWithoutFiltersWarn(totalStatsWithoutFiltersWarn)
                .withTotalGoalStats(calcTotalGoalStats(totalStats, mapList(rowsetFull, GdAd::getGoalStats)));
    }

    private static boolean hasAnyCodeFilter(GdAdFilter inputFilter) {
        return inputFilter.getArchived() != null
                || !isEmpty(inputFilter.getPrimaryStatusContains())
                || !isEmpty(inputFilter.getTypeIn())
                || inputFilter.getTurbolandingsExist() != null
                || nvl(inputFilter.getIsTouch(), false)
                || !isEmpty(inputFilter.getReasonsContainSome());
    }

    private static GdAdFeatures toGdAdFeatures(List<GdAd> rowsetFull) {
        long canBeEditedOrganizationPhoneBannersCount = StreamEx.of(rowsetFull)
                .filter(ad -> ad.getAccess() != null &&
                        ad.getAccess().getCanEditOrganizationPhone() != null &&
                        ad.getAccess().getCanEditOrganizationPhone())
                .count();

        long canBeEditedAgeModFlagsCount = StreamEx.of(rowsetFull)
                .filter(ad -> ad.getModFlags() != null && ad.getModFlags().getAge() != null)
                .count();

        long canBeEditedBabyFoodModFlagsCount = StreamEx.of(rowsetFull)
                .filter(ad -> ad.getModFlags() != null && ad.getModFlags().getBabyFood() != null)
                .count();

        long canBeSelfRemoderatedCount = StreamEx.of(rowsetFull)
                .filter(ad -> ad.getAccess() != null &&
                        ad.getAccess().getCanBeSelfRemoderated() != null &&
                        ad.getAccess().getCanBeSelfRemoderated())
                .count();

        return new GdAdFeatures()
                .withCanBeEditedOrganizationPhoneBannersCount((int) canBeEditedOrganizationPhoneBannersCount)
                .withCanBeEditedAgeModFlagsCount((int) canBeEditedAgeModFlagsCount)
                .withCanBeEditedBabyFoodModFlagsCount((int) canBeEditedBabyFoodModFlagsCount)
                .withCanBeSelfRemoderatedCount((int) canBeSelfRemoderatedCount);
    }

    @Nullable
    private static GdBannerTurbolanding toGdBannerTurbolanding(@Nullable GdiBannerTurboLanding turbolanding,
                                                               String turbolandingParams) {
        return ifNotNull(turbolanding,
                data -> new GdBannerTurbolanding()
                        .withId(data.getId())
                        .withClientId(data.getClientId())
                        .withName(data.getName())
                        .withHref(data.getUrl())
                        .withTurboSiteHref(data.getTurboSiteHref())
                        .withPreviewHref(data.getPreviewHref())
                        .withStatusModerate(toGdBannerTurbolandingStatusModerate(data.getStatusModerate()))
                        .withHrefParams(turbolandingParams)
        );
    }

    @Nullable
    private static GdSitelinkTurbolanding toGdSitelinkTurbolanding(@Nullable TurboLanding turboLanding) {
        return ifNotNull(turboLanding,
                data -> new GdSitelinkTurbolanding()
                        .withId(data.getId())
                        .withClientId(data.getClientId())
                        .withName(data.getName())
                        .withHref(data.getUrl())
                        .withTurboSiteHref(data.getTurboSiteHref())
                        .withPreviewHref(data.getPreviewHref())
        );
    }

    private static GdBannerTurbolandingStatusModerate toGdBannerTurbolandingStatusModerate(
            BannerTurboLandingStatusModerate bannerTurboLandingStatusModerate) {
        return GdBannerTurbolandingStatusModerate.valueOf(bannerTurboLandingStatusModerate.name());
    }

    private static GdPixel toGdPixel(String url) {
        return new GdPixel()
                .withUrl(url)
                .withKind(PixelProvider.fromUrl(url) == YANDEXAUDIENCE ? AUDIENCE : AUDIT);
    }

    private static GdBannerMeasurer toGdBannerMeasurer(BannerMeasurer bannerMeasurer) {
        return new GdBannerMeasurer()
                .withMeasurerSystem(toGdBannerMeasurerSystem(bannerMeasurer.getBannerMeasurerSystem()))
                .withParams(bannerMeasurer.getParams())
                .withHasIntegration(bannerMeasurer.getHasIntegration());
    }

    private static GdInternalAdImageResource toGdInternalAdImageResource(GdiInternalBannerImageResource imageResource) {
        return new GdInternalAdImageResource()
                .withTemplateResourceId(imageResource.getTemplateResourceId())
                .withImage(BannerDataConverter.toGdBannerImage(imageResource.getImage()));
    }

    private static GdAdImageType toGdAdImageType(ImageType imageType) {
        return GdAdImageType.valueOf(imageType.name());
    }

    private static GdBannerMeasurerSystem toGdBannerMeasurerSystem(BannerMeasurerSystem bannerMeasurerSystem) {
        switch (bannerMeasurerSystem) {
            case ADMETRICA:
                return GdBannerMeasurerSystem.ADMETRICA;
            case MEDIASCOPE:
                return GdBannerMeasurerSystem.MEDIASCOPE;
            case ADLOOX:
                return GdBannerMeasurerSystem.ADLOOX;
            case MOAT:
                return GdBannerMeasurerSystem.MOAT;
            case DV:
                return GdBannerMeasurerSystem.DV;
            case ADRIVER:
                return GdBannerMeasurerSystem.ADRIVER;
            case SIZMEK:
                return GdBannerMeasurerSystem.SIZMEK;
            case INTEGRAL_AD_SCIENCE:
                return GdBannerMeasurerSystem.INTEGRAL_AD_SCIENCE;
            case WEBORAMA:
                return GdBannerMeasurerSystem.WEBORAMA;
            case OMI:
                return GdBannerMeasurerSystem.OMI;
            default:
                return null;
        }
    }

    private static GdAdPermalinkAssignType toGdAdPermalinkAssignType(
            @Nullable PermalinkAssignType permalinkAssignType) {
        if (permalinkAssignType == null) {
            return null;
        }
        switch (permalinkAssignType) {
            case AUTO:
                return GdAdPermalinkAssignType.AUTO;
            case MANUAL:
                return GdAdPermalinkAssignType.MANUAL;
            default:
                throw new IllegalStateException("No such value: " + permalinkAssignType);
        }
    }

    static GdLastChangedAds toGdLastChangedAds(List<GdAd> lastChangedAds) {
        return new GdLastChangedAds()
                .withLastChangedAds(StreamEx.of(lastChangedAds)
                        .nonNull()
                        .map(ad -> new GdLastChangedAd()
                                .withCampaignId(ad.getCampaignId())
                                .withAdType(ad.getType())
                                .withAd(ad))
                        .toList());
    }

    private static List<GdBannerAdditionalHref> toGdBannerAdditionalHrefs(
            @Nullable List<BannerAdditionalHref> bannerAdditionalHrefs) {
        return mapList(nvl(bannerAdditionalHrefs, emptyList()),
                BannerDataConverter::toGdBannerAdditionalHref);
    }

    private static GdBannerAdditionalHref toGdBannerAdditionalHref(BannerAdditionalHref bannerAdditionalHref) {
        return new GdBannerAdditionalHref()
                .withHref(bannerAdditionalHref.getHref());
    }

    @Nullable
    static Optional<BabyFood> toBannerBabyFoodFlag(@Nullable Optional<GdBabyFoodValue> babyFoodValue) {
        if (babyFoodValue == null) {
            return null;
        } else if (babyFoodValue.isEmpty()) {
            return Optional.empty();
        }
        switch (babyFoodValue.get()) {
            case BABY_FOOD_0:
                return Optional.of(BabyFood.BABY_FOOD_0);
            case BABY_FOOD_1:
                return Optional.of(BabyFood.BABY_FOOD_1);
            case BABY_FOOD_2:
                return Optional.of(BabyFood.BABY_FOOD_2);
            case BABY_FOOD_3:
                return Optional.of(BabyFood.BABY_FOOD_3);
            case BABY_FOOD_4:
                return Optional.of(BabyFood.BABY_FOOD_4);
            case BABY_FOOD_5:
                return Optional.of(BabyFood.BABY_FOOD_5);
            case BABY_FOOD_6:
                return Optional.of(BabyFood.BABY_FOOD_6);
            case BABY_FOOD_7:
                return Optional.of(BabyFood.BABY_FOOD_7);
            case BABY_FOOD_8:
                return Optional.of(BabyFood.BABY_FOOD_8);
            case BABY_FOOD_9:
                return Optional.of(BabyFood.BABY_FOOD_9);
            case BABY_FOOD_10:
                return Optional.of(BabyFood.BABY_FOOD_10);
            case BABY_FOOD_11:
                return Optional.of(BabyFood.BABY_FOOD_11);
            case BABY_FOOD_12:
                return Optional.of(BabyFood.BABY_FOOD_12);
            default:
                throw new IllegalStateException("No such value: " + babyFoodValue);
        }
    }

    @Nullable
    static Optional<Age> toBannerAgeFlag(@Nullable Optional<GdAgeValue> ageValue) {
        if (ageValue == null) {
            return null;
        } else if (ageValue.isEmpty()) {
            return Optional.empty();
        }
        switch (ageValue.get()) {
            case AGE_0:
                return Optional.of(Age.AGE_0);
            case AGE_6:
                return Optional.of(Age.AGE_6);
            case AGE_12:
                return Optional.of(Age.AGE_12);
            case AGE_16:
                return Optional.of(Age.AGE_16);
            case AGE_18:
                return Optional.of(Age.AGE_18);
            default:
                throw new IllegalStateException("No such value: " + ageValue);
        }
    }

    @Nullable
    private static GdFlags toGdFlags(@Nullable BannerFlags modFlags) {
        if (modFlags == null) {
            return null;
        }
        GdFlags flags = new GdFlags()
                .withAbortion(modFlags.get(BannerFlags.ABORTION))
                .withMedicine(modFlags.get(BannerFlags.MEDICINE))
                .withMedServices(modFlags.get(BannerFlags.MED_SERVICES))
                .withMedEquipment(modFlags.get(BannerFlags.MED_EQUIPMENT))
                .withPharmacy(modFlags.get(BannerFlags.PHARMACY))
                .withAlcohol(modFlags.get(BannerFlags.ALCOHOL))
                .withTobacco(modFlags.get(BannerFlags.TOBACCO))
                .withPlus18(modFlags.get(BannerFlags.PLUS18))
                .withDietarysuppl(modFlags.get(BannerFlags.DIETARYSUPPL))
                .withProjectDeclaration(modFlags.get(BannerFlags.PROJECT_DECLARATION))
                .withTragic(modFlags.get(BannerFlags.TRAGIC))
                .withAsocial(modFlags.get(BannerFlags.ASOCIAL))
                .withPseudoweapon(modFlags.get(BannerFlags.PSEUDOWEAPON))
                .withForex(modFlags.get(BannerFlags.FOREX))
                .withEducation(modFlags.get(BannerFlags.EDUCATION))
                .withPeople(modFlags.get(BannerFlags.PEOPLE))
                .withNotAnimated(modFlags.get(BannerFlags.NOT_ANIMATED))
                .withGoodface(modFlags.get(BannerFlags.GOODFACE))
                .withAge(toGdAgeValue(modFlags.get(BannerFlags.AGE)))
                .withBabyFood(toGdBabyFoodValue(modFlags.get(BannerFlags.BABY_FOOD)));
        return flags.equals(new GdFlags()) ? null : flags;
    }

    @Nullable
    private static GdAgeValue toGdAgeValue(@Nullable Age age) {
        if (age != null) {
            switch (age) {
                case AGE_0:
                    return GdAgeValue.AGE_0;
                case AGE_6:
                    return GdAgeValue.AGE_6;
                case AGE_12:
                    return GdAgeValue.AGE_12;
                case AGE_16:
                    return GdAgeValue.AGE_16;
                case AGE_18:
                    return GdAgeValue.AGE_18;
            }
        }
        return null;
    }

    @Nullable
    private static GdBabyFoodValue toGdBabyFoodValue(@Nullable BabyFood babyFood) {
        if (babyFood != null) {
            switch (babyFood) {
                case BABY_FOOD_0:
                    return GdBabyFoodValue.BABY_FOOD_0;
                case BABY_FOOD_1:
                    return GdBabyFoodValue.BABY_FOOD_1;
                case BABY_FOOD_2:
                    return GdBabyFoodValue.BABY_FOOD_2;
                case BABY_FOOD_3:
                    return GdBabyFoodValue.BABY_FOOD_3;
                case BABY_FOOD_4:
                    return GdBabyFoodValue.BABY_FOOD_4;
                case BABY_FOOD_5:
                    return GdBabyFoodValue.BABY_FOOD_5;
                case BABY_FOOD_6:
                    return GdBabyFoodValue.BABY_FOOD_6;
                case BABY_FOOD_7:
                    return GdBabyFoodValue.BABY_FOOD_7;
                case BABY_FOOD_8:
                    return GdBabyFoodValue.BABY_FOOD_8;
                case BABY_FOOD_9:
                    return GdBabyFoodValue.BABY_FOOD_9;
                case BABY_FOOD_10:
                    return GdBabyFoodValue.BABY_FOOD_10;
                case BABY_FOOD_11:
                    return GdBabyFoodValue.BABY_FOOD_11;
                case BABY_FOOD_12:
                    return GdBabyFoodValue.BABY_FOOD_12;
            }
        }
        return null;
    }

    static GdMobileContentAdAction toGdMobileContentAdAction(NewMobileContentPrimaryAction primaryAction) {
        switch (primaryAction) {
            case DOWNLOAD:
                return GdMobileContentAdAction.DOWNLOAD;
            case GET:
                return GdMobileContentAdAction.GET;
            case INSTALL:
                return GdMobileContentAdAction.INSTALL;
            case MORE:
                return GdMobileContentAdAction.MORE;
            case OPEN:
                return GdMobileContentAdAction.OPEN;
            case UPDATE:
                return GdMobileContentAdAction.UPDATE;
            case PLAY:
                return GdMobileContentAdAction.PLAY;
            case BUY:
                return GdMobileContentAdAction.BUY_AUTODETECT;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + primaryAction);
        }
    }

    static Map<GdMobileContentAdFeature, Boolean> toGdMobileContentAdFeature(
            Set<NewReflectedAttribute> reflectedAttributes) {
        return StreamEx.of(GdMobileContentAdFeature.values())
                .mapToEntry(reflectedAttribute -> NewReflectedAttribute.valueOf(reflectedAttribute.name()))
                .mapValues(reflectedAttributes::contains)
                .toMap();
    }
}
