package ru.yandex.direct.grid.processing.service.banner;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerMulticard;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganizationAndPhone;
import ru.yandex.direct.core.entity.banner.model.BannerWithPrice;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperation;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode;
import ru.yandex.direct.core.entity.banner.type.flags.BannerWithFlagsConstraints;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.model.ImageMdsMeta;
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository;
import ru.yandex.direct.core.entity.image.service.ImageService;
import ru.yandex.direct.core.entity.image.service.SmartCenterUtils;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.banner.GdiAdMassChangeResult;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerFilter;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerOrderBy;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerPrimaryStatus;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannersWithTotals;
import ru.yandex.direct.grid.core.entity.banner.service.GridAdsMassChangesService;
import ru.yandex.direct.grid.core.entity.banner.service.GridBannerService;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.AdFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdAdAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.model.utils.GridModerationUtils;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.exception.GridValidationException;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.GdAd;
import ru.yandex.direct.grid.processing.model.banner.GdAdFilter;
import ru.yandex.direct.grid.processing.model.banner.GdAdOrderBy;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrice;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrimaryStatus;
import ru.yandex.direct.grid.processing.model.banner.GdAdType;
import ru.yandex.direct.grid.processing.model.banner.GdAdWithTotals;
import ru.yandex.direct.grid.processing.model.banner.GdAdsContainer;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAdContainer;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAdContainerItem;
import ru.yandex.direct.grid.processing.model.banner.GdLastChangedAds;
import ru.yandex.direct.grid.processing.model.banner.GdTurboGalleryParams;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdCreative;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdImage;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddContentPromotionAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddCpmAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddDynamicAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMcBannerAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMobileContentAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMulticardsToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSetPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSetToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSets;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSetsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSmartAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddTurboLandingToAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdAgeFlags;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdPrice;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdTurboGalleryHref;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateContentPromotionAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateCpmAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateDynamicAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMcBannerAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMobileContentAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateOrganization;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartCenters;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartCentersItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartCentersPayload;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendation;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKey;
import ru.yandex.direct.grid.processing.service.adgeneration.AdGenerationLoggingService;
import ru.yandex.direct.grid.processing.service.aggregatedstatuses.AdPrimaryStatusCalculator;
import ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter;
import ru.yandex.direct.grid.processing.service.banner.converter.AddAdMutationDataConverter;
import ru.yandex.direct.grid.processing.service.banner.converter.SitelinkConverter;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.grid.processing.service.group.GroupDataService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.util.GoalHelper;
import ru.yandex.direct.grid.processing.util.ReasonsFilterUtils;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.turbopages.client.TurbopagesClient;
import ru.yandex.direct.utils.Counter;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.emptyIfNull;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.common.db.PpcPropertyNames.MODERATION_REASONS_ALLOWABLE_TO_REMODERATE_BY_CLIENT;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.imageNotFound;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_FORMATS_BY_IMAGE_TYPE;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.feature.FeatureName.HIDE_OLD_SHOW_CAMPS_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.SHOW_AGGREGATED_STATUS_OPEN_BETA;
import static ru.yandex.direct.feature.FeatureName.SHOW_DNA_BY_DEFAULT;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.getAdGroupType;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toBannerAgeFlag;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toBannerBabyFoodFlag;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toBannerOrderByYtFields;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toBannerType;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toGdAdImplementation;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toGdLastChangedAds;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toGdiBannerStatusModerate;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toInternalAdType;
import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toInternalFilter;
import static ru.yandex.direct.grid.processing.service.banner.BannerOrderHelper.createOrderComparator;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.toCoreBannerPrice;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.toCoreBanners;
import static ru.yandex.direct.grid.processing.service.banner.converter.BannerUpdateConverter.bannersToCoreModelChanges;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toGdImage;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.VALIDATE_AD_PRICE_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getSuccessfullyResults;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Сервис, возвращающий данные о баннерах клиента
 */
@Service
@ParametersAreNonnullByDefault
public class BannerDataService {

    public static final Set<GdAdType> TYPES_WITH_TURBO_GALLERY_HREF = ImmutableSet.of(GdAdType.TEXT);

    private static final Map<GdiBannerStatusModerate, Integer> BANNER_STATUS_MERGE_PRIORITY =
            StreamEx.of(GdiBannerStatusModerate.NO,
                            GdiBannerStatusModerate.NEW, GdiBannerStatusModerate.READY, GdiBannerStatusModerate.SENDING,
                            GdiBannerStatusModerate.SENT, GdiBannerStatusModerate.YES, null)
                    .zipWith(IntStreamEx.range(Integer.MAX_VALUE, Integer.MIN_VALUE, -1))
                    .toMap();
    private static final int LATEST_CAMPAIGNS_LOOKUP_COUNT = 5;
    private static final Logger LOGGER = LoggerFactory.getLogger(BannerDataService.class);

    private final GridBannerService gridBannerService;
    private final GridRecommendationService gridRecommendationService;
    private final GridValidationService gridValidationService;
    private final AdValidationService adValidationService;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final BannerService newBannerService;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;
    private final CampaignInfoService campaignInfoService;
    private final GroupDataService groupDataService;
    private final SitelinkSetService sitelinkSetService;
    private final PathNodeConverterProvider adPricePathNodeConverterProvider;
    private final GridAdsMassChangesService gridAdsMassChangesService;
    private final FeatureService featureService;
    private final TurbopagesClient turbopagesClient;
    private final GridContextProvider gridContextProvider;
    private final OrganizationService organizationService;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannersAddOperationFactory bannersAddOperationFactory;
    private final OrganizationRepository organizationRepository;
    private final ShardHelper shardHelper;
    private final AdGenerationLoggingService adGenerationLoggingService;
    private final ImageDataRepository imageDataRepository;
    private final ImageService imageService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ModerationReasonService moderationReasonService;
    private final CampaignRepository campaignRepository;
    private final BannerCommonRepository bannerCommonRepository;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public BannerDataService(GridBannerService gridBannerService,
                             GridRecommendationService gridRecommendationService,
                             GridValidationService gridValidationService,
                             AdValidationService adValidationService,
                             BannerRelationsRepository bannerRelationsRepository,
                             BannerService newBannerService,
                             BannersUpdateOperationFactory bannersUpdateOperationFactory,
                             AggregatedStatusesViewService aggregatedStatusesViewService,
                             CampaignInfoService campaignInfoService,
                             GroupDataService groupDataService,
                             SitelinkSetService sitelinkSetService,
                             GridAdsMassChangesService gridAdsMassChangesService,
                             FeatureService featureService,
                             TurbopagesClient turbopagesClient,
                             GridContextProvider gridContextProvider,
                             OrganizationService organizationService,
                             BannerTypedRepository bannerTypedRepository,
                             BannersAddOperationFactory bannersAddOperationFactory,
                             OrganizationRepository organizationRepository,
                             ShardHelper shardHelper,
                             AdGenerationLoggingService adGenerationLoggingService,
                             ImageDataRepository imageDataRepository,
                             ImageService imageService,
                             PpcPropertiesSupport ppcPropertiesSupport,
                             ModerationReasonService moderationReasonService,
                             CampaignRepository campaignRepository,
                             BannerCommonRepository bannerCommonRepository) {

        this.gridBannerService = gridBannerService;
        this.gridRecommendationService = gridRecommendationService;
        this.gridValidationService = gridValidationService;
        this.adValidationService = adValidationService;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.newBannerService = newBannerService;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.campaignInfoService = campaignInfoService;
        this.groupDataService = groupDataService;
        this.sitelinkSetService = sitelinkSetService;
        this.gridAdsMassChangesService = gridAdsMassChangesService;
        this.featureService = featureService;
        this.turbopagesClient = turbopagesClient;
        this.gridContextProvider = gridContextProvider;
        this.organizationService = organizationService;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannersAddOperationFactory = bannersAddOperationFactory;
        this.organizationRepository = organizationRepository;
        this.shardHelper = shardHelper;
        this.adGenerationLoggingService = adGenerationLoggingService;
        this.imageDataRepository = imageDataRepository;
        this.imageService = imageService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.moderationReasonService = moderationReasonService;
        this.campaignRepository = campaignRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.adPricePathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, VALIDATE_AD_PRICE_PATH_CONVERTER)
                .build();
    }

    private static void updateFilterForRecommendations(GdiBannerFilter internalFilter,
                                                       Map<Long, List<GdRecommendation>> bannerIdToRecommendations) {
        //Корректируем фильтры для ускорения, все что возможно переносим в рекомендации
        List<GdRecommendationKey> filterRecommendations = flatMap(bannerIdToRecommendations.values(), identity())
                .stream()
                .flatMap(r -> r.getKeys().stream())
                .filter(key -> internalFilter.getAdGroupIdIn() == null
                        || internalFilter.getAdGroupIdIn().contains(key.getPid()))
                .filter(key -> internalFilter.getBannerIdIn() == null
                        || internalFilter.getBannerIdIn().contains(key.getBid()))
                .filter(key -> internalFilter.getBannerIdNotIn() == null
                        || !internalFilter.getBannerIdNotIn().contains(key.getBid()))
                .collect(toList());

        internalFilter.setRecommendations(filterRecommendations);
        internalFilter.setCampaignIdIn(null);
        internalFilter.setAdGroupIdIn(null);
        internalFilter.setBannerIdIn(null);
        internalFilter.setBannerIdNotIn(null);
    }

    private static StreamEx<GdAd> sortBanners(List<GdAdOrderBy> bannerOrderBy, StreamEx<GdAd> banners) {
        return isEmpty(bannerOrderBy) ? banners : banners.sorted(createOrderComparator(bannerOrderBy));
    }

    private static Set<Long> getCampaignsAllowedDomainMonitoring(List<GdAdGroupTruncated> groups) {
        return groups.stream()
                .map(GdAdGroupTruncated::getCampaign)
                .filter(c -> c.getStatus().getAllowDomainMonitoring())
                .map(GdCampaignTruncated::getId)
                .collect(toSet());
    }

    /**
     * Получить список баннеров клиента в формате, пригодном для возвращения оператору
     * <p>
     * При добавлении новых фильтров в коде нужно так же их учесть в методе:
     * {@link ru.yandex.direct.grid.processing.service.banner.BannerDataConverter#hasAnyCodeFilter}
     *
     * @param client  параметры клиента
     * @param input   входные параметры для получения баннеров
     * @param context контекст исполнения GraphQL запроса
     */
    public GdAdWithTotals getBanners(GdClientInfo client, GdAdsContainer input, GridGraphQLContext context) {
        ClientId clientId = ClientId.fromLong(client.getId());
        User operator = context.getOperator();
        List<GdiBannerOrderBy> bannerOrderByYtFields = toBannerOrderByYtFields(input.getOrderBy());
        GdAdFilter filter = input.getFilter();

        Set<GdiRecommendationType> recommendationTypes = nvl(filter.getRecommendations(), emptySet());
        GdiBannerFilter internalFilter = toInternalFilter(filter);
        Map<Long, List<GdRecommendation>> bannerIdToRecommendations = emptyMap();
        boolean isTouch = nvl(filter.getIsTouch(), false);

        if (isTouch) {
            updateFilterForTouch(internalFilter, client);
            // если тачевых кампаний нет, то и баннеров для тача нет
            if (internalFilter.getCampaignIdIn().isEmpty()) {
                return new GdAdWithTotals()
                        .withGdAds(emptyList());
            }
        }

        // При выборе баннеров стоит ограничение в GridBannerService.MAX_BANNER_ROWS и могут быть выбраны баннеры
        // без рекомендаций.
        // Поэтому сначала выбираем рекомендации, если они требуются в условии
        // и затем баннеры с фильтром на cid, pid, bid из рекомендаций
        if (!recommendationTypes.isEmpty()) {
            bannerIdToRecommendations
                    = getRecommendations(clientId, operator, recommendationTypes, internalFilter.getCampaignIdIn());

            updateFilterForRecommendations(internalFilter, bannerIdToRecommendations);
        }

        GdStatRequirements statRequirements = input.getStatRequirements();
        Set<Long> goalIds =
                GoalHelper.combineGoalIds(statRequirements.getGoalIds(), input.getFilter().getGoalStats());

        AdFetchedFieldsResolver adFetchedFieldsResolver = context.getFetchedFieldsReslover().getAd();

        boolean withFilterByAggrStatus = featureService.isEnabledForClientId(operator.getClientId(),
                SHOW_DNA_BY_DEFAULT)
                || featureService.isEnabledForClientId(operator.getClientId(), HIDE_OLD_SHOW_CAMPS_FOR_DNA)
                || featureService.isEnabledForClientId(operator.getClientId(), SHOW_AGGREGATED_STATUS_OPEN_BETA);

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        GdiBannersWithTotals bannersWithTotals = gridBannerService
                .getBanners(client.getShard(), operator.getUid(), clientId, enabledFeatures,
                        internalFilter, bannerOrderByYtFields,
                        statRequirements.getFrom(), statRequirements.getTo(),
                        statRequirements.getIsFlat(), goalIds,
                        statRequirements.getStatsByDaysFrom(), statRequirements.getStatsByDaysTo(),
                        adFetchedFieldsResolver, withFilterByAggrStatus);
        List<GdiBanner> banners = bannersWithTotals.getGdiBanners();
        GdEntityStats gdEntityStats = bannersWithTotals.getTotalStats() != null
                ? StatHelper.convertInternalStatsToOuter(bannersWithTotals.getTotalStats()) : null;

        // для тача пока подходят толькы баннеры с адаптивным канвасным креативом
        if (isTouch) {
            banners = filterList(banners, b -> b.getTypedCreative() != null && b.getTypedCreative().getIsAdaptive());
        }
        if (banners.isEmpty()) {
            return new GdAdWithTotals()
                    .withGdAds(emptyList())
                    .withTotalStats(gdEntityStats);
        }

        Map<Long, AggregatedStatusAdData> adStatusesByIds = aggregatedStatusesViewService
                .getAdStatusesByIds(client.getShard(), listToSet(banners, GdiBanner::getId));
        if (withFilterByAggrStatus) {
            banners = filterByAggregatedStatuses(adStatusesByIds, banners, internalFilter);
            if (banners.isEmpty()) {
                return new GdAdWithTotals()
                        .withGdAds(emptyList())
                        .withTotalStats(gdEntityStats);
            }
        }

        Set<Long> campaignIds = listToSet(banners, GdiBanner::getCampaignId);
        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService.getTruncatedCampaigns(clientId, campaignIds);
        banners = filterList(banners, b -> campaignsById.containsKey(b.getCampaignId()));
        if (campaignsById.isEmpty()) {
            return new GdAdWithTotals()
                    .withGdAds(emptyList())
                    .withTotalStats(gdEntityStats);
        }

        Set<Long> adGroupIds = listToSet(banners, GdiBanner::getGroupId);
        List<GdAdGroupTruncated> adGroups = groupDataService
                .getTruncatedAdGroups(client.getShard(), client.getCountryRegionId(), clientId,
                        operator, adGroupIds, campaignsById);
        if (adGroups.isEmpty()) {
            return new GdAdWithTotals()
                    .withGdAds(emptyList())
                    .withTotalStats(gdEntityStats);
        }

        boolean showAggregatedStatusesDebug = featureService
                .isEnabledForClientId(operator.getClientId(), FeatureName.SHOW_AGGREGATED_STATUS_DEBUG);
        enrichBannersAggregatedStatuses(banners, adStatusesByIds, showAggregatedStatusesDebug);

        enrichBannersData(client.getShard(), banners, adGroups);
        enrichBannersStatuses(banners);

        // Не кидаем исключение если справочник недоступен, т.к. в этом случае все равно хотим показать
        // список баннеров на фронте.
        Map<Long, Boolean> permalinkHasAccess = organizationService.hasAccessNoThrow(clientId, mapList(banners,
                GdiBanner::getPermalinkId));
        banners.forEach(b -> {
            b.setCanEditOrganizationPhone(
                    b.getPermalinkId() != null
                            && permalinkHasAccess.getOrDefault(b.getPermalinkId(), false));
        });

        //Если для фильтра рекомендации не требовались, то выбираем их здесь для заполнения только по нужным кампаниям
        if (adFetchedFieldsResolver.getRecommendations() && recommendationTypes.isEmpty()) {
            bannerIdToRecommendations = getRecommendations(clientId, operator, recommendationTypes, campaignIds);
        }
        Map<Long, List<GdRecommendation>> finalBannerIdToRecommendations = bannerIdToRecommendations;

        boolean isStatusFilterEmpty =
                filter.getPrimaryStatusContains() == null || filter.getPrimaryStatusContains().isEmpty();
        Map<Long, GdAdGroupTruncated> groupIdToGroup = listToMap(adGroups, GdAdGroupTruncated::getId);
        Counter counter = new Counter();

        List<GdAdOrderBy> bannerOrderByCalculatedFields =
                filterList(input.getOrderBy(), BannerOrderHelper::isOrderingOccursAfterDataFetching);

        Set<GdAdType> typeIn = filter.getTypeIn();
        var allowedReasonsToSelfRemoderate = allowedReasonsToSelfRemoderateSupplier().get();

        List<GdAd> gdAds = StreamEx.of(banners)
                //на группы может не быть доступа или тип группы может не поддерживаться
                .filter(b -> groupIdToGroup.containsKey(b.getGroupId()))
                .map(b -> toGdAdImplementation(counter.next(), context, groupIdToGroup.get(b.getGroupId()),
                        enabledFeatures, allowedReasonsToSelfRemoderate, b))
                .filter(Objects::nonNull)
                //Дополнительная фильтрация по типу для cpm_video, который не передавался в запрос
                .filter(b -> typeIn == null || typeIn.contains(b.getType()))
                //Не фильтруем по статусу TEMPORARILY_SUSPENDED, поэтому если фильтр не пустой то убираем баннеры с
                // таким статусом
                .filter(b -> isStatusFilterEmpty
                        || GdAdPrimaryStatus.TEMPORARILY_SUSPENDED != b.getStatus().getPrimaryStatus())
                //Сортировка по полям, которые не содержатся в YT, а вычисляются.
                .chain(s -> sortBanners(bannerOrderByCalculatedFields, s))
                .map(b -> b.withRecommendations(finalBannerIdToRecommendations.get(b.getId())))
                .toList();
        return new GdAdWithTotals()
                .withGdAds(gdAds)
                .withTotalStats(gdEntityStats);
    }

    private void updateFilterForTouch(GdiBannerFilter internalFilter, GdClientInfo client) {
        var touchCampaignIds = campaignInfoService.getTouchCampaignIds(client.getShard(),
                ClientId.fromLong(client.getId()));
        if (!internalFilter.getCampaignIdIn().isEmpty()) {
            touchCampaignIds.retainAll(internalFilter.getCampaignIdIn());
        }
        internalFilter.setCampaignIdIn(touchCampaignIds);
    }

    /**
     * Получить главные баннеры для групп, к которым относятся условия показа. Понятие главного баннера группы описано
     * тут —
     * {@link ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository#getMainBannerIdsByAdGroupIds}
     *
     * @return Мапа соответствий id групп к главным баннерам групп
     */
    public Map<Long, GdAd> getMainBanners(int shard, ClientId clientId, Map<Long, GdAdGroupTruncated> groups) {
        Map<Long, Long> mainBannersIds =
                bannerRelationsRepository.getMainBannerIdsByAdGroupIds(clientId, groups.keySet());
        List<GdAd> mainBanners =
                getBannersByIdsWithoutStats(shard, clientId, groups, mainBannersIds.values(), false);
        return listToMap(mainBanners, GdAd::getAdGroupId);
    }

    /**
     * Получить данные о баннерах. Достаем баннеры по идентификатору из MySQL без статистики с данными необходимыми
     * для последующего отображения в интерфейсе.
     *
     * @return Список баннеров
     */
    private List<GdAd> getBannersByIdsWithoutStats(int shard, ClientId clientId, Map<Long, GdAdGroupTruncated> groups,
                                                   Collection<Long> bannerIds, boolean addAutoAssignedPermalinks) {
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        List<GdiBanner> mainGdiBanners =
                gridBannerService.getBannersWithoutStats(shard, bannerIds, clientId,
                        enabledFeatures, addAutoAssignedPermalinks);
        Counter counter = new Counter();
        GridGraphQLContext context = gridContextProvider.getGridContext();
        var allowedReasonsToSelfRemoderate = allowedReasonsToSelfRemoderateSupplier().get();

        return mapAndFilterList(mainGdiBanners, b -> toGdAdImplementation(counter.next(),
                        context, groups.get(b.getGroupId()), enabledFeatures, allowedReasonsToSelfRemoderate, b),
                Objects::nonNull);
    }

    /**
     * Отправляет запрос в Турбо, возвращает ссылку в формате листинга турбо-галереи
     *
     * @param originalHref ссылка, введенная пользователем
     * @return Ссылка на турбо-галерею
     */
    public String checkTurboGalleryHref(String originalHref) {
        adValidationService.validateHref(originalHref);
        Map<String, String> turboGalleryHrefs = turbopagesClient.checkUrls(singletonList(originalHref));
        if (!turboGalleryHrefs.containsKey(originalHref)) {
            return "";
        }
        return turboGalleryHrefs.get(originalHref);
    }

    private void enrichBannersData(int shard, List<GdiBanner> banners, List<GdAdGroupTruncated> adGroups) {
        //Вычисляем statusMonitoringDomainStop. Данные берем из mysql
        Set<Long> campaignsAllowedDomainMonitoring = getCampaignsAllowedDomainMonitoring(adGroups);
        Set<Long> bannerIds = listToSet(banners, GdiBanner::getId);
        Set<Long> bannersMonitoringStopped = gridBannerService.getMonitoringStoppedBanners(shard, bannerIds);
        var bannersToGetModReasons = filterAndMapToSet(banners,
                gdiBanner -> gdiBanner.getAggregatedStatusInfo() == null ||
                        gdiBanner.getAggregatedStatusInfo().getMightHaveRejectReasons(),
                GdiBanner::getId);
        var modReasonIdsByBannerId = moderationReasonService.getReasonIdsForBannerAndResources(shard,
                bannersToGetModReasons);

        banners.forEach(b -> {
            b.setStatusMonitoringDomainStop(
                    campaignsAllowedDomainMonitoring.contains(b.getCampaignId())
                            && bannersMonitoringStopped.contains(b.getId())
            );
            b.setModerationReasonIds(modReasonIdsByBannerId.getOrDefault(b.getId(), emptySet()));
        });

    }

    private void enrichBannersStatuses(List<GdiBanner> banners) {

        // У смарт-банеров матчим статусы модерации креатива и банера и сохраняем в статус банера (DIRECT-93354)
        banners.stream()
                .filter(b -> b.getBannerType() == BannersBannerType.performance)
                .forEach(smartBanner -> {
                    Creative typedCreative = smartBanner.getTypedCreative();
                    StatusModerate creativeSM = typedCreative.getStatusModerate();
                    GdiBannerStatusModerate convertedCreativeSM = toGdiBannerStatusModerate(creativeSM);
                    Integer creativePriority = BANNER_STATUS_MERGE_PRIORITY.get(convertedCreativeSM);
                    Integer bannerPriority = BANNER_STATUS_MERGE_PRIORITY.get(smartBanner.getStatusModerate());
                    if (bannerPriority < creativePriority) {
                        smartBanner.setStatusModerate(convertedCreativeSM);
                    }
                });
    }

    private void enrichBannersAggregatedStatuses(List<GdiBanner> banners,
                                                 Map<Long, AggregatedStatusAdData> adStatusesByIds,
                                                 boolean debug) {
        Map<Long, String> statuses = debug
                ? aggregatedStatusesViewService.statusesToJsonString(adStatusesByIds)
                : emptyMap();
        for (GdiBanner banner : banners) {
            AggregatedStatusAdData statusData = adStatusesByIds.get(banner.getId());
            banner.setAggregatedStatusInfo(toGdAdAggregatedStatusInfo(statusData));
            if (debug) {
                banner.setAggregatedStatus(statuses.getOrDefault(banner.getId(), "{\"status\" : \"no data\"}"));
            }
        }
    }

    public GdAdAggregatedStatusInfo toGdAdAggregatedStatusInfo(AggregatedStatusAdData statusData) {
        if (statusData == null || statusData.getStatus().isEmpty()) {
            return null;
        }
        return new GdAdAggregatedStatusInfo()
                .withMightHaveRejectReasons(
                        aggregatedStatusesViewService.adCouldHaveRejectReasons(statusData))
                .withStatus(statusData.getStatus().get())
                .withReasons(statusData.getReasons())
                .withRejectReasons(GridModerationUtils.toGdRejectReasons(statusData.getRejectReasons()))
                .withIsObsolete(statusData.getIsObsolete());
    }

    private Map<Long, List<GdRecommendation>> getRecommendations(ClientId clientId, User operator,
                                                                 Set<GdiRecommendationType> recommendationTypes,
                                                                 Set<Long> campaignIds) {
        try (TraceProfile ignore = Trace.current().profile("recommendations:service:banners")) {
            return gridRecommendationService
                    .getBannerRecommendations(clientId.asLong(), operator, recommendationTypes, campaignIds);
        }
    }

    /**
     * Массовое обновление баннеров
     *
     * @param clientId    id клиента
     * @param operator    оператор
     * @param gdUpdateAds запрос на обновление баннеров
     * @return id обновленных баннеров
     */
    @SuppressWarnings("WeakerAccess")
    public GdUpdateAdsPayload updateAds(ClientId clientId, User operator, GdUpdateAds gdUpdateAds) {
        if (isEmpty(gdUpdateAds.getAdUpdateItems())) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }
        adValidationService.validateUpdateAdsRequest(gdUpdateAds);

        fillUnknownTurboGalleryHref(clientId, gdUpdateAds);

        var bannersToUpdate = toCoreBanners(gdUpdateAds.getAdUpdateItems());
        MassResult<Long> updateResult =
                updateCoreBanners(clientId, operator, bannersToUpdate, gdUpdateAds.getSaveDraft());

        adGenerationLoggingService.logBannerTitleAndSnippet(operator.getUid(), clientId, bannersToUpdate, updateResult);

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult, path(field(GdUpdateAds.AD_UPDATE_ITEMS)));
        List<GdUpdateAdPayloadItem> updatedAds =
                getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое обновление dynamic баннеров
     *
     * @param clientId    id клиента
     * @param operator    оператор
     * @param gdUpdateAds запрос на обновление баннеров
     * @return id обновленных баннеров
     */
    GdUpdateAdsPayload updateDynamicAds(ClientId clientId, User operator, GdUpdateDynamicAds gdUpdateAds) {
        if (isEmpty(gdUpdateAds.getAdUpdateItems())) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }
        adValidationService.validateUpdateDynamicAdsRequest(gdUpdateAds);

        List<BannerWithSystemFields> bannersToUpdate = mapList(gdUpdateAds.getAdUpdateItems(),
                AdMutationDataConverter::toDynamicBanner);
        MassResult<Long> updateResult =
                updateCoreBanners(clientId, operator, bannersToUpdate, gdUpdateAds.getSaveDraft());

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult,
                        path(field(GdUpdateDynamicAds.AD_UPDATE_ITEMS)));
        List<GdUpdateAdPayloadItem> updatedAds =
                getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    GdUpdateAdsPayload updateTurboGalleryHrefs(ClientId clientId, User operator,
                                               List<GdUpdateAdTurboGalleryHref> gdUpdateHrefs) {
        if (isEmpty(gdUpdateHrefs)) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList());
        }

        List<ModelChanges<BannerWithSystemFields>> modelChanges = StreamEx.of(gdUpdateHrefs)
                .map(this::getModelChangesForTurboGallery)
                .toList();

        var bannersUpdateOperation = bannersUpdateOperationFactory
                .createPartialUpdateOperation(ModerationMode.DEFAULT, modelChanges, operator.getUid(), clientId);

        MassResult<Long> updateResult = bannersUpdateOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult, path(field(GdUpdateAdsPayload.UPDATED_ADS)));
        List<GdUpdateAdPayloadItem> updatedAds = getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));

        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    private ModelChanges<BannerWithSystemFields> getModelChangesForTurboGallery(GdUpdateAdTurboGalleryHref input) {
        return new ModelChanges<>(input.getBannerId(), TextBanner.class)
                .process(input.getHref(), TextBanner.TURBO_GALLERY_HREF)
                .castModelUp(BannerWithSystemFields.class);
    }

    GdUpdateSmartCentersPayload updateSmartCenters(ClientId clientId, String imageHash,
                                                   List<GdUpdateSmartCentersItem> gdUpdateSmartCentersItems) {
        adValidationService.validateUpdateSmartCenters(gdUpdateSmartCentersItems);

        int shard = shardHelper.getShardByClientId(clientId);
        Image image = imageDataRepository.getImage(shard, clientId, imageHash);

        if (image == null) {
            GdValidationResult validationResult = gridValidationService
                    .getValidationResult(Result.broken(ValidationResult.failed(imageHash, imageNotFound())),
                            path(field(GdUpdateSmartCenters.IMAGE_HASH)));
            throw new GridValidationException(validationResult);
        }

        ImageType imageType = image.getImageType();
        ImageMdsMeta mdsMeta = image.getMdsMeta();

        Set<String> supportedFormats = SUPPORTED_FORMATS_BY_IMAGE_TYPE.get(imageType);

        ImageMdsMeta overriddenMdsMeta = new ImageMdsMeta().withSizes(new HashMap<>());
        for (GdUpdateSmartCentersItem item : gdUpdateSmartCentersItems) {
            SmartCenterUtils
                    .overrideSmartCenters(overriddenMdsMeta, supportedFormats, mdsMeta,
                            item.getSmartCenter().getX(),
                            item.getSmartCenter().getY(),
                            item.getSize(),
                            item.getSmartCenter().getRatio());
        }

        imageService.saveOverriddenSmartCenters(shard, clientId, imageHash, overriddenMdsMeta);

        Image updatedImage = imageDataRepository.getImage(shard, clientId, imageHash);

        return new GdUpdateSmartCentersPayload()
                .withImage(toGdImage(updatedImage));
    }

    /**
     * На время, пока фронт не поддержит работу с турбо-галереей. Дополняет модель ссылками из базы,
     * если фронт не отдает TurboGalleryParams
     */
    private void fillUnknownTurboGalleryHref(ClientId clientId, GdUpdateAds gdUpdateAds) {
        Set<Long> bannerIdsWithoutTurboGalleryHref = StreamEx.of(gdUpdateAds.getAdUpdateItems())
                .filter(ad -> ad.getTurboGalleryParams() == null)
                .map(GdUpdateAd::getId)
                .toSet();

        Map<Long, String> turboGalleryByBannerIds = gridBannerService.getTurboGalleryByBannerIds(clientId,
                bannerIdsWithoutTurboGalleryHref);

        gdUpdateAds.getAdUpdateItems().forEach(gd -> {
            if (gd.getTurboGalleryParams() == null) {
                gd.setTurboGalleryParams(new GdTurboGalleryParams()
                        .withTurboGalleryHref(turboGalleryByBannerIds.get(gd.getId())));
            }
        });
    }

    /**
     * Массовое обновление smart баннеров
     *
     * @param clientId         id клиента
     * @param operator         оператор
     * @param gdUpdateSmartAds запрос на обновление баннеров
     * @return id обновленных баннеров
     */
    GdUpdateAdsPayload updateSmartAds(ClientId clientId, User operator, GdUpdateSmartAds gdUpdateSmartAds) {
        if (isEmpty(gdUpdateSmartAds.getAdUpdateItems())) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }

        List<BannerWithSystemFields> bannersToUpdate = mapList(gdUpdateSmartAds.getAdUpdateItems(),
                AdMutationDataConverter::toPerformanceBanner);
        // клиенты без фичи не могут редактировать баннеры нового типа (performance_banner_main)
        // клиенты с фичей могут редактировать баннеры обоих типов (и старые, и новые)
        if (!featureService.isEnabledForClientId(clientId, FeatureName.SMART_NO_CREATIVES)) {
            bannersToUpdate = filterList(bannersToUpdate, b -> b instanceof PerformanceBanner);

            if (isEmpty(bannersToUpdate)) {
                return new GdUpdateAdsPayload()
                        .withUpdatedAds(emptyList())
                        .withValidationResult(null);
            }
        }
        MassResult<Long> updateResult =
                updateCoreBanners(clientId, operator, bannersToUpdate, gdUpdateSmartAds.getSaveDraft());

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult, path(field(GdUpdateSmartAds.AD_UPDATE_ITEMS)));
        List<GdUpdateAdPayloadItem> updatedAds =
                getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    private MassResult<Long> updateCoreBanners(ClientId clientId, User operator,
                                               List<BannerWithSystemFields> bannersToUpdate,
                                               boolean saveDraft) {
        ModerationMode moderationMode = saveDraft ? ModerationMode.FORCE_SAVE_DRAFT : ModerationMode.FORCE_MODERATE;

        var modelsToUpdate = bannersToCoreModelChanges(bannersToUpdate);
        populateBannerFlagsFromDb(filterList(modelsToUpdate,
                model -> model.isPropChanged(BannerWithSystemFields.FLAGS)));

        var bannersUpdateOperation = bannersUpdateOperationFactory
                .createPartialUpdateOperation(moderationMode,
                        modelsToUpdate,
                        operator.getUid(), clientId);

        return bannersUpdateOperation.prepareAndApply();
    }

    /**
     * Добавляет к флагам баннера от пользователя уже существующие из базы, которые пришли из модерации.
     * Сделанно в рамках DIRECT-138324
     *
     * @param bannersNeedToPopulate - изменения от пользователя
     */
    private void populateBannerFlagsFromDb(List<ModelChanges<BannerWithSystemFields>> bannersNeedToPopulate) {
        var modelsById = listToMap(bannersNeedToPopulate, ModelChanges::getId);
        var bannersInDb = newBannerService.getBannersByIds(modelsById.keySet());
        bannersInDb.forEach(banner -> mergeBannerFlagsWithDb(modelsById.get(banner.getId()), banner));
    }

    private void mergeBannerFlagsWithDb(ModelChanges<BannerWithSystemFields> modelChanges,
                                        BannerWithSystemFields banner) {
        Map<String, String> newFlags = new HashMap<>();
        var flagsChanges = nvl(
                modelChanges.getChangedProp(BannerWithSystemFields.FLAGS),
                BannerWithFlagsConstraints.EMPTY_FLAGS
        );
        var bannerFlags = nvl(banner.getFlags(), BannerWithFlagsConstraints.EMPTY_FLAGS);
        newFlags.putAll(bannerFlags.getFlags());
        newFlags.putAll(flagsChanges.getFlags());
        modelChanges.process(new BannerFlags().withFlags(newFlags), BannerWithSystemFields.FLAGS);
    }

    GdAddAdsPayload addAds(ClientId clientId, User operator, GdAddAds input) {
        if (isEmpty(input.getAdAddItems())) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(null);
        }
        adValidationService.validateAddAdsRequest(input);

        var banners = mapList(input.getAdAddItems(), AddAdMutationDataConverter::toCoreBanner);

        BannersAddOperation addOperation =
                bannersAddOperationFactory
                        .createPartialAddOperation(banners, clientId, operator.getUid(), input.getSaveDraft());

        MassResult<Long> result = addOperation.prepareAndApply();

        adGenerationLoggingService.logBannerTitleAndSnippet(operator.getUid(), clientId, banners, result);

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddAds.AD_ADD_ITEMS)));
        List<GdAddAdPayloadItem> addedAds =
                getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(addedAds)
                .withValidationResult(validationResult);
    }

    GdAddAdsPayload addDynamicAds(ClientId clientId, User operator, GdAddDynamicAds input) {
        if (isEmpty(input.getAdAddItems())) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(null);
        }
        adValidationService.validateAddDynamicAdsRequest(input);

        List<BannerWithAdGroupId> banners = mapList(input.getAdAddItems(),
                AddAdMutationDataConverter::toDynamicBanner);
        BannersAddOperation addOperation =
                bannersAddOperationFactory
                        .createPartialAddOperation(banners, clientId, operator.getUid(), input.getSaveDraft());

        MassResult<Long> result = addOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddDynamicAds.AD_ADD_ITEMS)));
        List<GdAddAdPayloadItem> addedAds =
                getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(addedAds)
                .withValidationResult(validationResult);
    }

    GdAddAdsPayload addSmartAds(ClientId clientId, User operator, GdAddSmartAds input) {
        if (isEmpty(input.getAdAddItems())) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(null);
        }

        List<BannerWithAdGroupId> banners = mapList(input.getAdAddItems(),
                AddAdMutationDataConverter::toPerformanceBanner);
        // клиенты без фичи не могут добавлять баннеры нового типа (performance_banner_main)
        // клиенты с фичей могут добавлять баннеры обоих типов (и старые, и новые)
        if (!featureService.isEnabledForClientId(clientId, FeatureName.SMART_NO_CREATIVES)) {
            banners = filterList(banners, b -> b instanceof PerformanceBanner);

            if (isEmpty(banners)) {
                return new GdAddAdsPayload()
                        .withAddedAds(emptyList())
                        .withValidationResult(null);
            }
        }
        BannersAddOperation addOperation =
                bannersAddOperationFactory
                        .createPartialAddOperation(banners, clientId, operator.getUid(), input.getSaveDraft());

        MassResult<Long> result = addOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddSmartAds.AD_ADD_ITEMS)));
        List<GdAddAdPayloadItem> addedAds =
                getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(addedAds)
                .withValidationResult(validationResult);
    }

    GdAddSitelinkSetsPayload addSitelinkSets(ClientId clientId, GdAddSitelinkSets input) {
        adValidationService.validateAddSitelinkSetsRequest(input);

        List<SitelinkSet> sitelinkSets = mapList(input.getSitelinkSetsAddItems(), SitelinkConverter::toCoreSet);
        MassResult<Long> result = sitelinkSetService.addSitelinkSetsPartial(clientId, sitelinkSets);

        GdValidationResult validationResult = gridValidationService
                .getValidationResult(result, path(field(GdAddSitelinkSets.SITELINK_SETS_ADD_ITEMS)));
        List<GdAddSitelinkSetPayloadItem> addedSitelinkSets =
                getResults(result, id -> new GdAddSitelinkSetPayloadItem().withId(id));
        return new GdAddSitelinkSetsPayload()
                .withAddedSitelinkSets(addedSitelinkSets)
                .withValidationResult(validationResult);
    }

    @Nullable
    GdValidationResult validateAdPrice(@Nonnull List<GdAdPrice> prices) {
        ValidationResult<List<GdAdPrice>, Defect> vr = adValidationService.validateAdPrices(prices);

        if (hasValidationIssues(vr)) {
            return GridValidationResultConversionService
                    .buildGridValidationResult(vr, emptyPath(), adPricePathNodeConverterProvider);
        }
        return null;
    }

    @Nonnull
    GdUpdateAdsPayload updateAdPrices(ClientId clientId, User operator,
                                      @Nonnull @GraphQLNonNull List<@GraphQLNonNull GdUpdateAdPrice> request) {
        adValidationService.validateUpdateAdPricesRequest(request);

        List<ModelChanges<BannerWithPrice>> modelChanges = StreamEx.of(request)
                .map(gdUpdateAdPrice -> ModelChanges.build(gdUpdateAdPrice.getId(), BannerWithPrice.class,
                        BannerWithPrice.BANNER_PRICE, toCoreBannerPrice(gdUpdateAdPrice.getPrice())))
                .toList();
        MassResult<Long> result = bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, operator.getUid(), clientId, BannerWithPrice.class)
                .prepareAndApply();

        return toGdUpdateAdsPayload(result);
    }

    @Nonnull
    GdUpdateAdsPayload updateOrganizations(
            @Nonnull ClientId clientId,
            @Nonnull User operator,
            @Nonnull @GraphQLNonNull List<@GraphQLNonNull GdUpdateOrganization> request) {
        adValidationService.validateUpdateOrganizationsRequest(request);

        List<ModelChanges<BannerWithSystemFields>> modelChanges = StreamEx.of(request)
                .map(org -> {
                    ModelChanges<TextBanner> changes = ModelChanges.build(org.getBannerId(),
                            TextBanner.class,
                            BannerWithOrganization.PERMALINK_ID,
                            org.getPermalinkId());
                    if (org.getPermalinkId() == null) {
                        changes.process(null, BannerWithOrganizationAndPhone.PREFER_V_CARD_OVER_PERMALINK);
                        changes.process(null, BannerWithOrganizationAndPhone.PHONE_ID);
                    }
                    return changes.castModelUp(BannerWithSystemFields.class);
                })
                .toList();
        MassResult<Long> result = bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, operator.getUid(), clientId)
                .prepareAndApply();

        return toGdUpdateAdsPayload(result);
    }

    @Nonnull
    GdUpdateAdsPayload rejectAutoOrganizations(
            @Nonnull ClientId clientId,
            @Nonnull @GraphQLNonNull List<@GraphQLNonNull GdUpdateOrganization> request) {
        if (request.isEmpty()) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<Long> bannerIds = StreamEx.of(request).map(GdUpdateOrganization::getBannerId).toList();
        Set<Long> existingBannerIds = bannerTypedRepository.getClientBannerIds(shard, clientId, bannerIds);
        adValidationService.validateRejectAutoOrganizationsRequest(request, existingBannerIds);

        Map<Long, Long> permalinkIdsByBannerIds = StreamEx.of(request)
                .mapToEntry(GdUpdateOrganization::getBannerId, GdUpdateOrganization::getPermalinkId)
                .nonNullKeys()
                .nonNullValues()
                .toMap();
        organizationRepository.rejectAutoOrganizations(shard, permalinkIdsByBannerIds);

        List<GdUpdateAdPayloadItem> updatedAds = StreamEx.of(bannerIds)
                .map(bannerId -> new GdUpdateAdPayloadItem().withId(bannerId))
                .toList();
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(null);
    }

    GdUpdateAdsPayload toGdUpdateAdsPayload(MassResult<Long> result) {
        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdUpdateAds.AD_UPDATE_ITEMS)));

        List<GdUpdateAdPayloadItem> successfullyUpdatedAds = getSuccessfullyResults(result,
                id -> new GdUpdateAdPayloadItem().withId(id));

        return new GdUpdateAdsPayload()
                .withUpdatedAds(successfullyUpdatedAds)
                .withValidationResult(validationResult);
    }

    @Nonnull
    GdUpdateAdsPayload addSitelinkSetsToAds(ClientId clientId, User operator, GdAddSitelinkSetToAds input) {
        adValidationService.validateAddSitelinkSetsToAdsRequest(input);

        GdiAdMassChangeResult result = gridAdsMassChangesService.addSitelinkSet(operator.getUid(),
                clientId,
                input.getAdIds(),
                input.getSitelinkSetId());
        return massChangeResultToGdUpdateAdsPayload(result);
    }

    @Nonnull
    GdUpdateAdsPayload addTurboLandingToAds(ClientId clientId, User operator, GdAddTurboLandingToAds input) {
        adValidationService.validateAddTurboLandingToAdsRequest(input);

        GdiAdMassChangeResult result = gridAdsMassChangesService.addTurboLanding(operator.getUid(),
                clientId,
                input.getAdIds(),
                input.getTurboLandingId());
        return massChangeResultToGdUpdateAdsPayload(result);
    }

    @Nonnull
    GdUpdateAdsPayload addBannerImage(ClientId clientId, User operator, GdAddAdImage input) {
        adValidationService.validateAddBannerImageRequest(input);
        GdiAdMassChangeResult result = gridAdsMassChangesService.addBannerImage(operator.getUid(), clientId,
                input.getAdIds(),
                input.getAdImageHash());
        return massChangeResultToGdUpdateAdsPayload(result);
    }

    @Nonnull
    GdUpdateAdsPayload addCreative(ClientId clientId, User operator, GdAddAdCreative input) {
        adValidationService.validateAddCreativeRequest(input);

        GdiAdMassChangeResult result = gridAdsMassChangesService.addCreative(operator.getUid(), clientId,
                input.getAdIds(),
                toBannerType(input.getAdType()),
                getAdGroupType(input.getAdType()),
                input.getCreativeId(),
                input.getShowTitleAndBody());
        return massChangeResultToGdUpdateAdsPayload(result);
    }

    @Nonnull
    GdUpdateAdsPayload addMulticardsToAds(ClientId clientId, User operator, GdAddMulticardsToAds input) {
        adValidationService.validateAddMulticardsRequest(input);

        List<BannerMulticard> multicards = mapList(input.getMulticards(),
                AddAdMutationDataConverter::toCoreBannerMulticard);

        GdiAdMassChangeResult result = gridAdsMassChangesService.addMulticards(operator.getUid(), clientId,
                input.getAdIds(),
                multicards);
        return massChangeResultToGdUpdateAdsPayload(result);
    }

    private GdUpdateAdsPayload massChangeResultToGdUpdateAdsPayload(GdiAdMassChangeResult result) {
        GdValidationResult validationResult = gridValidationService
                .toGdValidationResult(result.getValidationResult(), path(field(GdUpdateAds.AD_UPDATE_ITEMS)));

        List<GdUpdateAdPayloadItem> successfullyUpdatedAds =
                mapList(result.getSuccessfullyProcessedIds(), x -> new GdUpdateAdPayloadItem().withId(x));

        return new GdUpdateAdsPayload()
                .withUpdatedAds(successfullyUpdatedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое создание cpm баннеров
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param input    запрос на создание баннеров
     * @return id созданных баннеров
     */
    GdAddAdsPayload addCpmAds(ClientId clientId, User operator, GdAddCpmAds input) {
        if (isEmpty(input.getAdAddItems())) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(null);
        }

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);

        ValidationResult<GdAddCpmAds, Defect> preValidationResult =
                adValidationService.validateAddCpmAdsRequest(input, enabledFeatures);

        if (preValidationResult.hasAnyErrors()) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(gridValidationService.toGdValidationResult(preValidationResult, emptyPath()));
        }

        var banners = mapList(input.getAdAddItems(), AddAdMutationDataConverter::toCoreBanner);

        BannersAddOperation addOperation =
                bannersAddOperationFactory
                        .createPartialAddOperation(banners, clientId, operator.getUid(), input.getSaveDraft());

        MassResult<Long> result = addOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddSmartAds.AD_ADD_ITEMS)));
        List<GdAddAdPayloadItem> addedAds =
                getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(addedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое обновление cpm баннеров
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param input    запрос на обновление баннеров
     * @return id обновленных баннеров
     */
    GdUpdateAdsPayload updateCpmAds(ClientId clientId, User operator, GdUpdateCpmAds input) {
        if (isEmpty(input.getAdUpdateItems())) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);

        ValidationResult<GdUpdateCpmAds, Defect> preValidationResult =
                adValidationService.validateUpdateCpmAdsRequest(input, enabledFeatures);
        if (preValidationResult.hasAnyErrors()) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(gridValidationService.toGdValidationResult(preValidationResult, emptyPath()));
        }

        var bannersToUpdate = mapList(input.getAdUpdateItems(), AdMutationDataConverter::toCoreBanner);

        MassResult<Long> updateResult =
                updateCoreBanners(clientId, operator, bannersToUpdate, input.getSaveDraft());

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult, path(field(GdUpdateSmartAds.AD_UPDATE_ITEMS)));
        List<GdUpdateAdPayloadItem> updatedAds =
                getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое создание mcbanner баннеров
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param inputAds запрос на создание баннеров
     * @return id созданных баннеров
     */
    GdAddAdsPayload addMcBannerAds(ClientId clientId, User operator, GdAddMcBannerAds inputAds) {
        if (isEmpty(inputAds.getAdAddItems())) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(null);
        }
        adValidationService.validateAddMcBannerAdsRequest(inputAds);
        List<BannerWithAdGroupId> banners = mapList(inputAds.getAdAddItems(),
                AddAdMutationDataConverter::toMcBanner);
        BannersAddOperation addOperation =
                bannersAddOperationFactory
                        .createPartialAddOperation(banners, clientId, operator.getUid(), inputAds.getSaveDraft());

        MassResult<Long> result = addOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddMcBannerAds.AD_ADD_ITEMS)));
        List<GdAddAdPayloadItem> addedAds =
                getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(addedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое обновление mcbanner баннеров
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param inputAds запрос на обновление баннеров
     * @return id обновленных баннеров
     */
    GdUpdateAdsPayload updateMcBannerAds(ClientId clientId, User operator, GdUpdateMcBannerAds inputAds) {
        if (isEmpty(inputAds.getAdUpdateItems())) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }

        adValidationService.validateUpdateMcBannerAdsRequest(inputAds);
        List<BannerWithSystemFields> bannersToUpdate = mapList(inputAds.getAdUpdateItems(),
                AdMutationDataConverter::toMcBanner);
        MassResult<Long> updateResult =
                updateCoreBanners(clientId, operator, bannersToUpdate, inputAds.getSaveDraft());

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult,
                        path(field(GdUpdateMcBannerAds.AD_UPDATE_ITEMS)));
        List<GdUpdateAdPayloadItem> updatedAds =
                getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое создание mobile_content баннеров
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param inputAds запрос на создание баннеров
     * @return id созданных баннеров
     */
    GdAddAdsPayload addMobileContentAds(ClientId clientId, User operator, GdAddMobileContentAds inputAds) {
        if (isEmpty(inputAds.getAdAddItems())) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(null);
        }
        adValidationService.validateAddMobileContentAdsRequest(inputAds);
        List<BannerWithAdGroupId> banners = mapList(inputAds.getAdAddItems(),
                AddAdMutationDataConverter::toMobileAppBanner);
        BannersAddOperation addOperation =
                bannersAddOperationFactory
                        .createPartialAddOperation(banners, clientId, operator.getUid(), inputAds.getSaveDraft());

        MassResult<Long> result = addOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddMobileContentAds.AD_ADD_ITEMS)));
        List<GdAddAdPayloadItem> addedAds =
                getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(addedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое обновление mobile_content баннеров
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param inputAds запрос на обновление баннеров
     * @return id обновленных баннеров
     */
    GdUpdateAdsPayload updateMobileContentAds(ClientId clientId, User operator, GdUpdateMobileContentAds inputAds) {
        if (isEmpty(inputAds.getAdUpdateItems())) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }

        adValidationService.validateUpdateMobileContentAdsRequest(inputAds);
        List<BannerWithSystemFields> bannersToUpdate = mapList(inputAds.getAdUpdateItems(),
                AdMutationDataConverter::toMobileAppBanner);
        MassResult<Long> updateResult =
                updateCoreBanners(clientId, operator, bannersToUpdate, inputAds.getSaveDraft());

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult,
                        path(field(GdUpdateMobileContentAds.AD_UPDATE_ITEMS)));
        List<GdUpdateAdPayloadItem> updatedAds =
                getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое создание баннеров продвижения контента
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param inputAds запрос на создание баннеров
     * @return id созданных баннеров
     */
    GdAddAdsPayload addContentPromotionAds(ClientId clientId, User operator, GdAddContentPromotionAds inputAds) {
        if (isEmpty(inputAds.getAdAddItems())) {
            return new GdAddAdsPayload()
                    .withAddedAds(emptyList())
                    .withValidationResult(null);
        }

        adValidationService.validateAddContentPromotionAdsRequest(inputAds);
        List<BannerWithAdGroupId> banners = mapList(inputAds.getAdAddItems(),
                AddAdMutationDataConverter::toContentPromotion);
        BannersAddOperation addOperation =
                bannersAddOperationFactory
                        .createPartialAddOperation(banners, clientId, operator.getUid(), inputAds.getSaveDraft());

        MassResult<Long> result = addOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddContentPromotionAds.AD_ADD_ITEMS)));
        List<GdAddAdPayloadItem> addedAds =
                getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(addedAds)
                .withValidationResult(validationResult);
    }

    /**
     * Массовое обновление баннеров продвижения контента
     *
     * @param clientId id клиента
     * @param operator оператор
     * @param inputAds запрос на обновление баннеров
     * @return id обновленных баннеров
     */
    GdUpdateAdsPayload updateContentPromotionAds(ClientId clientId, User operator,
                                                 GdUpdateContentPromotionAds inputAds) {
        if (isEmpty(inputAds.getAdUpdateItems())) {
            return new GdUpdateAdsPayload()
                    .withUpdatedAds(emptyList())
                    .withValidationResult(null);
        }

        adValidationService.validateUpdateContentPromotionAdsRequest(inputAds);
        List<BannerWithSystemFields> bannersToUpdate = mapList(inputAds.getAdUpdateItems(),
                AdMutationDataConverter::toContentPromotionBanner);
        MassResult<Long> updateResult =
                updateCoreBanners(clientId, operator, bannersToUpdate, inputAds.getSaveDraft());

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult,
                        path(field(GdUpdateContentPromotionAds.AD_UPDATE_ITEMS)));
        List<GdUpdateAdPayloadItem> updatedAds =
                getResults(updateResult, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(updatedAds)
                .withValidationResult(validationResult);
    }

    public GdLastChangedAds getLastChangedAds(GdLastChangedAdContainer input) {
        adValidationService.validateGetLastChangedAdRequest(input);

        GridGraphQLContext context = gridContextProvider.getGridContext();
        GdClientInfo client = context.getQueriedClient();
        User operator = context.getOperator();
        int shard = client.getShard();
        ClientId clientId = ClientId.fromLong(client.getId());


        Map<Long, List<BannersBannerType>> bannerTypesByCampaignId = StreamEx.of(input.getItems())
                .mapToEntry(GdLastChangedAdContainerItem::getCampaignId, item -> toInternalAdType(item.getAdType()))
                .grouping();

        Map<Long, Long> campaignIdByBannerId = newBannerService.getLastChangedBannerIds(shard, clientId,
                bannerTypesByCampaignId);

        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService.getTruncatedCampaigns(clientId,
                StreamEx.of(campaignIdByBannerId.values()).distinct().toSet());

        Map<Long, Long> adGroupIdByBannerId = newBannerService.getAdGroupIdsByBannerIds(clientId,
                campaignIdByBannerId.keySet());

        Set<Long> adGroupIds = Set.copyOf(adGroupIdByBannerId.values());

        List<GdAdGroupTruncated> adGroups = groupDataService.getTruncatedAdGroups(shard,
                client.getCountryRegionId(), clientId, operator, adGroupIds, campaignsById);

        List<GdAd> ads = getBannersByIdsWithoutStats(shard, clientId, listToMap(adGroups, GdAdGroupTruncated::getId),
                campaignIdByBannerId.keySet(), true);

        //Из базы берем по аггрегированным "cids, types", потому можем получить больше, чем просили.
        return toGdLastChangedAds(filterList(ads, gdAd -> input.getItems().contains(new GdLastChangedAdContainerItem()
                .withAdType(gdAd.getType())
                .withCampaignId(gdAd.getCampaignId()))));
    }

    @Nonnull
    GdUpdateAdsPayload updateAdAgeFlags(ClientId clientId, User operator, GdUpdateAdAgeFlags input) {
        adValidationService.validateUpdateAdAgeFlags(operator, input);
        GdiAdMassChangeResult result = gridAdsMassChangesService.updateAdAgeFlags(operator.getUid(),
                clientId,
                input.getAdIds(),
                toBannerAgeFlag(input.getAdAgeValue()),
                toBannerBabyFoodFlag(input.getAdBabyFoodValue()));
        return massChangeResultToGdUpdateAdsPayload(result);
    }

    private List<GdiBanner> filterByAggregatedStatuses(Map<Long, AggregatedStatusAdData> adStatusesByIds,
                                                       List<GdiBanner> banners,
                                                       GdiBannerFilter filter) {
        // Значения filter.getArchived(): true - показ архивных; false - не показываем архивные (для показа всех,
        // кроме архивных);  null - для показа всех, либо показа по фильтру статусов
        boolean withoutArchived = Boolean.FALSE.equals(filter.getArchived());
        Set<GdiBannerPrimaryStatus> statuses = new HashSet<>(emptyIfNull(filter.getPrimaryStatusContains()));
        if (Boolean.TRUE.equals(filter.getArchived())) {
            statuses.add(GdiBannerPrimaryStatus.ARCHIVED);
        }
        return banners.stream()
                .filter(b -> statuses.isEmpty()
                        || statuses.contains(AdPrimaryStatusCalculator
                        .convertToPrimaryStatus(adStatusesByIds.get(b.getId()))))
                .filter(b -> !withoutArchived
                        || !adStatusesByIds.containsKey(b.getId())
                        || adStatusesByIds.get(b.getId()).getStatus().isEmpty()
                        || !adStatusesByIds.get(b.getId()).getStatus().get().equals(GdSelfStatusEnum.ARCHIVED))
                .filter(b -> ReasonsFilterUtils.isValid(filter.getReasonsContainSome(), adStatusesByIds.get(b.getId())))
                .collect(Collectors.toList());
    }

    public Supplier<Set<Long>> allowedReasonsToSelfRemoderateSupplier() {
        return () -> ppcPropertiesSupport
                .get(MODERATION_REASONS_ALLOWABLE_TO_REMODERATE_BY_CLIENT)
                .getOrDefault(emptySet());
    }

    @Nullable
    public String getPreviousAdUrl(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var campaignIds = campaignRepository
                .getLatestCampaignIds(shard, clientId, Set.of(CampaignType.TEXT), LATEST_CAMPAIGNS_LOOKUP_COUNT);
        if (campaignIds.isEmpty()) {
            return null;
        }
        var suggest = bannerCommonRepository.getLatestBannerUrlForCampaigns(shard, campaignIds);
        LOGGER.info("PreviousAdUrl suggest response: {}", suggest);
        return suggest;
    }
}
