package ru.yandex.direct.grid.processing.service.banner

import com.google.common.base.Preconditions.checkState
import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLContext
import io.leangen.graphql.annotations.GraphQLMutation
import io.leangen.graphql.annotations.GraphQLNonNull
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import ru.yandex.direct.core.entity.banner.model.InternalBanner
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChange
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldState
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateValue
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannersAggregatedState
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannersAggregatedStatePayload
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannersMassUpdate
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannersMassUpdatePayload
import ru.yandex.direct.grid.processing.model.client.GdClient
import ru.yandex.direct.grid.processing.service.banner.internalad.InternalBannerMassUpdateFieldSupport
import ru.yandex.direct.grid.processing.service.banner.internalad.getSingleValueFromUnion
import ru.yandex.direct.grid.processing.service.banner.validation.BannerInternalMassUpdateValidationService
import ru.yandex.direct.grid.processing.service.internalad.MassUpdateFieldSupport
import ru.yandex.direct.grid.processing.service.internalad.groupObjectsByValue
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.result.MassResult

@GridGraphQLService
open class BannerInternalMassUpdateGraphQlService(
    private val bannerService: BannerService,
    private val bannersUpdateOperationFactory: BannersUpdateOperationFactory,
    private val validationService: BannerInternalMassUpdateValidationService,
    private val massUpdateFieldSupports: List<InternalBannerMassUpdateFieldSupport<*>>,
    private val internalAdsProductService: InternalAdsProductService,
) {

    @PreAuthorizeRead
    @GraphQLQuery(name = "internalBannersAggregatedState")
    open fun internalBannersAggregatedState(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLContext client: GdClient,
        @GraphQLArgument(name = "input") input: GdInternalBannersAggregatedState
    ): GdInternalBannersAggregatedStatePayload {
        val clientInfo = context.queriedClient ?: throw IllegalStateException("No queriedClient in graphql context")
        val clientId = ClientId.fromLong(clientInfo.id)
        val operatorUid = context.operator.uid
        checkState(
            internalAdsProductService.clientCanHaveInternalAdCampaigns(clientId),
            "Internal ads are not supported for client"
        )
        validationService.validateInternalBannersAggregatedState(input, clientId, operatorUid)

        val internalBanners = loadInternalBanners(clientId, operatorUid, input.bannerIds)

        // сбор агрегированного состояния
        return GdInternalBannersAggregatedStatePayload().apply {
            bannerIds = internalBanners.map { it.id }
            aggregatedState = massUpdateFieldSupports.flatMap {
                aggregateBannerFieldState(internalBanners, it)
            }
        }
    }

    private fun <V, G : GdInternalBannerFieldStateValue> aggregateBannerFieldState(
        objects: Collection<InternalBanner>,
        fieldSupport: MassUpdateFieldSupport<InternalBanner, V, G, *>
    ): List<GdInternalBannerFieldState> {
        return groupObjectsByValue(objects, fieldSupport.operations::extract)
            .map { valueGroup ->
                GdInternalBannerFieldState().apply {
                    ids = valueGroup.objects.map { it.id }
                    value = fieldSupport.convertToGdFieldState(valueGroup.value)
                }
            }
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "internalBannersMassUpdate")
    open fun internalBannersMassUpdate(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: @GraphQLNonNull GdInternalBannersMassUpdate
    ): GdInternalBannersMassUpdatePayload {
        val clientId = context.subjectUser?.clientId
            ?: throw IllegalStateException("No subjectUser in graphql context")
        val subjectUid = context.subjectUser?.uid
            ?: throw IllegalStateException("No subjectUser in graphql context")
        val operatorUid = context.operator.uid

        val owner = UidAndClientId.of(subjectUid, clientId)
        checkState(
            internalAdsProductService.clientCanHaveInternalAdCampaigns(owner.clientId),
            "Internal ads are not supported for client"
        )
        validationService.validateInternalBannersMassUpdate(input, clientId, operatorUid)

        val affectedBannerIds = input.changes.asSequence().flatMap { it.ids }.toSet()
        val internalBannerIndex = loadInternalBanners(clientId, operatorUid, affectedBannerIds)
            .associateBy { it.id }

        input.changes.forEach { change ->
            applySingleChange(change, massUpdateFieldSupports, internalBannerIndex)
        }

        val modelChanges: List<ModelChanges<InternalBanner>> = internalBannerIndex.values.map {
            ModelChanges(it.id, InternalBanner::class.java)
                .process(it.moderationInfo, InternalBanner.MODERATION_INFO)
                .process(it.statusShow, InternalBanner.STATUS_SHOW)
                .process(it.templateVariables, InternalBanner.TEMPLATE_VARIABLES)
        }

        val result: MassResult<Long> = bannersUpdateOperationFactory.createPartialUpdateOperation(
            modelChanges,
            operatorUid,
            clientId,
            InternalBanner::class.java
        ).prepareAndApply()

        return GdInternalBannersMassUpdatePayload().withValidationResult(
            validationService.getValidationResult(result.validationResult)
        )
    }

    private fun applySingleChange(
        change: GdInternalBannerFieldChange,
        massUpdateFieldSupports: List<InternalBannerMassUpdateFieldSupport<*>>,
        internalBannerIndex: Map<Long, InternalBanner>
    ) {
        val value = change.value.getSingleValueFromUnion()
        val banners = change.ids.mapNotNull { internalBannerIndex[it] }
        massUpdateFieldSupports.forEach { it.applyAdd(banners, value) }
    }

    private fun loadInternalBanners(clientId: ClientId, operatorId: Long, ids: Collection<Long>): List<InternalBanner> {
        return bannerService.get(clientId, operatorId, ids).filterIsInstance<InternalBanner>()
    }
}
