package ru.yandex.direct.grid.processing.service.banner;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;

import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.processing.model.banner.GdAd;
import ru.yandex.direct.grid.processing.model.banner.GdAdOrderBy;
import ru.yandex.direct.grid.processing.model.banner.GdAdOrderByField;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrimaryStatus;

import static com.google.common.base.Preconditions.checkArgument;

@ParametersAreNonnullByDefault
class BannerOrderHelper {
    private static final Map<GdAdOrderByField, ComparatorWithGdAdFieldExtractor<?>>
            COMPARATOR_WITH_VALUE_EXTRACTOR_BY_FIELD =
            ImmutableMap.<GdAdOrderByField, ComparatorWithGdAdFieldExtractor<?>>builder()
                    .put(GdAdOrderByField.PRIMARY_STATUS,
                            new ComparatorWithGdAdFieldExtractor<>(
                                    banner -> banner.getStatus().getPrimaryStatus(),
                                    Comparator.comparing(GdAdPrimaryStatus::getTypedValue)))
                    .build();


    static boolean isOrderingOccursAfterDataFetching(GdAdOrderBy ordersBy) {
        return COMPARATOR_WITH_VALUE_EXTRACTOR_BY_FIELD.containsKey(ordersBy.getField());
    }

    /**
     * Создает компоратор по указанным полям для сортировки и направлению сортировки
     */
    static Comparator<GdAd> createOrderComparator(List<GdAdOrderBy> ordersBy) {
        boolean hasOrderingOccursAfterDataFetching =
                ordersBy.stream().allMatch(BannerOrderHelper::isOrderingOccursAfterDataFetching);
        checkArgument(hasOrderingOccursAfterDataFetching, "field is not supported");

        Optional<Comparator<GdAd>> gdAdComparator = StreamEx.of(ordersBy)
                .mapToEntry(BannerOrderHelper::getGdAdComparator)
                .mapKeys(order -> COMPARATOR_WITH_VALUE_EXTRACTOR_BY_FIELD.get(order.getField()).getValueExtractor())
                .mapKeyValue(Comparator::comparing)
                .foldLeft(Comparator::thenComparing);

        //noinspection ConstantConditions
        return gdAdComparator.get();
    }

    private static Comparator<Object> getGdAdComparator(GdAdOrderBy order) {
        ComparatorWithGdAdFieldExtractor comparatorWithGdAdFieldExtractor =
                COMPARATOR_WITH_VALUE_EXTRACTOR_BY_FIELD.get(order.getField());
        //noinspection unchecked
        Comparator<Object> comparator = comparatorWithGdAdFieldExtractor.getComparator();
        return order.getOrder().equals(Order.ASC) ? comparator : comparator.reversed();
    }

    private static class ComparatorWithGdAdFieldExtractor<T> {
        private final Function<GdAd, T> valueExtractor;
        private final Comparator<T> comparator;

        ComparatorWithGdAdFieldExtractor(Function<GdAd, T> valueExtractor, Comparator<T> comparator) {
            this.valueExtractor = valueExtractor;
            this.comparator = comparator;
        }

        Function<GdAd, T> getValueExtractor() {
            return valueExtractor;
        }

        public Comparator<T> getComparator() {
            return comparator;
        }
    }
}
