package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.service.GridFindAndReplaceBannerHrefService;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefDomain;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefPayloadItem;
import ru.yandex.direct.grid.processing.model.common.GdCachedResult;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.service.banner.container.FindAndReplaceBannerHrefDomainCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.util.findandreplace.BannerHrefDomainReplaceRule;

import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerHrefConverter.getHrefDomainCacheRecordInfo;
import static ru.yandex.direct.validation.result.PathHelper.pathFromStrings;

/**
 * Сервис для массового поиска и замены доменов баннеров
 */
@Service
@ParametersAreNonnullByDefault
public class FindAndReplaceBannerHrefDomainService extends AbstractFindAndReplaceBannerHrefService {
    private final GridValidationService inputValidator;
    private final GridFindAndReplaceBannerHrefService hrefService;
    private final BannersUrlHelper bannersUrlHelper;

    @Autowired
    public FindAndReplaceBannerHrefDomainService(
            GridCacheService cacheService,
            GridFindAndReplaceBannerHrefService hrefService,
            GridValidationService inputValidator, GridValidationResultConversionService validationResultConverter,
            BannersUrlHelper bannersUrlHelper) {
        super(cacheService, hrefService, validationResultConverter);
        this.inputValidator = inputValidator;
        this.hrefService = hrefService;
        this.bannersUrlHelper = bannersUrlHelper;
    }

    /**
     * Получить все домены баннеров и соответствующих сайтлинков
     */
    public Set<String> getDomains(List<Long> bannerIds, ClientId clientId, String rootName) {
        inputValidator.validateAdIds(bannerIds, pathFromStrings(rootName));
        return hrefService.getDomains(bannerIds, clientId);
    }

    /**
     * Получить превью для Поиска и замены домена кликовой ссылки баннера
     */
    public GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem> preview(
            GdFindAndReplaceAdsHrefDomain input,
            Long operatorUid, ClientId clientId,
            String rootName) {
        inputValidator.validateFindAndReplaceBannerHrefDomainRequest(input, pathFromStrings(rootName));
        ReplaceRule replaceRule = new BannerHrefDomainReplaceRule(input.getReplaceInstruction(), bannersUrlHelper);
        FindAndReplaceBannerHrefDomainCacheRecordInfo cacheRecordInfo = getHrefDomainCacheRecordInfo(clientId, input);

        return preview(input, operatorUid, clientId, replaceRule, replaceRule, cacheRecordInfo);
    }

    /**
     * Поиск и замена домена ключевой ссылки баннера
     */
    public GdResult<GdFindAndReplaceAdsHrefPayloadItem> replace(
            GdFindAndReplaceAdsHrefDomain input,
            Long operatorUid, ClientId clientId,
            String rootName) {
        inputValidator.validateFindAndReplaceBannerHrefDomainRequest(input, pathFromStrings(rootName));
        ReplaceRule replaceRule = new BannerHrefDomainReplaceRule(input.getReplaceInstruction(), bannersUrlHelper);

        return replace(input, operatorUid, clientId, replaceRule, replaceRule);
    }
}
