package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdReplaceDisplayHrefBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiReplaceDisplayHrefBanner;
import ru.yandex.direct.grid.core.entity.banner.service.GridFindAndReplaceBannerDisplayHrefService;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsDisplayHref;
import ru.yandex.direct.grid.processing.model.common.GdCachedResult;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.model.common.GdiOperationResult;
import ru.yandex.direct.grid.processing.service.banner.container.FindAndReplaceBannerDisplayHrefCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerDisplayHrefConverter.convertGdiToGd;
import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerDisplayHrefConverter.createPayload;
import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerDisplayHrefConverter.getCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerDisplayHrefConverter.preparePreviewPayload;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.util.OperationResultSorter.sortByValidationResult;
import static ru.yandex.direct.validation.result.PathHelper.pathFromStrings;

@Service
public class FindAndReplaceBannersDisplayHrefService {

    private final GridCacheService cacheService;
    private final GridValidationService inputValidator;
    private final GridFindAndReplaceBannerDisplayHrefService hrefService;
    private final GridValidationResultConversionService validationResultConverter;

    @Autowired
    public FindAndReplaceBannersDisplayHrefService(GridCacheService cacheService,
                                                   GridValidationService inputValidator,
                                                   GridFindAndReplaceBannerDisplayHrefService hrefService,
                                                   GridValidationResultConversionService validationResultConverter) {
        this.cacheService = cacheService;
        this.inputValidator = inputValidator;
        this.hrefService = hrefService;
        this.validationResultConverter = validationResultConverter;
    }

    public Set<String> getDisplayHrefs(List<Long> bannerIds, ClientId clientId, String rootName) {
        inputValidator.validateAdIds(bannerIds, pathFromStrings(rootName));
        return hrefService.getDisplayHrefs(bannerIds, clientId);
    }

    /**
     * Получить превью для замены отображаемой ссылки баннера
     */
    public GdCachedResult<GdReplaceDisplayHrefBanner> preview(
            GdFindAndReplaceAdsDisplayHref input,
            Long operatorUid, ClientId clientId,
            String rootName) {
        inputValidator.validateFindAndReplaceBannerDisplayHrefRequest(input, pathFromStrings(rootName));

        FindAndReplaceBannerDisplayHrefCacheRecordInfo cacheRecordInfo = getCacheRecordInfo(clientId, input);
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        Optional<GdCachedResult<GdReplaceDisplayHrefBanner>> res = cacheService.getFromCache(cacheRecordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        List<GdiReplaceDisplayHrefBanner> banners = hrefService
                .getBanners(clientId, input.getAdIds(), input.getSearchDisplayHrefs(), input.getNewDisplayHref());

        GridBannerUpdateInfo updateInfo = hrefService.preview(banners, operatorUid, clientId);

        GdiOperationResult<GdiReplaceDisplayHrefBanner, BannerWithSystemFields> sortedOperationResult =
                sortByValidationResult(new GdiOperationResult<>(banners, updateInfo.getValidationResult()));

        GdValidationResult validationResult =
                validationResultConverter.buildGridValidationResult(sortedOperationResult.getValidationResult());

        List<GdReplaceDisplayHrefBanner> rowSetFull = convertGdiToGd(sortedOperationResult.getRowset());
        GdCachedResult<GdReplaceDisplayHrefBanner> payload = preparePreviewPayload(rowSetFull.size(), validationResult);
        return cacheService.getResultAndSaveToCache(cacheRecordInfo, payload, rowSetFull, range);
    }

    /**
     * Заменить отображаемые ссылки баннера
     */
    public GdResult<GdReplaceDisplayHrefBanner> replace(
            GdFindAndReplaceAdsDisplayHref input,
            Long operatorUid, ClientId clientId,
            String rootName) {
        inputValidator.validateFindAndReplaceBannerDisplayHrefRequest(input, pathFromStrings(rootName));

        List<GdiReplaceDisplayHrefBanner> banners = hrefService
                .getBanners(clientId, input.getAdIds(), input.getSearchDisplayHrefs(), input.getNewDisplayHref());

        GridBannerUpdateInfo updateInfo = hrefService.replace(banners, operatorUid, clientId);

        GdiOperationResult<GdiReplaceDisplayHrefBanner, BannerWithSystemFields> sortedOperationResult =
                sortByValidationResult(new GdiOperationResult<>(banners, updateInfo.getValidationResult()));

        GdValidationResult validationResult =
                validationResultConverter.buildGridValidationResult(sortedOperationResult.getValidationResult());

        List<GdReplaceDisplayHrefBanner> rowSetFull = convertGdiToGd(sortedOperationResult.getRowset());
        return createPayload(rowSetFull, validationResult, updateInfo.getUpdatedBannerCount());
    }


}
