package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerCalloutsItem;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerCalloutsReplaceInstruction;
import ru.yandex.direct.grid.core.entity.banner.service.GridFindAndReplaceCalloutsService;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceCallouts;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceCalloutsPayloadItem;
import ru.yandex.direct.grid.processing.model.common.GdCachedResult;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.model.common.GdiOperationResult;
import ru.yandex.direct.grid.processing.service.banner.container.FindAndReplaceBannerCalloutsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerCalloutsConverter;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerCalloutsConverter.getCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerCalloutsConverter.toGdCachedResult;
import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerCalloutsConverter.toGdResult;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.util.OperationResultSorter.sortByValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.pathFromStrings;

/**
 * Сервис для массового поиска и замены уточнений баннеров
 */
@Service
@ParametersAreNonnullByDefault
public class FindAndReplaceCalloutsService {
    private final GridFindAndReplaceCalloutsService calloutsService;
    private final GridCacheService cacheService;
    private final GridValidationService inputValidator;
    private final GridValidationResultConversionService validationResultConverter;

    @Autowired
    public FindAndReplaceCalloutsService(GridFindAndReplaceCalloutsService calloutsService,
                                         GridCacheService cacheService,
                                         GridValidationService inputValidator,
                                         GridValidationResultConversionService validationResultConverter) {
        this.calloutsService = calloutsService;
        this.cacheService = cacheService;
        this.inputValidator = inputValidator;
        this.validationResultConverter = validationResultConverter;
    }

    public GdResult<GdFindAndReplaceCalloutsPayloadItem> replace(
            GdFindAndReplaceCallouts input, Long operatorUid,
            ClientId clientId,
            String rootName) {

        inputValidator.validateFindAndReplaceBannerCalloutsRequest(input, pathFromStrings(rootName));

        GdiFindAndReplaceBannerCalloutsReplaceInstruction replaceInstruction =
                FindAndReplaceBannerCalloutsConverter.convertReplaceInstruction(input);

        List<GdiFindAndReplaceBannerCalloutsItem> banners = calloutsService.getBanners(clientId, input.getAdIds(),
                replaceInstruction);

        GridBannerUpdateInfo updateInfo = calloutsService.update(operatorUid, clientId, banners);

        var sortedOperationResult = sortByValidationResult(new GdiOperationResult<>(banners,
                updateInfo.getValidationResult()));
        var validationResult =
                validationResultConverter.buildGridValidationResult(sortedOperationResult.getValidationResult());
        var rowSetFull = mapList(sortedOperationResult.getRowset(),
                FindAndReplaceBannerCalloutsConverter::convertToPayloadItem);

        return toGdResult(rowSetFull, validationResult, updateInfo);
    }

    public GdCachedResult<GdFindAndReplaceCalloutsPayloadItem> preview(
            GdFindAndReplaceCallouts input, Long operatorUid,
            ClientId clientId,
            String rootName) {

        inputValidator.validateFindAndReplaceBannerCalloutsRequest(input, pathFromStrings(rootName));

        FindAndReplaceBannerCalloutsCacheRecordInfo cacheRecordInfo = getCacheRecordInfo(clientId, input);
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        Optional<GdCachedResult<GdFindAndReplaceCalloutsPayloadItem>> res = cacheService.getFromCache(
                cacheRecordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        GdiFindAndReplaceBannerCalloutsReplaceInstruction replaceInstruction =
                FindAndReplaceBannerCalloutsConverter.convertReplaceInstruction(input);

        List<GdiFindAndReplaceBannerCalloutsItem> banners = calloutsService.getBanners(clientId, input.getAdIds(),
                replaceInstruction);

        GridBannerUpdateInfo updateInfo = calloutsService.preview(operatorUid, clientId, banners);

        GdiOperationResult<GdiFindAndReplaceBannerCalloutsItem, BannerWithSystemFields> sortedOperationResult =
                sortByValidationResult(new GdiOperationResult<>(banners, updateInfo.getValidationResult()));

        GdValidationResult validationResult =
                validationResultConverter.buildGridValidationResult(sortedOperationResult.getValidationResult());

        List<GdFindAndReplaceCalloutsPayloadItem> rowSetFull = mapList(sortedOperationResult.getRowset(),
                FindAndReplaceBannerCalloutsConverter::convertToPayloadItem);
        GdCachedResult<GdFindAndReplaceCalloutsPayloadItem> payload = toGdCachedResult(rowSetFull.size(),
                validationResult, updateInfo);
        return cacheService.getResultAndSaveToCache(cacheRecordInfo, payload, rowSetFull, range);
    }
}
