package ru.yandex.direct.grid.processing.service.banner;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.LogHelper;
import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerTextItem;
import ru.yandex.direct.grid.core.entity.banner.service.GridFindAndReplaceBannerTextService;
import ru.yandex.direct.grid.processing.model.banner.mutation.FindAndReplaceTextCacheFilterData;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceLogData;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceText;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceTextPayloadItem;
import ru.yandex.direct.grid.processing.model.common.GdCachedResult;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.model.common.GdiOperationResult;
import ru.yandex.direct.grid.processing.service.banner.container.FindAndReplaceBannerTextCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerTextConverter;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;

import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerTextConverter.toGdCachedResult;
import static ru.yandex.direct.grid.processing.service.banner.converter.FindAndReplaceBannerTextConverter.toGdResult;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.util.OperationResultSorter.sortByValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.toJson;
import static ru.yandex.direct.validation.result.PathHelper.pathFromStrings;

@Service
@ParametersAreNonnullByDefault
public class FindAndReplaceTextService {
    private static final Logger LOGGER = LoggerFactory.getLogger(FindAndReplaceTextService.class.getName());

    private final GridCacheService cacheService;
    private final GridValidationService inputValidator;
    private final GridFindAndReplaceBannerTextService textService;
    private final GridValidationResultConversionService validationResultConverter;

    private final LogHelper logHelper = new LogHelper(LogType.UNKNOWN);
    private final FindAndReplaceBannerTextConverter findAndReplaceBannerTextConverter;

    @Autowired
    public FindAndReplaceTextService(GridCacheService cacheService, GridValidationService inputValidator,
                                     GridFindAndReplaceBannerTextService textService,
                                     GridValidationResultConversionService validationResultConverter,
                                     FindAndReplaceBannerTextConverter findAndReplaceBannerTextConverter) {
        this.cacheService = cacheService;
        this.inputValidator = inputValidator;
        this.textService = textService;
        this.validationResultConverter = validationResultConverter;
        this.findAndReplaceBannerTextConverter = findAndReplaceBannerTextConverter;
    }

    public GdCachedResult<GdFindAndReplaceTextPayloadItem> preview(GdFindAndReplaceText input,
                                                                   Long operatorUid, ClientId clientId, String rootName) {
        inputValidator.validateFindAndReplaceBannerTextRequest(input, pathFromStrings(rootName));

        var cacheRecordInfo = new FindAndReplaceBannerTextCacheRecordInfo(
                clientId.asLong(), input.getCacheKey(),
                new FindAndReplaceTextCacheFilterData()
                        .withAdIds(input.getAdIds())
                        .withTargetTypes(input.getTargetTypes())
                        .withReplaceInstruction(input.getReplaceInstruction())
        );
        var range = normalizeLimitOffset(input.getLimitOffset());
        var res = cacheService.getFromCache(cacheRecordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        var banners = getBanners(clientId, input);
        var updateInfo = textService.updateBannersAndSitelinksTextPreview(banners, operatorUid, clientId);
        var sortedOperationResult = sortByValidationResult(new GdiOperationResult<>(banners, updateInfo.getValidationResult()));
        var validationResult = validationResultConverter.buildGridValidationResult(sortedOperationResult.getValidationResult());
        var rowSetFull = mapList(sortedOperationResult.getRowset(), FindAndReplaceBannerTextConverter::convertToPayloadItem);

        var result = toGdCachedResult(rowSetFull, validationResult, updateInfo);

        return cacheService.getResultAndSaveToCache(cacheRecordInfo, result, rowSetFull, range);
    }

    public GdResult<GdFindAndReplaceTextPayloadItem> replace(GdFindAndReplaceText input,
                                                             Long operatorUid, ClientId clientId, String rootName) {
        inputValidator.validateFindAndReplaceBannerTextRequest(input, pathFromStrings(rootName));

        var banners = getBanners(clientId, input);
        var updateInfo = textService.updateBannerAndSitelinksText(banners, operatorUid, clientId);
        var sortedOperationResult = sortByValidationResult(new GdiOperationResult<>(banners, updateInfo.getValidationResult()));
        var validationResult =
                validationResultConverter.buildGridValidationResult(sortedOperationResult.getValidationResult());
        var rowSetFull = mapList(sortedOperationResult.getRowset(), FindAndReplaceBannerTextConverter::convertToPayloadItem);

        var result = toGdResult(rowSetFull, validationResult, updateInfo);
        logRequestData(input, result);
        return result;
    }

    private List<GdiFindAndReplaceBannerTextItem> getBanners(ClientId clientId, GdFindAndReplaceText input) {
        return textService.getBanners(clientId, input.getAdIds(), findAndReplaceBannerTextConverter.toReplaceRequest(input));
    }

    private void logRequestData(GdFindAndReplaceText input, GdResult<GdFindAndReplaceTextPayloadItem> result) {
        var logData = new GdFindAndReplaceLogData()
                .withRequestSize(input.getAdIds().size())
                .withTotal(result.getTotalCount())
                .withSuccess(result.getSuccessCount());
        var logEntry = logHelper.getLogEntry(logData);
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info(toJson(logEntry));
        }
    }
}
