package ru.yandex.direct.grid.processing.service.banner.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.Age;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurer;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurerSystem;
import ru.yandex.direct.core.entity.banner.model.BannerMulticard;
import ru.yandex.direct.core.entity.banner.model.BannerMulticardsCurrencyValues;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerPricesCurrency;
import ru.yandex.direct.core.entity.banner.model.BannerPricesPrefix;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLandingParams;
import ru.yandex.direct.core.entity.banner.model.ButtonAction;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmAudioBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.CpmGeoPinBanner;
import ru.yandex.direct.core.entity.banner.model.CpmIndoorBanner;
import ru.yandex.direct.core.entity.banner.model.CpmOutdoorBanner;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.McBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.NewMobileContentPrimaryAction;
import ru.yandex.direct.core.entity.banner.model.NewReflectedAttribute;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.grid.processing.model.banner.GdAdPrice;
import ru.yandex.direct.grid.processing.model.banner.GdBannerAdditionalHref;
import ru.yandex.direct.grid.processing.model.banner.GdBannerButton;
import ru.yandex.direct.grid.processing.model.banner.GdBannerMeasurer;
import ru.yandex.direct.grid.processing.model.banner.GdBannerMeasurerSystem;
import ru.yandex.direct.grid.processing.model.banner.GdMobileContentAdAction;
import ru.yandex.direct.grid.processing.model.banner.GdMobileContentAdAgeLabel;
import ru.yandex.direct.grid.processing.model.banner.GdMobileContentAdFeature;
import ru.yandex.direct.grid.processing.model.banner.mutation.BannerMulticardsCurrency;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateContentPromotionAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateCpmAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateDynamicAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMcBannerAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMobileContentAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateMulticard;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateSmartAd;
import ru.yandex.direct.grid.processing.model.cliententity.GdPixel;
import ru.yandex.direct.grid.processing.model.constants.GdButtonAction;
import ru.yandex.direct.grid.processing.service.banner.BannerDataConverter;

import static java.lang.Boolean.TRUE;
import static java.lang.String.format;
import static java.math.RoundingMode.HALF_UP;
import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.core.entity.domain.DomainUtils.refineDomain;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.StringUtils.ifNotBlank;
import static ru.yandex.direct.utils.StringUtils.nullIfBlank;

@ParametersAreNonnullByDefault
public class AdMutationDataConverter {

    public static List<BannerWithSystemFields> toCoreBanners(List<GdUpdateAd> ads) {
        return mapList(ads, AdMutationDataConverter::toCoreBanner);
    }

    private static BannerWithSystemFields toCoreBanner(@Nullable GdUpdateAd ad) {
        if (ad == null) {
            return null;
        }
        BannerWithSystemFields banner;
        BannersBannerType internalBannerType = BannerDataConverter.toInternalAdType(ad.getAdType());
        if (BannersBannerType.image_ad.equals(internalBannerType)) {

            if (ad.getCreativeId() == null) {
                banner = toImageHashBanner(ad);
            } else {
                banner = toImageCreativeBanner(ad);
            }
        } else if (BannersBannerType.text.equals(internalBannerType)) {
            banner = toTextBanner(ad);
        } else if (BannersBannerType.cpc_video.equals(internalBannerType)) {
            banner = toCpcVideoBanner(ad);
        } else {
            throw new UnsupportedOperationException(format("Ads of type %s are not supported.", ad.getAdType()));
        }

        //noinspection ConstantConditions
        if (banner instanceof BannerWithTurboLanding) {
            ((BannerWithTurboLanding) banner)
                    .withTurboLandingId(ad.getTurbolandingId());
        }

        //noinspection ConstantConditions
        if (banner instanceof BannerWithTurboLandingParams) {
            ((BannerWithTurboLandingParams) banner)
                    .withTurboLandingHrefParams(ad.getTurbolandingHrefParams());
        }

        //noinspection ConstantConditions
        if (banner instanceof BannerWithHref) {
            ((BannerWithHref) banner)
                    .withHref(ad.getHref())
                    .withDomain(refineDomain(ad.getDomain()));
        }

        return banner
                .withId(ad.getId());
    }

    private static CpcVideoBanner toCpcVideoBanner(GdUpdateAd ad) {
        return new CpcVideoBanner()
                .withIsMobileVideo(nvl(ad.getIsMobile(), false))
                .withCreativeId(ad.getCreativeId());
    }

    private static TextBanner toTextBanner(GdUpdateAd ad) {
        return new TextBanner()
                .withSitelinksSetId(ad.getSitelinksSetId())
                .withVcardId(ad.getVcardId())
                .withIsMobile(ad.getIsMobile())
                .withTitle(ad.getTitle())
                .withTitleExtension(ad.getTitleExtension())
                .withBody(ad.getBody())
                .withDisplayHref(ad.getDisplayHref())
                .withImageHash(ifNotBlank(ad.getTextBannerImageHash(), Function.identity()))
                .withCalloutIds(ad.getCalloutIds())
                .withCreativeId(ad.getCreativeId())
                .withShowTitleAndBody(ad.getShowTitleAndBody())
                .withBannerPrice(toCoreBannerPrice(ad.getAdPrice()))
                .withTurboGalleryHref(ad.getTurboGalleryParams() == null ?
                        null : ad.getTurboGalleryParams().getTurboGalleryHref())
                .withPermalinkId(ad.getPermalinkId())
                .withPhoneId(ad.getPhoneId())
                .withLogoImageHash(ad.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(ad.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(ad.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(ad.getButton(), GdBannerButton::getHref))
                .withPreferVCardOverPermalink(ad.getPreferVCardOverPermalink())
                .withName(ad.getName())
                .withMulticards(mapList(ad.getMulticards(), AdMutationDataConverter::toCoreBannerMulticard));
    }

    public static DynamicBanner toDynamicBanner(GdUpdateDynamicAd ad) {
        return new DynamicBanner()
                .withId(ad.getId())
                .withSitelinksSetId(ad.getSitelinksSetId())
                .withVcardId(ad.getVcardId())
                .withBody(ad.getBody())
                .withImageHash(ifNotBlank(ad.getBannerImageHash(), Function.identity()))
                .withCalloutIds(ad.getCalloutIds());
    }

    private static ImageBanner toImageCreativeBanner(GdUpdateAd ad) {
        return new ImageBanner()
                .withIsMobileImage(nvl(ad.getIsMobile(), false))
                .withCreativeId(ad.getCreativeId())
                .withTitle(nullIfBlank(ad.getTitle()))
                .withTitleExtension(ad.getTitleExtension())
                .withBody(nullIfBlank(ad.getBody()))
                .withLogoImageHash(ad.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(ad.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(ad.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(ad.getButton(), GdBannerButton::getHref));
    }

    private static ImageBanner toImageHashBanner(GdUpdateAd ad) {
        return new ImageBanner()
                .withIsMobileImage((nvl(ad.getIsMobile(), false)))
                .withImageHash(ad.getImageCreativeHash())
                .withTitle(nullIfBlank(ad.getTitle()))
                .withTitleExtension(ad.getTitleExtension())
                .withBody(nullIfBlank(ad.getBody()))
                .withLogoImageHash(ad.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(ad.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(ad.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(ad.getButton(), GdBannerButton::getHref));
    }

    public static BannerWithSystemFields toPerformanceBanner(GdUpdateSmartAd ad) {
        if (ad.getCreativeId() != null) {
            return new PerformanceBanner()
                    .withId(ad.getId())
                    .withCreativeId(ad.getCreativeId());
        } else {
            return new PerformanceBannerMain()
                    .withId(ad.getId())
                    .withLogoImageHash(ad.getLogoImageHash());
        }
    }

    public static BannerWithSystemFields toCoreBanner(GdUpdateCpmAd ad) {
        BannersBannerType internalBannerType = BannerDataConverter.toInternalAdType(ad.getAdType());
        if (internalBannerType == BannersBannerType.cpm_banner) {
            return toCpmBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_audio) {
            return toCpmAudioBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_outdoor) {
            return toCpmOutdoorBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_indoor) {
            return toCpmIndoorBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_geo_pin) {
            return toCpmGeoPinBanner(ad);
        } else {
            throw new UnsupportedOperationException(format("Ads of type %s are not supported.", ad.getAdType()));
        }
    }

    private static CpmBanner toCpmBanner(GdUpdateCpmAd ad) {
        return new CpmBanner()
                .withId(ad.getId())
                .withTitle(ad.getTitle())
                .withTitleExtension(ad.getTitleExtension())
                .withBody(ad.getBody())
                .withLogoImageHash(ad.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(ad.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(ad.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(ad.getButton(), GdBannerButton::getHref))
                .withCreativeId(ad.getCreativeId())
                .withHref(ad.getHref())
                .withTurboLandingId(ad.getTurbolandingId())
                .withTurboLandingHrefParams(ad.getTurbolandingHrefParams())
                .withPixels(mapList(nvl(ad.getPixels(), emptyList()), GdPixel::getUrl))
                .withMeasurers(mapList(nvl(ad.getMeasurers(), emptyList()), m -> toBannerMeasurer(ad, m)))
                .withTnsId(ad.getTnsId())
                .withMulticards(mapList(ad.getMulticards(), AdMutationDataConverter::toCoreBannerMulticard))
                .withBigKingImageHash(ad.getBigKingImageHash())
                .withAdditionalHrefs(toBannerAdditionalHrefs(ad.getAdditionalHrefs()));
    }

    private static CpmAudioBanner toCpmAudioBanner(GdUpdateCpmAd ad) {
        return new CpmAudioBanner()
                .withId(ad.getId())
                .withCreativeId(ad.getCreativeId())
                .withHref(ad.getHref())
                .withPixels(mapList(nvl(ad.getPixels(), emptyList()), GdPixel::getUrl))
                .withMeasurers(mapList(nvl(ad.getMeasurers(), emptyList()), m -> toBannerMeasurer(ad, m)));
    }

    private static BannerMeasurer toBannerMeasurer(GdUpdateCpmAd ad, GdBannerMeasurer gdBannerMeasurer) {
        // OMI всегда считаем сынтегрированными, но не озадачиваем логикой фронт. Подменяем на входе.
        Boolean hasIntegration = gdBannerMeasurer.getMeasurerSystem() == GdBannerMeasurerSystem.OMI || gdBannerMeasurer
                .getHasIntegration();
        return new BannerMeasurer()
                .withBannerMeasurerSystem(BannerMeasurerSystem.valueOf(gdBannerMeasurer.getMeasurerSystem().name()))
                .withParams(gdBannerMeasurer.getParams())
                .withHasIntegration(hasIntegration);
    }

    private static CpmOutdoorBanner toCpmOutdoorBanner(GdUpdateCpmAd ad) {
        return new CpmOutdoorBanner()
                .withId(ad.getId())
                .withCreativeId(ad.getCreativeId())
                .withHref(ad.getHref());
    }

    private static CpmIndoorBanner toCpmIndoorBanner(GdUpdateCpmAd ad) {
        return new CpmIndoorBanner()
                .withId(ad.getId())
                .withCreativeId(ad.getCreativeId())
                .withHref(ad.getHref());
    }

    private static CpmGeoPinBanner toCpmGeoPinBanner(GdUpdateCpmAd ad) {
        return new CpmGeoPinBanner()
                .withId(ad.getId())
                .withCreativeId(ad.getCreativeId())
                .withPermalinkId(ad.getPermalinkId())
                .withPixels(mapList(nvl(ad.getPixels(), emptyList()), GdPixel::getUrl))
                .withMeasurers(mapList(nvl(ad.getMeasurers(), emptyList()), m -> toBannerMeasurer(ad, m)));
    }

    public static McBanner toMcBanner(GdUpdateMcBannerAd ad) {
        return new McBanner()
                .withId(ad.getId())
                .withHref(ad.getHref())
                .withImageHash(ad.getAdImageHash());
    }

    public static MobileAppBanner toMobileAppBanner(GdUpdateMobileContentAd ad) {
        return new MobileAppBanner()
                .withId(ad.getId())
                .withImageHash(ad.getAdImageHash())
                .withBody(ad.getBody())
                .withTitle(ad.getTitle())
                .withCreativeId(ad.getCreativeId())
                .withCalloutIds(ad.getCalloutIds())
                .withHref(ad.getTrackingUrl())
                .withImpressionUrl(ad.getImpressionUrl())
                .withFlags(createBannerFlagsForAgeLabel(ad.getAgeLabel()))
                .withPrimaryAction(toNewMobileContentPrimaryAction(ad.getAction()))
                .withReflectedAttributes(toNewReflectedAttribute(ad.getFeatures()));
    }

    public static ContentPromotionBanner toContentPromotionBanner(GdUpdateContentPromotionAd ad) {
        return new ContentPromotionBanner()
                .withId(ad.getId())
                .withVisitUrl(ad.getVisitUrl())
                .withContentPromotionId(ad.getContentPromotion().getContentPromotionId())
                .withTitle(ad.getTitle())
                .withBody(ad.getDescription());
    }

    @Nullable
    public static BannerMulticard toCoreBannerMulticard(GdUpdateMulticard multicard) {
        return new BannerMulticard()
                .withMulticardId(multicard.getId())
                .withText(multicard.getText())
                .withImageHash(multicard.getImageHash())
                .withHref(multicard.getHref())
                .withPrice(multicard.getPrice())
                .withPriceOld(multicard.getPriceOld())
                .withCurrency(ru.yandex.direct.core.entity.banner.model.BannerMulticardsCurrency
                        .fromSource(Optional.ofNullable(multicard.getCurrency())
                                .map(BannerMulticardsCurrency::toSource)
                                .map(BannerMulticardsCurrencyValues::toSource)
                                .orElse(null)));
    }

    @Nullable
    public static BannerPrice toCoreBannerPrice(@Nullable GdAdPrice adPrice) {
        if (adPrice == null || isBlank(adPrice.getPrice())) {
            return null;
        }
        return new BannerPrice()
                .withPrefix(ifNotNull(adPrice.getPrefix(), p -> BannerPricesPrefix.valueOf(p.name())))
                .withCurrency(ifNotNull(adPrice.getCurrency(), c -> BannerPricesCurrency.valueOf(c.name())))
                .withPrice(webPriceToCorePrice(adPrice.getPrice()))
                .withPriceOld(ifNotBlank(adPrice.getPriceOld(), AdMutationDataConverter::webPriceToCorePrice));
    }

    public static ButtonAction toCoreButtonAction(@Nullable GdBannerButton bannerButton){
        return bannerButton != null ? GdButtonAction.toSource(bannerButton.getAction()) : null;
    }

    @Nonnull
    public static BigDecimal webPriceToCorePrice(@Nonnull String webPrice) {
        return new BigDecimal(webPrice.replaceAll(" ", "").replaceAll(",", "."))
                .setScale(2, HALF_UP);
    }

    static List<BannerAdditionalHref> toBannerAdditionalHrefs(
            @Nullable List<GdBannerAdditionalHref> gdBannerAdditionalHrefs) {
        return mapList(nvl(gdBannerAdditionalHrefs, emptyList()),
                AdMutationDataConverter::toBannerAdditionalHref);
    }

    private static BannerAdditionalHref toBannerAdditionalHref(GdBannerAdditionalHref gdBannerAdditionalHref) {
        return new BannerAdditionalHref()
                .withHref(gdBannerAdditionalHref.getHref());
    }

    static BannerFlags createBannerFlagsForAgeLabel(@Nullable GdMobileContentAdAgeLabel gdMobileContentAdAgeLabel) {
        return gdMobileContentAdAgeLabel == null ? new BannerFlags()
                : new BannerFlags().with(BannerFlags.AGE, toGdMobileContentAdAgeLabel(gdMobileContentAdAgeLabel));
    }

    public static Age toGdMobileContentAdAgeLabel(GdMobileContentAdAgeLabel gdMobileContentAdAgeLabel) {
        return Age.valueOf(gdMobileContentAdAgeLabel.name());
    }

    public static NewMobileContentPrimaryAction toNewMobileContentPrimaryAction(
            GdMobileContentAdAction primaryAction) {
        switch (primaryAction) {
            case DOWNLOAD:
                return NewMobileContentPrimaryAction.DOWNLOAD;
            case GET:
                return NewMobileContentPrimaryAction.GET;
            case INSTALL:
                return NewMobileContentPrimaryAction.INSTALL;
            case MORE:
                return NewMobileContentPrimaryAction.MORE;
            case OPEN:
                return NewMobileContentPrimaryAction.OPEN;
            case UPDATE:
                return NewMobileContentPrimaryAction.UPDATE;
            case PLAY:
                return NewMobileContentPrimaryAction.PLAY;
            case BUY_AUTODETECT:
                return NewMobileContentPrimaryAction.BUY;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + primaryAction);
        }
    }

    static Map<NewReflectedAttribute, Boolean> toNewReflectedAttribute(
            Map<GdMobileContentAdFeature, Boolean> features) {
        return StreamEx.of(GdMobileContentAdFeature.values())
                .mapToEntry(ftr -> TRUE.equals(nvl(features.get(ftr), false)))
                .mapKeys(AdMutationDataConverter::toCoreReflectedAttr)
                .toMap();
    }

    private static NewReflectedAttribute toCoreReflectedAttr(GdMobileContentAdFeature mobileContentAdFeature) {
        return NewReflectedAttribute.valueOf(mobileContentAdFeature.name());
    }
}
