package ru.yandex.direct.grid.processing.service.banner.converter;

import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.banner.model.BannerMeasurer;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurerSystem;
import ru.yandex.direct.core.entity.banner.model.BannerMulticard;
import ru.yandex.direct.core.entity.banner.model.BannerMulticardsCurrencyValues;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLandingParams;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmAudioBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.CpmGeoPinBanner;
import ru.yandex.direct.core.entity.banner.model.CpmIndoorBanner;
import ru.yandex.direct.core.entity.banner.model.CpmOutdoorBanner;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.McBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.grid.processing.model.banner.GdBannerButton;
import ru.yandex.direct.grid.processing.model.banner.GdBannerMeasurer;
import ru.yandex.direct.grid.processing.model.banner.GdBannerMeasurerSystem;
import ru.yandex.direct.grid.processing.model.banner.mutation.BannerMulticardsCurrency;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddContentPromotionAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddCpmAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddDynamicAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMcBannerAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMobileContentAd;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddMulticard;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSmartAd;
import ru.yandex.direct.grid.processing.model.cliententity.GdPixel;
import ru.yandex.direct.grid.processing.model.constants.GdButtonAction;
import ru.yandex.direct.grid.processing.service.banner.BannerDataConverter;

import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.domain.DomainUtils.refineDomain;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.createBannerFlagsForAgeLabel;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.toBannerAdditionalHrefs;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.toCoreBannerPrice;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.toNewMobileContentPrimaryAction;
import static ru.yandex.direct.grid.processing.service.banner.converter.AdMutationDataConverter.toNewReflectedAttribute;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.StringUtils.ifNotBlank;
import static ru.yandex.direct.utils.StringUtils.nullIfBlank;

@ParametersAreNonnullByDefault
public class AddAdMutationDataConverter {

    public static BannerWithAdGroupId toCoreBanner(GdAddAd gdAddAd) {
        BannerWithAdGroupId banner;
        BannersBannerType internalBannerType = BannerDataConverter.toInternalAdType(gdAddAd.getAdType());
        if (BannersBannerType.image_ad.equals(internalBannerType)) {

            if (gdAddAd.getCreativeId() == null) {
                banner = toImageHashBanner(gdAddAd);
            } else {
                banner = toImageCreativeBanner(gdAddAd);
            }
        } else if (BannersBannerType.text.equals(internalBannerType)) {
            banner = toTextBanner(gdAddAd);
        } else if (BannersBannerType.cpc_video.equals(internalBannerType)) {
            banner = toCpcVideoBanner(gdAddAd);
        } else {
            throw new UnsupportedOperationException(format("Ads of type %s are not supported.", gdAddAd.getAdType()));
        }

        //noinspection ConstantConditions
        if (banner instanceof BannerWithTurboLanding) {
            ((BannerWithTurboLanding) banner)
                    .withTurboLandingId(gdAddAd.getTurbolandingId());
        }

        //noinspection ConstantConditions
        if (banner instanceof BannerWithTurboLandingParams) {
            ((BannerWithTurboLandingParams) banner)
                    .withTurboLandingHrefParams(gdAddAd.getTurbolandingHrefParams());
        }

        //noinspection ConstantConditions
        if (banner instanceof BannerWithHref) {
            ((BannerWithHref) banner)
                    .withHref(gdAddAd.getHref())
                    .withDomain(refineDomain(gdAddAd.getDomain()));
        }

        return banner
                .withAdGroupId(gdAddAd.getAdGroupId());
    }

    private static CpcVideoBanner toCpcVideoBanner(GdAddAd gdAddAd) {
        return new CpcVideoBanner()
                .withIsMobileVideo(nvl(gdAddAd.getIsMobile(), false))
                .withCreativeId(gdAddAd.getCreativeId());
    }

    private static TextBanner toTextBanner(GdAddAd gdAddAd) {
        return new TextBanner()
                .withSitelinksSetId(gdAddAd.getSitelinksSetId())
                .withVcardId(gdAddAd.getVcardId())
                .withIsMobile(gdAddAd.getIsMobile())
                .withTitle(gdAddAd.getTitle())
                .withTitleExtension(gdAddAd.getTitleExtension())
                .withBody(gdAddAd.getBody())
                .withDisplayHref(gdAddAd.getDisplayHref())
                .withImageHash(ifNotBlank(gdAddAd.getTextBannerImageHash(), Function.identity()))
                .withCalloutIds(gdAddAd.getCalloutIds())
                .withCreativeId(gdAddAd.getCreativeId())
                .withShowTitleAndBody(gdAddAd.getShowTitleAndBody())
                .withBannerPrice(toCoreBannerPrice(gdAddAd.getAdPrice()))
                .withTurboGalleryHref(gdAddAd.getTurboGalleryHref())
                .withPermalinkId(gdAddAd.getPermalinkId())
                .withPhoneId(gdAddAd.getPhoneId())
                .withPreferVCardOverPermalink(gdAddAd.getPreferVCardOverPermalink())
                .withLogoImageHash(gdAddAd.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(gdAddAd.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(gdAddAd.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(gdAddAd.getButton(), GdBannerButton::getHref))
                .withName(gdAddAd.getName())
                .withMulticards(mapList(gdAddAd.getMulticards(), AddAdMutationDataConverter::toCoreBannerMulticard))
                ;
    }

    public static DynamicBanner toDynamicBanner(GdAddDynamicAd gdAddAd) {
        return new DynamicBanner()
                .withAdGroupId(gdAddAd.getAdGroupId())
                .withSitelinksSetId(gdAddAd.getSitelinksSetId())
                .withVcardId(gdAddAd.getVcardId())
                .withBody(gdAddAd.getBody())
                .withImageHash(ifNotBlank(gdAddAd.getBannerImageHash(), Function.identity()))
                .withCalloutIds(gdAddAd.getCalloutIds());
    }

    private static ImageBanner toImageCreativeBanner(GdAddAd gdAddAd) {
        return new ImageBanner()
                .withIsMobileImage(nvl(gdAddAd.getIsMobile(), false))
                .withCreativeId(gdAddAd.getCreativeId())
                .withTitle(nullIfBlank(gdAddAd.getTitle()))
                .withTitleExtension(gdAddAd.getTitleExtension())
                .withBody(nullIfBlank(gdAddAd.getBody()))
                .withLogoImageHash(gdAddAd.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(gdAddAd.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(gdAddAd.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(gdAddAd.getButton(), GdBannerButton::getHref));
    }

    private static ImageBanner toImageHashBanner(GdAddAd gdAddAd) {
        return new ImageBanner()
                .withIsMobileImage(nvl(gdAddAd.getIsMobile(), false))
                .withImageHash(gdAddAd.getImageCreativeHash())
                .withTitle(nullIfBlank(gdAddAd.getTitle()))
                .withTitleExtension(gdAddAd.getTitleExtension())
                .withBody(nullIfBlank(gdAddAd.getBody()))
                .withLogoImageHash(gdAddAd.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(gdAddAd.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(gdAddAd.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(gdAddAd.getButton(), GdBannerButton::getHref));
    }

    public static BannerWithAdGroupId toPerformanceBanner(GdAddSmartAd gdAddAd) {
        if (gdAddAd.getCreativeId() != null) {
            return new PerformanceBanner()
                    .withAdGroupId(gdAddAd.getAdGroupId())
                    .withCreativeId(gdAddAd.getCreativeId());
        } else {
            return new PerformanceBannerMain()
                    .withAdGroupId(gdAddAd.getAdGroupId())
                    .withLogoImageHash(gdAddAd.getLogoImageHash());
        }
    }

    public static BannerWithAdGroupId toCoreBanner(GdAddCpmAd ad) {
        BannersBannerType internalBannerType = BannerDataConverter.toInternalAdType(ad.getAdType());
        if (internalBannerType == BannersBannerType.cpm_banner) {
            return toCpmBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_audio) {
            return toCpmAudioBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_outdoor) {
            return toCpmOutdoorBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_indoor) {
            return toCpmIndoorBanner(ad);
        } else if (internalBannerType == BannersBannerType.cpm_geo_pin) {
            return toCpmGeoPinBanner(ad);
        } else {
            throw new UnsupportedOperationException(format("Ads of type %s are not supported.", ad.getAdType()));
        }
    }

    private static CpmBanner toCpmBanner(GdAddCpmAd ad) {
        return new CpmBanner()
                .withTitle(ad.getTitle())
                .withTitleExtension(ad.getTitleExtension())
                .withBody(ad.getBody())
                .withLogoImageHash(ad.getLogoImageHash())
                .withButtonAction(GdButtonAction.toSource(ifNotNull(ad.getButton(), GdBannerButton::getAction)))
                .withButtonCaption(ifNotNull(ad.getButton(), GdBannerButton::getCustomText))
                .withButtonHref(ifNotNull(ad.getButton(), GdBannerButton::getHref))
                .withAdGroupId(ad.getAdGroupId())
                .withCreativeId(ad.getCreativeId())
                .withHref(ad.getHref())
                .withTurboLandingId(ad.getTurbolandingId())
                .withTurboLandingHrefParams(ad.getTurbolandingHrefParams())
                .withPixels(mapList(nvl(ad.getPixels(), emptyList()), GdPixel::getUrl))
                .withTnsId(ad.getTnsId())
                .withMeasurers(mapList(nvl(ad.getMeasurers(), emptyList()),
                        AddAdMutationDataConverter::toBannerMeasurer))
                .withMulticards(mapList(ad.getMulticards(), AddAdMutationDataConverter::toCoreBannerMulticard))
                .withBigKingImageHash(ad.getBigKingImageHash())
                .withAdditionalHrefs(toBannerAdditionalHrefs(ad.getAdditionalHrefs()));
    }

    private static CpmAudioBanner toCpmAudioBanner(GdAddCpmAd ad) {
        return new CpmAudioBanner()
                .withAdGroupId(ad.getAdGroupId())
                .withCreativeId(ad.getCreativeId())
                .withHref(ad.getHref())
                .withPixels(mapList(nvl(ad.getPixels(), emptyList()), GdPixel::getUrl))
                .withMeasurers(mapList(nvl(ad.getMeasurers(), emptyList()),
                        AddAdMutationDataConverter::toBannerMeasurer));
    }

    private static BannerMeasurer toBannerMeasurer(GdBannerMeasurer gdBannerMeasurer) {
        // OMI всегда считаем сынтегрированными, но не озадачиваем логикой фронт. Подменяем на входе.
        Boolean hasIntegration = gdBannerMeasurer.getMeasurerSystem() == GdBannerMeasurerSystem.OMI || gdBannerMeasurer
                .getHasIntegration();
        return new BannerMeasurer()
                .withBannerMeasurerSystem(BannerMeasurerSystem.valueOf(gdBannerMeasurer.getMeasurerSystem().name()))
                .withParams(gdBannerMeasurer.getParams())
                .withHasIntegration(hasIntegration);
    }

    private static CpmOutdoorBanner toCpmOutdoorBanner(GdAddCpmAd gdAddAd) {
        return new CpmOutdoorBanner()
                .withAdGroupId(gdAddAd.getAdGroupId())
                .withCreativeId(gdAddAd.getCreativeId())
                .withHref(gdAddAd.getHref());
    }

    private static CpmIndoorBanner toCpmIndoorBanner(GdAddCpmAd gdAddAd) {
        return new CpmIndoorBanner()
                .withAdGroupId(gdAddAd.getAdGroupId())
                .withCreativeId(gdAddAd.getCreativeId())
                .withHref(gdAddAd.getHref());
    }

    private static CpmGeoPinBanner toCpmGeoPinBanner(GdAddCpmAd ad) {
        return new CpmGeoPinBanner()
                .withAdGroupId(ad.getAdGroupId())
                .withCreativeId(ad.getCreativeId())
                .withPixels(mapList(nvl(ad.getPixels(), emptyList()), GdPixel::getUrl))
                .withMeasurers(mapList(nvl(ad.getMeasurers(), emptyList()),
                        AddAdMutationDataConverter::toBannerMeasurer))
                .withPermalinkId(ad.getPermalinkId());
    }

    public static McBanner toMcBanner(GdAddMcBannerAd ad) {
        return new McBanner()
                .withAdGroupId(ad.getAdGroupId())
                .withHref(ad.getHref())
                .withImageHash(ad.getAdImageHash());
    }

    public static MobileAppBanner toMobileAppBanner(GdAddMobileContentAd ad) {
        return new MobileAppBanner()
                .withAdGroupId(ad.getAdGroupId())
                .withImageHash(ad.getAdImageHash())
                .withBody(ad.getBody())
                .withTitle(ad.getTitle())
                .withCreativeId(ad.getCreativeId())
                .withCalloutIds(ad.getCalloutIds())
                .withHref(ad.getTrackingUrl())
                .withFlags(createBannerFlagsForAgeLabel(ad.getAgeLabel()))
                .withPrimaryAction(toNewMobileContentPrimaryAction(ad.getAction()))
                .withReflectedAttributes(toNewReflectedAttribute(ad.getFeatures()))
                .withImpressionUrl(ad.getImpressionUrl());
    }

    public static ContentPromotionBanner toContentPromotion(GdAddContentPromotionAd ad) {
        return new ContentPromotionBanner()
                .withContentPromotionId(ad.getContentPromotion().getContentPromotionId())
                .withAdGroupId(ad.getAdGroupId())
                .withVisitUrl(ad.getVisitUrl())
                .withTitle(ad.getTitle())
                .withBody(ad.getDescription());
    }

    @Nullable
    public static BannerMulticard toCoreBannerMulticard(GdAddMulticard multicard) {
        return new BannerMulticard()
                .withText(multicard.getText())
                .withImageHash(multicard.getImageHash())
                .withHref(multicard.getHref())
                .withPrice(multicard.getPrice())
                .withPriceOld(multicard.getPriceOld())
                .withCurrency(ru.yandex.direct.core.entity.banner.model.BannerMulticardsCurrency
                        .fromSource(Optional.ofNullable(multicard.getCurrency())
                                        .map(BannerMulticardsCurrency::toSource)
                                        .map(BannerMulticardsCurrencyValues::toSource)
                                                .orElse(null)));
    }
}
