package ru.yandex.direct.grid.processing.service.banner.converter;

import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerHrefItem;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerHrefItemSitelink;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.mutation.FindAndReplaceAdsHrefDomainCacheFilterData;
import ru.yandex.direct.grid.processing.model.banner.mutation.FindAndReplaceAdsHrefParamsCacheFilterData;
import ru.yandex.direct.grid.processing.model.banner.mutation.FindAndReplaceAdsHrefTextCacheFilterData;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefDomain;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefParams;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefPreviewPayloadItemSitelink;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefText;
import ru.yandex.direct.grid.processing.model.common.GdCachedResult;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.service.banner.container.FindAndReplaceBannerHrefDomainCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.banner.container.FindAndReplaceBannerHrefParamsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.banner.container.FindAndReplaceBannerHrefTextCacheRecordInfo;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetValidationService.MAX_SITELINKS_PER_SET;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class FindAndReplaceBannerHrefConverter {

    private FindAndReplaceBannerHrefConverter() {
    }

    public static GdResult<GdFindAndReplaceAdsHrefPayloadItem> getFindAndReplaceAdsHrefPayload(
            List<GdiFindAndReplaceBannerHrefItem> bannersToUpdate,
            GridBannerUpdateInfo updateInfo,
            GdValidationResult vr) {
        List<GdFindAndReplaceAdsHrefPayloadItem> rowset = convertToPreviewItems(bannersToUpdate);

        return new GdResult<GdFindAndReplaceAdsHrefPayloadItem>()
                .withSuccessCount(updateInfo.getUpdatedBannerCount())
                .withRowset(rowset)
                .withTotalCount(rowset.size())
                .withValidationResult(vr);
    }

    /**
     * Сконвертировать баннеры в GdFindAndReplaceAdsHrefPreviewPayloadItem
     *
     * @param banners - banners исходные баннеры
     * @return - список сконвертированных баннеров
     */
    public static List<GdFindAndReplaceAdsHrefPayloadItem> convertToPreviewItems(
            List<GdiFindAndReplaceBannerHrefItem> banners) {
        return banners.stream()
                .map(FindAndReplaceBannerHrefConverter::gdiFindAndReplaceBannerHrefItemToPreview)
                .collect(Collectors.toList());
    }

    private static GdFindAndReplaceAdsHrefPayloadItem gdiFindAndReplaceBannerHrefItemToPreview(
            GdiFindAndReplaceBannerHrefItem gdiFindAndReplaceBannerHrefParamsItem) {
        List<GdiFindAndReplaceBannerHrefItemSitelink> sitelinkList =
                gdiFindAndReplaceBannerHrefParamsItem.getSitelinks();
        checkState(sitelinkList.size() <= MAX_SITELINKS_PER_SET,
                "sitelinks size must be no more than " + MAX_SITELINKS_PER_SET + ". %s", sitelinkList);
        return new GdFindAndReplaceAdsHrefPayloadItem()
                .withAdId(gdiFindAndReplaceBannerHrefParamsItem.getBannerId())
                .withOldHref(gdiFindAndReplaceBannerHrefParamsItem.getOldHref())
                .withNewHref(gdiFindAndReplaceBannerHrefParamsItem.getNewHref())
                .withSitelinks(mapList(gdiFindAndReplaceBannerHrefParamsItem.getSitelinks(),
                        sl -> new GdFindAndReplaceAdsHrefPreviewPayloadItemSitelink()
                                .withSitelinkId(ifNotNull(sl, GdiFindAndReplaceBannerHrefItemSitelink::getSitelinkId))
                                .withTitle(ifNotNull(sl, item -> item.getSitelink().getTitle()))
                                .withOldHref(ifNotNull(sl, GdiFindAndReplaceBannerHrefItemSitelink::getOldHref))
                                .withNewHref(ifNotNull(sl, FindAndReplaceBannerHrefConverter::getSitelinkNewHref))));
    }

    /**
     * Если сайтлинк не был изменен, в Sitelink.HREF будет лежать старый href, потому возвращаем null
     */
    private static String getSitelinkNewHref(GdiFindAndReplaceBannerHrefItemSitelink sitelinkItem) {
        return sitelinkItem.getChanged()
                ? sitelinkItem.getSitelink().getHref()
                : null;
    }

    public static FindAndReplaceBannerHrefParamsCacheRecordInfo getHrefParamCacheRecordInfo(ClientId clientId,
                                                                                            GdFindAndReplaceAdsHrefParams input) {
        return new FindAndReplaceBannerHrefParamsCacheRecordInfo(clientId.asLong(), input.getCacheKey(),
                new FindAndReplaceAdsHrefParamsCacheFilterData()
                        .withAdIds(ImmutableList.copyOf(input.getAdIds()))
                        .withAdIdsHrefExceptions(ImmutableSet.copyOf(input.getAdIdsHrefExceptions()))
                        .withSitelinkIdsHrefExceptions(ImmutableMap.copyOf(input.getSitelinkIdsHrefExceptions()))
                        .withTargetTypes(ImmutableSet.copyOf(input.getTargetTypes()))
                        .withReplaceInstruction(input.getReplaceInstruction())
        );
    }

    public static FindAndReplaceBannerHrefDomainCacheRecordInfo getHrefDomainCacheRecordInfo(ClientId clientId,
                                                                                             GdFindAndReplaceAdsHrefDomain input) {
        return new FindAndReplaceBannerHrefDomainCacheRecordInfo(clientId.asLong(), input.getCacheKey(),
                new FindAndReplaceAdsHrefDomainCacheFilterData()
                        .withAdIds(ImmutableList.copyOf(input.getAdIds()))
                        .withAdIdsHrefExceptions(ImmutableSet.copyOf(input.getAdIdsHrefExceptions()))
                        .withSitelinkIdsHrefExceptions(ImmutableMap.copyOf(input.getSitelinkIdsHrefExceptions()))
                        .withTargetTypes(ImmutableSet.copyOf(input.getTargetTypes()))
                        .withReplaceInstruction(input.getReplaceInstruction())
        );
    }

    public static FindAndReplaceBannerHrefTextCacheRecordInfo getHrefTextCacheRecordInfo(ClientId clientId,
                                                                                         GdFindAndReplaceAdsHrefText input) {
        return new FindAndReplaceBannerHrefTextCacheRecordInfo(clientId.asLong(), input.getCacheKey(),
                new FindAndReplaceAdsHrefTextCacheFilterData()
                        .withAdIds(ImmutableList.copyOf(input.getAdIds()))
                        .withAdIdsHrefExceptions(ImmutableSet.copyOf(input.getAdIdsHrefExceptions()))
                        .withSitelinkIdsHrefExceptions(ImmutableMap.copyOf(input.getSitelinkIdsHrefExceptions()))
                        .withTargetTypes(ImmutableSet.copyOf(input.getTargetTypes()))
                        .withReplaceInstruction(input.getReplaceInstruction())
        );
    }

    public static GdResult<GdFindAndReplaceAdsHrefPayloadItem> getEmptyPayload() {
        return new GdResult<GdFindAndReplaceAdsHrefPayloadItem>()
                .withTotalCount(0)
                .withSuccessCount(0)
                .withRowset(emptyList());
    }

    public static GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem> getEmptyPreviewPayload() {
        return new GdCachedResult<GdFindAndReplaceAdsHrefPayloadItem>()
                .withTotalCount(0)
                .withSuccessCount(0)
                .withCacheKey("")
                .withRowset(emptyList());
    }
}
