package ru.yandex.direct.grid.processing.service.banner.converter;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerTextItem;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceRequest;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.grid.model.findandreplace.ChangeMode;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceBannerChangeItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceChangeItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceSitelinkChangeItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceText;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceTextInstruction;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceTextPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdReplacementMode;
import ru.yandex.direct.grid.processing.model.common.GdCachedResult;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.util.findandreplace.AlwaysReplaceRule;
import ru.yandex.direct.grid.processing.util.findandreplace.BannerHrefDomainReplaceRule;
import ru.yandex.direct.grid.processing.util.findandreplace.BannerHrefParamsReplaceRule;
import ru.yandex.direct.grid.processing.util.findandreplace.BannerHrefTextReplaceRule;
import ru.yandex.direct.grid.processing.util.findandreplace.BannerTurbolandingParamsReplaceRule;
import ru.yandex.direct.grid.processing.util.findandreplace.DeleteWhenContainsRule;
import ru.yandex.direct.grid.processing.util.findandreplace.FindAndReplaceRule;
import ru.yandex.direct.grid.processing.util.findandreplace.ReplaceWhenContainsRule;

import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.BODY;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.DISPLAY_HREF;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.HREF;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.SITELINK_DESCRIPTION;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.SITELINK_HREF;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.SITELINK_TITLE;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.TITLE;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.TITLE_EXTENSION;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType.TURBOLANDING_PARAMS;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Component
public class FindAndReplaceBannerTextConverter {

    private final BannersUrlHelper bannersUrlHelper;

    private FindAndReplaceBannerTextConverter(
            BannersUrlHelper bannersUrlHelper) {
        this.bannersUrlHelper = bannersUrlHelper;
    }

    public static GdFindAndReplaceTextPayloadItem convertToPayloadItem(GdiFindAndReplaceBannerTextItem item) {
        List<GdFindAndReplaceChangeItem> changes = new ArrayList<>();

        addChangeItemIfNeeded(changes, TITLE, item.getOldTitle(), item.getNewTitle());
        addChangeItemIfNeeded(changes, TITLE_EXTENSION, item.getOldTitleExtension(), item.getNewTitleExtension());
        addChangeItemIfNeeded(changes, BODY, item.getOldBody(), item.getNewBody());
        addChangeItemIfNeeded(changes, HREF, item.getOldHref(), item.getNewHref());
        addChangeItemIfNeeded(changes, DISPLAY_HREF, item.getOldDisplayHref(), item.getNewDisplayHref());
        addChangeItemIfNeeded(changes, TURBOLANDING_PARAMS, item.getOldTurbolandingParams(),
                item.getNewTurbolandingParams());

        item.getSitelinks().forEach(sitelink -> {
            if (sitelink == null) {
                return;
            }
            var sitelinkId = sitelink.getSitelinkId();
            var orderNum = sitelink.getSitelink().getOrderNum();
            var newTitle = sitelink.getSitelink().getTitle();
            var newDescription = sitelink.getSitelink().getDescription();
            var newHref = sitelink.getSitelink().getHref();
            addSitelinkChangeItemIfNeeded(changes, SITELINK_TITLE, sitelinkId, orderNum,
                    sitelink.getOldTitle(), newTitle);
            addSitelinkChangeItemIfNeeded(changes, SITELINK_DESCRIPTION, sitelinkId, orderNum,
                    sitelink.getOldDescription(), newDescription);
            addSitelinkChangeItemIfNeeded(changes, SITELINK_HREF, sitelinkId, orderNum, sitelink.getOldHref(), newHref);
        });
        return new GdFindAndReplaceTextPayloadItem()
                .withAdId(item.getBannerId())
                .withChanges(changes);
    }

    private static void addChangeItemIfNeeded(List<GdFindAndReplaceChangeItem> changes,
                                              GdFindAndReplaceAdsTargetType target, String oldValue, String newValue) {
        if (!Objects.equals(oldValue, newValue)) {
            changes.add(toBannerChangeItem(target, oldValue, newValue));
        }
    }

    private static void addSitelinkChangeItemIfNeeded(List<GdFindAndReplaceChangeItem> changes,
                                                      GdFindAndReplaceAdsTargetType target,
                                                      long sitelinkId, long orderNum,
                                                      String oldValue, String newValue) {
        if (!Objects.equals(oldValue, newValue)) {
            changes.add(toSitelinkChangeItem(target, sitelinkId, orderNum, oldValue, newValue));
        }
    }

    private static GdFindAndReplaceChangeItem toBannerChangeItem(GdFindAndReplaceAdsTargetType target,
                                                                 String oldValue, String newValue) {
        return new GdFindAndReplaceBannerChangeItem()
                .withTarget(target)
                .withOldValue(oldValue)
                .withNewValue(newValue);
    }

    private static GdFindAndReplaceSitelinkChangeItem toSitelinkChangeItem(GdFindAndReplaceAdsTargetType target,
                                                                           long sitelinkId, long orderNum,
                                                                           String oldValue, String newValue) {
        return new GdFindAndReplaceSitelinkChangeItem()
                .withTarget(target)
                .withSitelinkId(sitelinkId)
                .withOrderNum(orderNum)
                .withOldValue(oldValue)
                .withNewValue(newValue);
    }

    public static GdResult<GdFindAndReplaceTextPayloadItem> toGdResult(List<GdFindAndReplaceTextPayloadItem> rowset,
                                                                       @Nullable GdValidationResult validationResult,
                                                                       GridBannerUpdateInfo updateInfo) {
        return new GdResult<GdFindAndReplaceTextPayloadItem>()
                .withRowset(rowset)
                .withValidationResult(validationResult)
                .withTotalCount(rowset.size())
                .withSuccessCount(updateInfo.getUpdatedBannerCount());
    }

    public static GdCachedResult<GdFindAndReplaceTextPayloadItem> toGdCachedResult(
            List<GdFindAndReplaceTextPayloadItem> rowset, @Nullable GdValidationResult validationResult,
            GridBannerUpdateInfo updateInfo) {
        return new GdCachedResult<GdFindAndReplaceTextPayloadItem>()
                .withValidationResult(validationResult)
                .withTotalCount(rowset.size())
                .withSuccessCount(updateInfo.getUpdatedBannerCount());
    }

    public GdiFindAndReplaceRequest toReplaceRequest(GdFindAndReplaceText input) {
        return new GdiFindAndReplaceRequest()
                // Update fields
                .withTargetTypes(mapSet(input.getTargetTypes(), GdFindAndReplaceAdsTargetType::toSource))
                .withUpdateTitle(input.getTargetTypes().contains(TITLE))
                .withUpdateTitleExtension(input.getTargetTypes().contains(TITLE_EXTENSION))
                .withUpdateBody(input.getTargetTypes().contains(BODY))
                .withUpdateHref(input.getTargetTypes().contains(HREF))
                .withUpdateDisplayHref(input.getTargetTypes().contains(DISPLAY_HREF))
                .withUpdateSitelinkTitle(input.getTargetTypes().contains(SITELINK_TITLE))
                .withUpdateSitelinkDescription(input.getTargetTypes().contains(SITELINK_DESCRIPTION))
                .withUpdateSitelinkHref(input.getTargetTypes().contains(SITELINK_HREF))
                .withUpdateSitelink(input.getTargetTypes().contains(SITELINK_TITLE)
                        || input.getTargetTypes().contains(SITELINK_DESCRIPTION)
                        || input.getTargetTypes().contains(SITELINK_HREF))
                .withUpdateTurbolandingParams(input.getTargetTypes().contains(TURBOLANDING_PARAMS))
                // Find and replace strings
                .withFind(input.getReplaceInstruction().getSearch())
                .withReplace(input.getReplaceInstruction().getReplace())
                // Replacement options
                .withCaseSensitive(input.getReplaceInstruction().getOptions().getCaseSensitive())
                .withReplaceRule(toReplaceRule(input.getReplaceInstruction()))
                .withLinkReplaceRule(toLinkReplaceRule(input.getReplaceInstruction()))
                .withTurbolandingParamsReplaceRule(toTurbolandingParamsReplaceRule(input.getReplaceInstruction()))
                .withSitelinkOrderNumsToUpdateTitle(
                        input.getReplaceInstruction().getOptions().getSitelinkOrderNumsToUpdateTitle())
                .withSitelinkOrderNumsToUpdateDescription(
                        input.getReplaceInstruction().getOptions().getSitelinkOrderNumsToUpdateDescription())
                .withSitelinkOrderNumsToUpdateHref(
                        input.getReplaceInstruction().getOptions().getSitelinkOrderNumsToUpdateHref());
    }

    private static ReplaceRule toReplaceRule(GdFindAndReplaceTextInstruction replaceTextInstruction) {
        var mode = replaceTextInstruction.getOptions().getReplacementMode();
        var search = replaceTextInstruction.getSearch();
        var replace = replaceTextInstruction.getReplace();
        var caseSensitive = replaceTextInstruction.getOptions().getCaseSensitive();
        switch (mode) {
            case APPEND:
            case PREPEND:
                return new ReplaceWhenContainsRule(search, replace, toChangeMode(mode), caseSensitive);
            case REPLACE_ALL:
                return new AlwaysReplaceRule(replace, toChangeMode(mode));
            case DELETE:
                return new DeleteWhenContainsRule(search, caseSensitive);
            case FIND_AND_REPLACE:

                return new FindAndReplaceRule(search, replace, caseSensitive);
            default:
                throw new IllegalArgumentException("Unknown replacement mode: " + mode);
        }
    }

    private ReplaceRule toLinkReplaceRule(GdFindAndReplaceTextInstruction replaceTextInstruction) {
        var changeMode = toChangeMode(replaceTextInstruction.getOptions().getReplacementMode());
        var linkReplacementMode = replaceTextInstruction.getOptions().getLinkReplacementMode();
        var search = replaceTextInstruction.getSearch();
        var replace = replaceTextInstruction.getReplace();
        switch (linkReplacementMode) {
            case DOMAIN:
                return new BannerHrefDomainReplaceRule(search, replace, changeMode, bannersUrlHelper);
            case PARAMS:
                return new BannerHrefParamsReplaceRule(search, replace, changeMode);
            case FULL:
                // В режиме FULL рассматриваем ссылку как полную строку, применяем обычное правило для замены
                return toReplaceRule(replaceTextInstruction);
            case PROTOCOL_DOMAIN_PATH:
                return new BannerHrefTextReplaceRule(BannerHrefTextReplaceRule.ReplaceParts.PROTOCOL_DOMAIN_PATH,
                        search, replace);
            case QUERY_AND_FRAGMENT:
                return new BannerHrefTextReplaceRule(BannerHrefTextReplaceRule.ReplaceParts.QUERY_AND_FRAGMENT,
                        search, replace);
            default:
                throw new IllegalArgumentException("Unknown link replacement mode: " + linkReplacementMode);
        }
    }

    private static ReplaceRule toTurbolandingParamsReplaceRule(GdFindAndReplaceTextInstruction replaceTextInstruction) {
        var changeMode = toChangeMode(replaceTextInstruction.getOptions().getReplacementMode());
        var search = replaceTextInstruction.getSearch();
        var replace = replaceTextInstruction.getReplace();
        return new BannerTurbolandingParamsReplaceRule(search, replace, changeMode);
    }

    private static ChangeMode toChangeMode(GdReplacementMode replacementMode) {
        switch (replacementMode) {
            case APPEND:
                return ChangeMode.POSTFIX;
            case PREPEND:
                return ChangeMode.PREFIX;
            case REPLACE_ALL:
            default:
                // Если не можем найти соответствие, по умолчанию используем REPLACE
                return ChangeMode.REPLACE;
        }
    }
}
