package ru.yandex.direct.grid.processing.service.banner.internalad

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.model.InternalBanner
import ru.yandex.direct.core.entity.banner.model.InternalModerationInfo
import ru.yandex.direct.core.entity.banner.model.TemplateVariable
import ru.yandex.direct.core.entity.internalads.Constants.isModeratedTemplate
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeCustomComment
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeIsSecretAd
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeSendToModeration
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeStatusShow
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeStatusShowAfterModeration
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeTemplateVariable
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeTicketUrl
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeValue
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateCustomComment
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateIsSecretAd
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateSendToModeration
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateStatusShow
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateStatusShowAfterModeration
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateTemplateVariable
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldStateTicketUrl
import ru.yandex.direct.grid.processing.model.banner.GdTemplateVariable
import ru.yandex.direct.model.ModelProperty

class SimpleInternalBannerFieldOperations<T>(
    val getter: (banner: InternalBanner) -> T?,
    val setter: (banner: InternalBanner, value: T?) -> Unit
) : InternalBannerFieldOperations<T> {

    override fun extract(obj: InternalBanner): List<T> {
        return getter(obj)?.let { listOf(it) } ?: emptyList()
    }

    override fun remove(obj: InternalBanner, value: T) {
        setter(obj, null)
    }

    override fun add(obj: InternalBanner, value: T) {
        setter(obj, value)
    }
}

object TemplateVariableFieldOperations : InternalBannerFieldOperations<TemplateVariable> {
    override fun extract(obj: InternalBanner): List<TemplateVariable> {
        return obj.templateVariables
    }

    override fun remove(obj: InternalBanner, value: TemplateVariable) {
        obj.templateVariables.single {
            it.templateResourceId == value.templateResourceId
        }.internalValue = null
    }

    override fun add(obj: InternalBanner, value: TemplateVariable) {
        obj.templateVariables.single {
            it.templateResourceId == value.templateResourceId
        }.internalValue = value.internalValue
    }
}

abstract class ModerationInfoUpdateFieldSupport<T>(
    val modelProperty: ModelProperty<InternalModerationInfo, T>
) : InternalBannerMassUpdateFieldSupport<T> {

    override val operations = SimpleInternalBannerFieldOperations(
        { banner -> banner.moderationInfo?.let { modelProperty.get(it) } },
        { banner, value -> banner.moderationInfo?.let { modelProperty.set(it, value) } }
    )
}

@Component
class SendToModerationUpdateFieldSupport
    : ModerationInfoUpdateFieldSupport<Boolean>(InternalModerationInfo.SEND_TO_MODERATION) {

    override fun convertToGdFieldState(value: Boolean) =
        GdInternalBannerFieldStateSendToModeration().apply {
            sendToModeration = value
        }

    override fun extractInputInnerValue(input: GdInternalBannerFieldChangeValue<Any>) =
        when (input) {
            is GdInternalBannerFieldChangeSendToModeration -> input.innerValue
            else -> null
        }
}

@Component
class StatusShowAfterModerationUpdateFieldSupport
    : ModerationInfoUpdateFieldSupport<Boolean>(InternalModerationInfo.STATUS_SHOW_AFTER_MODERATION) {

    override fun convertToGdFieldState(value: Boolean) =
        GdInternalBannerFieldStateStatusShowAfterModeration().apply {
            statusShowAfterModeration = value
        }

    override fun extractInputInnerValue(input: GdInternalBannerFieldChangeValue<Any>) =
        when (input) {
            is GdInternalBannerFieldChangeStatusShowAfterModeration -> input.innerValue
            else -> null
        }
}

@Component
class CustomCommentMassUpdateFieldSupport
    : ModerationInfoUpdateFieldSupport<String>(InternalModerationInfo.CUSTOM_COMMENT) {

    override fun convertToGdFieldState(value: String) =
        GdInternalBannerFieldStateCustomComment().apply {
            customComment = value
        }

    override fun extractInputInnerValue(input: GdInternalBannerFieldChangeValue<Any>) =
        when (input) {
            is GdInternalBannerFieldChangeCustomComment -> input.innerValue
            else -> null
        }
}

@Component
class IsSecretAdMassUpdateFieldSupport
    : ModerationInfoUpdateFieldSupport<Boolean>(InternalModerationInfo.IS_SECRET_AD) {

    override fun convertToGdFieldState(value: Boolean) =
        GdInternalBannerFieldStateIsSecretAd().apply {
            isSecretAd = value
        }

    override fun extractInputInnerValue(input: GdInternalBannerFieldChangeValue<Any>) =
        when (input) {
            is GdInternalBannerFieldChangeIsSecretAd -> input.innerValue
            else -> null
        }
}

@Component
class TicketUrlMassUpdateFieldSupport
    : ModerationInfoUpdateFieldSupport<String>(InternalModerationInfo.TICKET_URL) {

    override fun convertToGdFieldState(value: String) =
        GdInternalBannerFieldStateTicketUrl().apply {
            ticketUrl = value
        }

    override fun extractInputInnerValue(input: GdInternalBannerFieldChangeValue<Any>) =
        when (input) {
            is GdInternalBannerFieldChangeTicketUrl -> input.innerValue
            else -> null
        }
}

@Component
class StatusShowUpdateFieldSupport : InternalBannerMassUpdateFieldSupport<Boolean> {
    override val operations = SimpleInternalBannerFieldOperations(
        { banner -> if (isModeratedTemplate(banner.templateId)) banner.statusShow else null },
        InternalBanner::setStatusShow
    )

    override fun convertToGdFieldState(value: Boolean) =
        GdInternalBannerFieldStateStatusShow().apply {
            statusShow = value
        }

    override fun extractInputInnerValue(input: GdInternalBannerFieldChangeValue<Any>) =
        when (input) {
            is GdInternalBannerFieldChangeStatusShow -> input.innerValue
            else -> null
        }
}

@Component
class TemplateVariableMassUpdateFieldSupport : InternalBannerMassUpdateFieldSupport<TemplateVariable> {
    override val operations = TemplateVariableFieldOperations

    override fun convertToGdFieldState(value: TemplateVariable) =
        GdInternalBannerFieldStateTemplateVariable().apply {
            templateVariable = GdTemplateVariable()
                .withTemplateResourceId(value.templateResourceId)
                .withValue(value.internalValue)
        }

    override fun extractInputInnerValue(input: GdInternalBannerFieldChangeValue<Any>) =
        when (input) {
            is GdInternalBannerFieldChangeTemplateVariable -> TemplateVariable().apply {
                templateResourceId = input.innerValue.templateResourceId
                internalValue = input.innerValue.value
            }
            else -> null
        }
}
