package ru.yandex.direct.grid.processing.service.banner.loader

import org.dataloader.BatchLoaderEnvironment
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Scope
import org.springframework.context.annotation.ScopedProxyMode
import org.springframework.stereotype.Component
import org.springframework.web.context.WebApplicationContext
import ru.yandex.direct.core.entity.image.container.ImageFilterContainer
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImage
import ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider
import java.util.concurrent.CompletableFuture

/**
 * Получение картинок [GdImage] клиента (из `banner_images_pool` + `banner_images_formats`) по `imageHash`
 */
@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class BannerImageDataLoader(
    gridContextProvider: GridContextProvider,
    private val imageDataRepository: ImageDataRepository,
) : GridBatchingDataLoader<String, GdImage>() {
    companion object {
        val logger: Logger = LoggerFactory.getLogger(BannerImageDataLoader::class.java)
    }

    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::load)
    }

    private fun load(
        imageHashes: Set<String>,
        environment: BatchLoaderEnvironment
    ): CompletableFuture<Map<String, GdImage>> {
        if (imageHashes.isEmpty()) {
            return CompletableFuture.completedFuture(emptyMap())
        }

        val context = environment.getContext<GridGraphQLContext>()
        val clientInfo = context.queriedClient
            ?: throw IllegalStateException("No queriedClient in graphql context")

        val filter = ImageFilterContainer().withImageHashes(imageHashes)
        val images = imageDataRepository.getImages(clientInfo.shard, ClientId.fromLong(clientInfo.id), filter)

        val result = images
            .associateBy { it.imageHash }
            .mapValues { (_, image) -> ClientEntityConverter.toGdImage(image) }

        if (result.keys != imageHashes) {
            logger.error(
                "Failed to find images for client {} and image hashes [{}]",
                clientInfo.id, imageHashes - result.keys,
            )
        }

        return CompletableFuture.completedFuture(result)
    }
}
