package ru.yandex.direct.grid.processing.service.banner.loader;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.DataLoader;
import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * {@link DataLoader} для отложенной batch'евой загрузки информации о возможности удаления объявлений
 *
 * @see BannerService#getCanBeDeletedBanners(int, ru.yandex.direct.dbutil.model.ClientId, java.util.List)
 * @see GridBatchingDataLoader
 * @see ru.yandex.direct.grid.processing.service.dataloader.GridDataLoaderRegistry
 */
@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CanBeDeletedAdsDataLoader extends GridBatchingDataLoader<Long, Boolean> {

    public CanBeDeletedAdsDataLoader(GridContextProvider gridContextProvider, BannerService bannerService) {
        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(bannerService));
    }

    private MappedBatchLoaderWithContext<Long, Boolean> getBatchLoadFunction(BannerService bannerService) {
        return (adIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");
            ClientId clientId = ClientId.fromLong(queriedClient.getId());

            Map<Long, Boolean> canBeDeletedBanners = bannerService
                    .getCanBeDeletedBannersByIds(queriedClient.getShard(), clientId, adIds);

            return CompletableFuture.completedFuture(canBeDeletedBanners);
        };
    }
}
