package ru.yandex.direct.grid.processing.service.banner.loader

import org.dataloader.BatchLoaderEnvironment
import org.springframework.context.annotation.Scope
import org.springframework.context.annotation.ScopedProxyMode
import org.springframework.stereotype.Component
import org.springframework.web.context.WebApplicationContext
import ru.yandex.direct.core.entity.banner.type.multicard.BannerMulticardSetsRepository
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider
import java.util.concurrent.CompletableFuture

/**
 * Алиасы для более понятного возвращаемого значения
 */
typealias ReasonId = Long
typealias MulticardId = Long

/**
 * Получение причин отклонения карточек мультибаннера
 */
@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class MulticardReasonDataLoader(
    gridContextProvider: GridContextProvider,
    private val bannerMulticardSetsRepository: BannerMulticardSetsRepository,
    private val moderationReasonRepository: ModerationReasonRepository,
) : GridBatchingDataLoader<MulticardId, Set<Long>>() {

    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::load)
    }

    private fun load(
        multicardIds: Set<Long>,
        environment: BatchLoaderEnvironment
    ): CompletableFuture<Map<MulticardId, Set<ReasonId>>> {
        if (multicardIds.isEmpty()) {
            return CompletableFuture.completedFuture(emptyMap())
        }

        val context = environment.getContext<GridGraphQLContext>()
        val queriedClient = context.queriedClient

        val bannerIds = bannerMulticardSetsRepository.getBannerIdsByMulticardIds(queriedClient.shard, multicardIds)

        val moderationReasons = moderationReasonRepository.fetchRejected(
            queriedClient.shard,
            ModerationReasonObjectType.BANNER_MULTICARD, bannerIds
        )

        val reasonIdSetByMulticardId: MutableMap<MulticardId, MutableSet<ReasonId>> = mutableMapOf()

        moderationReasons.forEach { record ->
            record.reasons.forEach { reason ->
                reason.itemIds.forEach { multicardId ->
                    val multicardReasons = reasonIdSetByMulticardId
                        .computeIfAbsent(multicardId) { mutableSetOf() }
                    multicardReasons += reason.id
                }
            }
        }

        return CompletableFuture.completedFuture(reasonIdSetByMulticardId)
    }
}
