package ru.yandex.direct.grid.processing.service.banner.mutation;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.GdInternalModerationInfo;
import ru.yandex.direct.grid.processing.model.banner.GdTemplateVariable;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddAdsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddInternalAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddInternalAdsItem;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.StringUtils;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toCoreInternalModerationInfo;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResultIfErrors;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class AddInternalAdsMutationService {

    private final InternalAdsProductService internalAdsProductService;
    private final GridValidationService gridValidationService;
    private final PathNodeConverterProvider errorPathConverterProvider;
    private final BannersAddOperationFactory bannersAddOperationFactory;

    public AddInternalAdsMutationService(InternalAdsProductService internalAdsProductService,
                                         GridValidationService gridValidationService,
                                         BannersAddOperationFactory bannersAddOperationFactory) {
        this.internalAdsProductService = internalAdsProductService;
        this.gridValidationService = gridValidationService;
        this.bannersAddOperationFactory = bannersAddOperationFactory;
        this.errorPathConverterProvider = createErrorPathConverters();
    }

    /**
     * Создание объявлений для внутренней рекламы
     *
     * @param clientId    идентификатор клиента, которому принадлежат создаваемые объявления
     * @param operatorUid идентификатор пользователя оператора
     * @param input       запрос, содержащий список данных добавленных объявлений
     */
    public GdAddAdsPayload addInternalAds(ClientId clientId, Long operatorUid, GdAddInternalAds input) {
        Preconditions.checkState(internalAdsProductService.clientCanHaveInternalAdCampaigns(clientId),
                "Internal ads are not supported for client");

        gridValidationService.applyValidator(this::validateInput, input, false);

        List<BannerWithAdGroupId> banners = mapList(input.getAdAddItems(),
                AddInternalAdsMutationService::toInternalBanner);

        MassResult<Long> result = bannersAddOperationFactory.createPartialAddOperation(banners, clientId, operatorUid,
                input.getSaveDraft()).prepareAndApply();

        GdValidationResult validationResult = buildGridValidationResultIfErrors(
                result.getValidationResult(), path(field(GdAddInternalAds.AD_ADD_ITEMS)), errorPathConverterProvider);

        List<GdAddAdPayloadItem> payloadItems = getResults(result, id -> new GdAddAdPayloadItem().withId(id));
        return new GdAddAdsPayload()
                .withAddedAds(payloadItems)
                .withValidationResult(validationResult);
    }

    private ValidationResult<GdAddInternalAds, Defect> validateInput(GdAddInternalAds input) {
        ModelItemValidationBuilder<GdAddInternalAds> vb = ModelItemValidationBuilder.of(input);
        vb.list(GdAddInternalAds.AD_ADD_ITEMS)
                .checkEachBy(this::validateInputItem);
        return vb.getResult();
    }

    private ValidationResult<GdAddInternalAdsItem, Defect> validateInputItem(GdAddInternalAdsItem item) {
        ModelItemValidationBuilder<GdAddInternalAdsItem> vb = ModelItemValidationBuilder.of(item);
        vb.list(GdAddInternalAdsItem.TEMPLATE_VARS)
                .checkEachBy(this::validateTemplateVariable);
        return vb.getResult();
    }

    private ValidationResult<GdTemplateVariable, Defect> validateTemplateVariable(GdTemplateVariable variable) {
        ModelItemValidationBuilder<GdTemplateVariable> vb = ModelItemValidationBuilder.of(variable);
        vb.item(GdTemplateVariable.VALUE)
                .check(notBlank()); // фронтенд должен посылать null для пустых значений
        return vb.getResult();
    }

    private static InternalBanner toInternalBanner(GdAddInternalAdsItem item) {
        Boolean statusShow = ifNotNull(item.getModerationInfo(), GdInternalModerationInfo::getStatusShow);

        return new InternalBanner()
                .withAdGroupId(item.getAdGroupId())
                .withDescription(preprocessString(item.getDescription()))
                .withTemplateId(item.getTemplateId())
                .withTemplateVariables(
                        mapList(item.getTemplateVars(), AddInternalAdsMutationService::toCoreTemplateVariable))
                .withStatusShow(nvl(statusShow, true))
                .withModerationInfo(toCoreInternalModerationInfo(item.getModerationInfo()));
    }

    private static TemplateVariable toCoreTemplateVariable(GdTemplateVariable gdTemplateVariable) {
        return new TemplateVariable()
                .withTemplateResourceId(gdTemplateVariable.getTemplateResourceId())
                .withInternalValue(preprocessString(gdTemplateVariable.getValue()));
    }

    private DefaultPathNodeConverterProvider createErrorPathConverters() {
        PathNodeConverter adsConverter = SkipByDefaultMappingPathNodeConverter.builder()
                .replace(InternalBanner.AD_GROUP_ID, GdAddInternalAdsItem.AD_GROUP_ID)
                .replace(InternalBanner.DESCRIPTION, GdAddInternalAdsItem.DESCRIPTION)
                .replace(InternalBanner.TEMPLATE_ID, GdAddInternalAdsItem.TEMPLATE_ID)
                .replace(InternalBanner.TEMPLATE_VARIABLES, GdAddInternalAdsItem.TEMPLATE_VARS)
                .replace(InternalBanner.MODERATION_INFO, GdAddInternalAdsItem.MODERATION_INFO)
                .build();
        return DefaultPathNodeConverterProvider.builder()
                .register(InternalBanner.class, adsConverter)
                .build();
    }

    @Nullable
    private static String preprocessString(@Nullable String str) {
        return ifNotNull(str, s -> StringUtils.nullIfBlank(s.trim()));
    }
}
