package ru.yandex.direct.grid.processing.service.banner.mutation;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.GdTemplateVariable;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdPayloadItem;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateAdsPayload;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateInternalAds;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdUpdateInternalAdsItem;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.StringUtils;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.banner.BannerDataConverter.toCoreInternalModerationInfo;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResultIfErrors;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class UpdateInternalAdsMutationService {

    private final InternalAdsProductService internalAdsProductService;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final GridValidationService gridValidationService;
    private final PathNodeConverterProvider errorPathConverterProvider;

    public UpdateInternalAdsMutationService(InternalAdsProductService internalAdsProductService,
                                            BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                            GridValidationService gridValidationService) {
        this.internalAdsProductService = internalAdsProductService;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.gridValidationService = gridValidationService;
        this.errorPathConverterProvider = createErrorPathConverters();
    }

    /**
     * Обновление объявлений для внутренней рекламы
     *
     * @param clientId    идентификатор клиента, которому принадлежат объявления
     * @param operatorUid идентификатор пользователя оператора
     * @param input       запрос, содержащий список данных обновлённых объявлений
     */
    public GdUpdateAdsPayload updateInternalAds(ClientId clientId, Long operatorUid, GdUpdateInternalAds input) {
        Preconditions.checkState(internalAdsProductService.clientCanHaveInternalAdCampaigns(clientId),
                "Internal ads are not supported for client");

        gridValidationService.applyValidator(this::validateInput, input, false);

        List<ModelChanges<BannerWithSystemFields>> banners =
                mapList(input.getAdUpdateItems(), UpdateInternalAdsMutationService::toInternalBannerChanges);

        MassResult<Long> result =
                bannersUpdateOperationFactory.createPartialUpdateOperation(banners, operatorUid, clientId)
                        .prepareAndApply();

        GdValidationResult validationResult = buildGridValidationResultIfErrors(
                result.getValidationResult(), path(field(GdUpdateInternalAds.AD_UPDATE_ITEMS)),
                errorPathConverterProvider);

        List<GdUpdateAdPayloadItem> payloadItems = getResults(result, id -> new GdUpdateAdPayloadItem().withId(id));
        return new GdUpdateAdsPayload()
                .withUpdatedAds(payloadItems)
                .withValidationResult(validationResult);
    }

    private ValidationResult<GdUpdateInternalAds, Defect> validateInput(GdUpdateInternalAds input) {
        ModelItemValidationBuilder<GdUpdateInternalAds> vb = ModelItemValidationBuilder.of(input);
        vb.list(GdUpdateInternalAds.AD_UPDATE_ITEMS)
                .checkEachBy(this::validateInputItem);
        return vb.getResult();
    }

    private ValidationResult<GdUpdateInternalAdsItem, Defect> validateInputItem(GdUpdateInternalAdsItem item) {
        ModelItemValidationBuilder<GdUpdateInternalAdsItem> vb = ModelItemValidationBuilder.of(item);
        vb.list(GdUpdateInternalAdsItem.TEMPLATE_VARS)
                .checkEachBy(this::validateTemplateVariable);
        return vb.getResult();
    }

    private ValidationResult<GdTemplateVariable, Defect> validateTemplateVariable(GdTemplateVariable variable) {
        ModelItemValidationBuilder<GdTemplateVariable> vb = ModelItemValidationBuilder.of(variable);
        vb.item(GdTemplateVariable.VALUE)
                .check(notBlank()); // фронтенд должен посылать null для пустых значений
        return vb.getResult();
    }

    private static ModelChanges<BannerWithSystemFields> toInternalBannerChanges(GdUpdateInternalAdsItem item) {
        var modelChanges = new ModelChanges<>(item.getId(),
                InternalBanner.class)
                .process(preprocessString(item.getDescription()), InternalBanner.DESCRIPTION)
                .process(mapList(item.getTemplateVars(), UpdateInternalAdsMutationService::toCoreTemplateVariable),
                        InternalBanner.TEMPLATE_VARIABLES)
                .process(toCoreInternalModerationInfo(item.getModerationInfo()), InternalBanner.MODERATION_INFO)
                .castModelUp(BannerWithSystemFields.class);

        if (item.getModerationInfo() != null) {
            modelChanges.process(item.getModerationInfo().getStatusShow(), InternalBanner.STATUS_SHOW);
        }
        return modelChanges;
    }

    private static TemplateVariable toCoreTemplateVariable(GdTemplateVariable gdTemplateVariable) {
        return new TemplateVariable()
                .withTemplateResourceId(gdTemplateVariable.getTemplateResourceId())
                .withInternalValue(preprocessString(gdTemplateVariable.getValue()));
    }

    private DefaultPathNodeConverterProvider createErrorPathConverters() {
        PathNodeConverter adsConverter = SkipByDefaultMappingPathNodeConverter.builder()
                .replace(InternalBanner.ID, GdUpdateInternalAdsItem.ID)
                .replace(InternalBanner.DESCRIPTION, GdUpdateInternalAdsItem.DESCRIPTION)
                .replace(InternalBanner.TEMPLATE_VARIABLES, GdUpdateInternalAdsItem.TEMPLATE_VARS)
                .replace(InternalBanner.MODERATION_INFO, GdUpdateInternalAdsItem.MODERATION_INFO)
                .build();
        return DefaultPathNodeConverterProvider.builder()
                .register(InternalBanner.class, adsConverter)
                .build();
    }

    @Nullable
    private static String preprocessString(@Nullable String str) {
        return ifNotNull(str, s -> StringUtils.nullIfBlank(s.trim()));
    }
}
