package ru.yandex.direct.grid.processing.service.banner.validation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.banner.model.InternalBanner
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.differentTemplateIdsNotAllowed
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.templateMustBeModerated
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.templateResourceIdNotFound
import ru.yandex.direct.core.entity.internalads.Constants.isModeratedTemplate
import ru.yandex.direct.core.entity.internalads.model.ReadOnlyDirectTemplateResource
import ru.yandex.direct.core.entity.internalads.service.TemplateResourceService
import ru.yandex.direct.core.validation.ValidationUtils
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.model.api.GdValidationResult
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChange
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeModerationInfoValue
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeTemplateVariable
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeValue
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannerFieldChangeValueUnion
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannersAggregatedState
import ru.yandex.direct.grid.processing.model.banner.GdInternalBannersMassUpdate
import ru.yandex.direct.grid.processing.service.banner.internalad.getSingleValueFromUnion
import ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.eachValidId
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.Path
import ru.yandex.direct.validation.result.PathHelper
import ru.yandex.direct.validation.result.PathNode
import ru.yandex.direct.validation.result.PathNodeConverter
import ru.yandex.direct.validation.result.PathNodeConverterProvider
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.item
import ru.yandex.direct.validation.util.list
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateModel

const val MAX_UPDATE_ITEMS = 200

@Service
class BannerInternalMassUpdateValidationService(
    private val gridValidationService: GridValidationService,
    private val bannerService: BannerService,
    private val templateResourceService: TemplateResourceService
) {
    fun validateInternalBannersAggregatedState(
        input: GdInternalBannersAggregatedState,
        clientId: ClientId,
        operatorUid: Long,
    ) = gridValidationService.applyValidator(
        { validateGdInternalBannersAggregatedState(it, clientId, operatorUid) }, input, false
    )

    fun validateInternalBannersMassUpdate(
        input: GdInternalBannersMassUpdate,
        clientId: ClientId,
        operatorUid: Long,
    ) = gridValidationService.applyValidator(
        { validateGdInternalBannersMassUpdate(it, clientId, operatorUid) }, input, false
    )

    fun validateGdInternalBannersAggregatedState(
        obj: GdInternalBannersAggregatedState,
        clientId: ClientId,
        operatorUid: Long
    ): ValidationResult<GdInternalBannersAggregatedState, Defect<*>> {
        return validateModel(obj) {
            check(notNull())
            list(GdInternalBannersAggregatedState.BANNER_IDS) {
                check(notEmptyCollection())
                check(maxListSize(MAX_UPDATE_ITEMS))
                check(eachValidId())
                checkEach(notNull())
                check(
                    allBannersHaveSameTemplateId(clientId, operatorUid) { it },
                    When.isValid()
                )
            }
        }
    }

    fun validateGdInternalBannersMassUpdate(
        obj: GdInternalBannersMassUpdate,
        clientId: ClientId,
        operatorUid: Long
    ): ValidationResult<GdInternalBannersMassUpdate, Defect<*>> {
        return validateModel(obj) {
            check(notNull())
            list(GdInternalBannersMassUpdate.CHANGES) {
                check(notEmptyCollection())
                check(maxListSize(MAX_UPDATE_ITEMS))
                checkEach(notNull())
                checkEachBy(::validateGdInternalBannerFieldChangeIds)
                check(
                    allBannersHaveSameTemplateId(clientId, operatorUid) { changes -> changes.flatMap { it.ids } },
                    When.isValid()
                )
                checkBy(
                    { validateGdInternalBannerFieldChangesValues(it, clientId, operatorUid) },
                    When.isValid()
                )
            }
        }
    }

    private fun validateGdInternalBannerFieldChangeIds(
        change: GdInternalBannerFieldChange
    ): ValidationResult<GdInternalBannerFieldChange, Defect<*>> {
        return validateModel(change) {
            list(GdInternalBannerFieldChange.IDS) {
                check(notEmptyCollection())
                check(maxListSize(MAX_UPDATE_ITEMS))
                check(eachValidId())
                checkEach(notNull())
            }
        }
    }

    private fun validateGdInternalBannerFieldChangesValues(
        changes: List<GdInternalBannerFieldChange>,
        clientId: ClientId,
        operatorUid: Long
    ): ValidationResult<List<GdInternalBannerFieldChange>, Defect<*>> {
        val allBannerIds = changes.flatMap { it.ids }
        val allBanners = bannerService.get(clientId, operatorUid, allBannerIds)
        val templateId = allBanners
            .asSequence()
            .filterIsInstance<InternalBanner>()
            .map { it.templateId }
            .distinct()
            .single()
        val templateResources = templateResourceService.getReadonlyByTemplateIds(listOf(templateId))

        return validateList(changes) {
            checkEach(notNull())
            checkEachBy { change -> validateGdInternalBannerFieldChangeValue(change, templateId, templateResources) }
        }
    }

    private fun validateGdInternalBannerFieldChangeValue(
        change: GdInternalBannerFieldChange,
        templateId: Long,
        templateResources: Map<Long, ReadOnlyDirectTemplateResource>
    ): ValidationResult<GdInternalBannerFieldChange, Defect<*>> {
        return validateModel(change) {
            item(GdInternalBannerFieldChange.VALUE) {
                check(unionHasSingleValue())
                check(fieldChangeModerationInfoIsValid(templateId), When.isValid())
                check(fieldChangeTemplateVariableIsValid(templateResources), When.isValid())
            }
        }
    }

    private fun fieldChangeModerationInfoIsValid(
        templateId: Long
    ): Constraint<GdInternalBannerFieldChangeValueUnion, Defect<*>?> {
        return Constraint.fromPredicate(
            { value ->
                val valueFromUnion = value.getSingleValueFromUnion()
                valueFromUnion !is GdInternalBannerFieldChangeModerationInfoValue<*>
                    || isModeratedTemplate(templateId)
            }, templateMustBeModerated()
        )
    }

    private fun fieldChangeTemplateVariableIsValid(
        templateResources: Map<Long, ReadOnlyDirectTemplateResource>
    ): Constraint<GdInternalBannerFieldChangeValueUnion, Defect<*>?> {
        return Constraint.fromPredicate(
            { value ->
                val valueFromUnion = value.getSingleValueFromUnion()
                valueFromUnion !is GdInternalBannerFieldChangeTemplateVariable
                    || valueFromUnion.innerValue.templateResourceId in templateResources
            }, templateResourceIdNotFound()
        )
    }

    private fun <T> allBannersHaveSameTemplateId(
        clientId: ClientId,
        operatorUid: Long,
        getBannerIds: (T) -> List<Long>
    ): Constraint<T, Defect<*>?> {
        return Constraint { value ->
            val bannerIds = getBannerIds(value)
            val banners = bannerService.get(clientId, operatorUid, bannerIds)
            val templateIds = banners
                .asSequence()
                .filterIsInstance<InternalBanner>()
                .map { it.templateId }
                .toList()
            // Все баннеры являются баннерами внутренней рекламы и имеют один шаблон
            if (banners.size == templateIds.size && templateIds.toSet().size == 1) {
                null
            } else {
                differentTemplateIdsNotAllowed();
            }
        }
    }

    private fun <T> notNull(): Constraint<T?, Defect<*>?> {
        return Constraint { value: T? ->
            if (value != null) {
                null
            } else CommonDefects.notNull()
        }
    }

    private fun unionHasSingleValue(): Constraint<GdInternalBannerFieldChangeValueUnion, Defect<*>> {
        return Constraint.fromPredicate(
            { union -> collectUnionValues(union).size == 1 },
            GridDefectDefinitions.invalidUnion()
        )
    }

    private fun collectUnionValues(value: GdInternalBannerFieldChangeValueUnion): List<GdInternalBannerFieldChangeValue<Any>> {
        return GdInternalBannerFieldChangeValueUnion.allModelProperties()
            .mapNotNull { it.getRaw(value) }
            .filterIsInstance<GdInternalBannerFieldChangeValue<Any>>()
    }

    fun getValidationResult(vr: ValidationResult<*, Defect<*>>): GdValidationResult? {
        return if (ValidationUtils.hasValidationIssues(vr)) {
            GridValidationResultConversionService.buildGridValidationResult(
                vr,
                PathHelper.path(PathHelper.field("banners")),
                createErrorPathConverters()
            )
        } else null
    }

    private fun createErrorPathConverters(): PathNodeConverterProvider {
        val identityPathConverter = object : PathNodeConverter {
            override fun convert(field: PathNode.Field) = Path(listOf(field))
            override fun convert(field: PathNode.Field, index: PathNode.Index) = Path(listOf(field, index))
        }
        return PathNodeConverterProvider { identityPathConverter }
    }
}
