package ru.yandex.direct.grid.processing.service.bidmodifier;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;

import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.BannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegmentAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnlyAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgoAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilterAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTVAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPositionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideoAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherLiteral;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.core.entity.bidmodifier.InventoryType;
import ru.yandex.direct.core.entity.bidmodifier.OperationType;
import ru.yandex.direct.core.entity.bidmodifier.OsType;
import ru.yandex.direct.core.entity.bidmodifier.TabletOsType;
import ru.yandex.direct.core.entity.bidmodifier.TrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.WeatherType;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionLiteral;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.expression.ParameterType;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierExpressionFactory;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdAdjustmentAdType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdAgeType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifier;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierABSegment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierABSegmentAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierAdType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierAdTypeAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierDemographics;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierDemographicsAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierDesktop;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierDesktopAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierDesktopOnly;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierDesktopOnlyAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierExpress;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierExpressAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierExpressLiteral;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierGeo;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierInventory;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierInventoryAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierMobile;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierMobileAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierRegionalAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierRetargeting;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierRetargetingAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierSmartTV;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierSmartTVAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierSmartTgo;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierSmartTgoAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierTablet;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierTabletAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierTrafaretPosition;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierTrafaretPositionAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierVideo;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierVideoAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierWeather;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierWeatherAdjustment;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierWeatherExpression;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdGenderType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdInventoryType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdOperationType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdOsType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdTrafaretPosition;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdWeatherType;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierABSegment;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierABSegmentAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierAdType;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierAdTypeAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierDemographics;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierDemographicsAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierDesktop;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierDesktopAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierDesktopOnly;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierDesktopOnlyAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierExpress;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierExpressAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierExpressLiteral;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierGeo;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierInventory;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierInventoryAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierMobile;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierMobileAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierRegionalAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierRetargeting;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierRetargetingAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierSmartTV;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierSmartTVAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierSmartTgo;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierSmartTgoAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierTablet;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierTabletAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierTrafaretPosition;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierTrafaretPositionAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierVideo;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierVideoAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierWeather;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierWeatherAdjustmentItem;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifierWeatherExpression;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifiers;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifiersItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRetargetingItem;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargeting;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifierType.AB_SEGMENT_MULTIPLIER;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class BidModifierDataConverter {
    public static GdBidModifier toGdBidModifierImplementation(BidModifier bidModifier) {
        return toGdBidModifierImplementation(bidModifier, null);
    }

    public static GdBidModifier toGdBidModifierImplementation(BidModifier bidModifier, CampaignType campaignType) {
        GdBidModifier gdBidModifier;

        GdBidModifierType gdBidModifierType;
        if (bidModifier instanceof BidModifierExpression) {
            gdBidModifier = toGdBidModExpress((BidModifierExpression) bidModifier);
            // гридовые названия универсальных корректировок совпадают с ядровыми.
            // если не добавить новый ядровый тип в енум GdBidModifierType, тут упадём с IllegalArgumentException
            gdBidModifierType = GdBidModifierType.valueOf(bidModifier.getType().toString());
        } else {
            gdBidModifier = fromLegacyCoreModels(bidModifier, campaignType);
            gdBidModifierType = toGdBidModifierType(bidModifier.getType());
        }

        return gdBidModifier.withAdGroupId(bidModifier.getAdGroupId())
                .withCampaignId(bidModifier.getCampaignId())
                .withEnabled(bidModifier.getEnabled())
                .withId(bidModifier.getId())
                .withType(gdBidModifierType);
    }

    private static GdBidModifier fromLegacyCoreModels(BidModifier bidModifier, CampaignType campaignType) {
        switch (bidModifier.getType()) {
            case AB_SEGMENT_MULTIPLIER:
                return new GdBidModifierABSegment().withAdjustments(
                        toGdAdjustments(((BidModifierABSegment) bidModifier).getAbSegmentAdjustments(),
                                BidModifierDataConverter::toGdBidModifierABSegmentAdjustment));

            case BANNER_TYPE_MULTIPLIER:
                return new GdBidModifierAdType().withAdjustments(
                        toGdAdjustments(((BidModifierBannerType) bidModifier).getBannerTypeAdjustments(),
                                BidModifierDataConverter::toGdBidModifierAdTypeAdjustment));

            case DEMOGRAPHY_MULTIPLIER:
                return new GdBidModifierDemographics().withAdjustments(
                        toGdAdjustments(((BidModifierDemographics) bidModifier).getDemographicsAdjustments(),
                                BidModifierDataConverter::toGdBidModifierDemographicsAdjustment));

            case DESKTOP_MULTIPLIER:
                return new GdBidModifierDesktop().withAdjustments(
                        toGdAdjustments(singletonList(((BidModifierDesktop) bidModifier).getDesktopAdjustment()),
                                BidModifierDataConverter::toGdBidModifierDesktopAdjustment));

            case DESKTOP_ONLY_MULTIPLIER:
                return new GdBidModifierDesktopOnly().withAdjustments(
                        toGdAdjustments(singletonList(((BidModifierDesktopOnly) bidModifier).getDesktopOnlyAdjustment()),
                                BidModifierDataConverter::toGdBidModifierDesktopOnlyAdjustment));

            case SMARTTV_MULTIPLIER:
                return new GdBidModifierSmartTV().withAdjustments(
                        toGdAdjustments(singletonList(((BidModifierSmartTV) bidModifier).getSmartTVAdjustment()),
                                BidModifierDataConverter::toGdBidModifierSmartTVAdjustment));

            case GEO_MULTIPLIER:
                return new GdBidModifierGeo().withAdjustments(
                        toGdAdjustments(((BidModifierGeo) bidModifier).getRegionalAdjustments(), adjustment ->
                                convertToGd(adjustment, BidModifierDataConverter::toGdBidModifierRegionalAdjustment)));

            case INVENTORY_MULTIPLIER:
                Map<InventoryType, BidModifierInventoryAdjustment> adjustments = listToMap(
                        ((BidModifierInventory) bidModifier).getInventoryAdjustments(),
                        BidModifierInventoryAdjustment::getInventoryType);
                if (campaignType == CampaignType.CPM_BANNER) {
                    // Для медийных кампаний показываем только inapp
                    if (adjustments.containsKey(InventoryType.INBANNER) && adjustments.containsKey(InventoryType.INAPP)) {
                        adjustments.remove(InventoryType.INBANNER);
                    } else if (adjustments.containsKey(InventoryType.INBANNER) && !adjustments.containsKey(InventoryType.INAPP)) {
                        // Если в БД есть inbanner, и нет inapp, то меняем тип inbanner на inapp
                        adjustments.get(InventoryType.INBANNER).setInventoryType(InventoryType.INAPP);
                    }
                }
                return new GdBidModifierInventory().withAdjustments(
                        toGdAdjustments(adjustments.values().stream().collect(Collectors.toList()),
                                BidModifierDataConverter::toGdBidModifierInventoryAdjustment));

            case MOBILE_MULTIPLIER:
                return new GdBidModifierMobile().withAdjustments(
                        toGdAdjustments(singletonList(((BidModifierMobile) bidModifier).getMobileAdjustment()),
                                BidModifierDataConverter::toGdBidModifierMobileAdjustment));

            case TABLET_MULTIPLIER:
                return new GdBidModifierTablet().withAdjustments(
                        toGdAdjustments(singletonList(((BidModifierTablet) bidModifier).getTabletAdjustment()),
                                BidModifierDataConverter::toGdBidModifierTabletAdjustment));

            case PERFORMANCE_TGO_MULTIPLIER:
                return new GdBidModifierSmartTgo().withAdjustments(
                        toGdAdjustments(
                                singletonList(((BidModifierPerformanceTgo) bidModifier).getPerformanceTgoAdjustment()),
                                BidModifierDataConverter::toGdBidModifierSmartTgoAdjustment));

            case RETARGETING_MULTIPLIER:
                return new GdBidModifierRetargeting().withAdjustments(
                        toGdAdjustments(((BidModifierRetargeting) bidModifier).getRetargetingAdjustments(),
                                BidModifierDataConverter::toGdBidModifierRetargetingAdjustment));

            case VIDEO_MULTIPLIER:
                return new GdBidModifierVideo().withAdjustments(
                        toGdAdjustments(singletonList(((BidModifierVideo) bidModifier).getVideoAdjustment()),
                                BidModifierDataConverter::toGdBidModifierVideoAdjustment));

            case WEATHER_MULTIPLIER:
                return new GdBidModifierWeather().withAdjustments(
                        toGdAdjustments(((BidModifierWeather) bidModifier).getWeatherAdjustments(),
                                BidModifierDataConverter::toGdBidModifierWeatherAdjustment));

            case TRAFARET_POSITION_MULTIPLIER:
                return new GdBidModifierTrafaretPosition().withAdjustments(
                        toGdAdjustments(((BidModifierTrafaretPosition) bidModifier).getTrafaretPositionAdjustments(),
                                adjustment ->
                                        convertToGd(adjustment,
                                                BidModifierDataConverter::toGdBidModifierTrafaretPositionAdjustment)));

            case DISTANCE_MULTIPLIER:
                throw new IllegalArgumentException("Unsupported core bid modifier type: " + bidModifier.getType());
            default:
                throw new IllegalArgumentException("Unknown core bid modifier type: " + bidModifier.getType());
        }
    }

    private static GdBidModifier toGdBidModExpress(BidModifierExpression coreBidModifier) {
        return new GdBidModifierExpress().withAdjustments(
                toGdAdjustments(coreBidModifier.getExpressionAdjustments(),
                        adj -> toGdBidModExpressAdjustment(adj, coreBidModifier.getType())
                ));
    }

    private static GdBidModifierExpressAdjustment toGdBidModExpressAdjustment(
            BidModifierExpressionAdjustment bidModifierAdjustment,
            BidModifierType type
    ) {
        return new GdBidModifierExpressAdjustment().withCondition(
                mapList(
                        bidModifierAdjustment.getCondition(),
                        list -> toGdBidModExpressConditionList(list, type)
                ));
    }

    private static List<GdBidModifierExpressLiteral> toGdBidModExpressConditionList(
            List<BidModifierExpressionLiteral> conditionList,
            BidModifierType type
    ) {
        return mapList(conditionList, lit -> toGdBidModExpressLiteral(lit, type));
    }

    private static GdBidModifierExpressLiteral toGdBidModExpressLiteral(
            BidModifierExpressionLiteral bidModifierWeatherLiteral,
            BidModifierType type
    ) {
        var gdBidModifierExpressLiteral = new GdBidModifierExpressLiteral()
                .withOperation(bidModifierWeatherLiteral.getOperation())
                .withParameter(bidModifierWeatherLiteral.getParameter());

        var param = bidModifierWeatherLiteral.getParameter();
        var parameterInfo = Constants.ALL_PARAMETERS.get(type).get(param);
        checkNotNull(parameterInfo);
        if (parameterInfo.getType() == ParameterType.INTEGER) {
            gdBidModifierExpressLiteral.withValue(bidModifierWeatherLiteral.getValueInteger().toString());
        } else {
            gdBidModifierExpressLiteral.withValue(bidModifierWeatherLiteral.getValueString());
        }
        return gdBidModifierExpressLiteral;
    }

    private static List<GdBidModifierWeatherExpression> toGdBidModifierWeatherExpressionList(
            List<BidModifierWeatherLiteral> expressionList) {
        return mapList(expressionList, BidModifierDataConverter::toGdBidModifierWeatherExpression);
    }

    private static GdBidModifierWeatherExpression toGdBidModifierWeatherExpression(
            BidModifierWeatherLiteral bidModifierWeatherLiteral) {
        return new GdBidModifierWeatherExpression()
                .withOperation(toGdOperationType(bidModifierWeatherLiteral.getOperation()))
                .withParameter(toGdWeatherType(bidModifierWeatherLiteral.getParameter()))
                .withValue(bidModifierWeatherLiteral.getValue());
    }

    static GdWeatherType toGdWeatherType(WeatherType weatherType) {
        return GdWeatherType.fromSource(weatherType);
    }

    static GdOperationType toGdOperationType(OperationType operation) {
        return GdOperationType.fromSource(operation);
    }

    @Nullable
    public static GdGenderType toGdGender(@Nullable GenderType gender) {
        return ifNotNull(gender, t -> GdGenderType.fromSource(GenderType.toSource(t)));
    }

    @Nullable
    public static GdAgeType toGdAge(@Nullable AgeType age) {
        return ifNotNull(age, t -> GdAgeType.fromSource(AgeType.toSource(t)));
    }

    @Nullable
    public static GdOsType toGdOsType(@Nullable OsType type) {
        return ifNotNull(type, t -> GdOsType.fromSource(OsType.toSource(t)));
    }

    @Nullable
    public static GdOsType toGdOsType(@Nullable TabletOsType type) {
        return ifNotNull(type, t -> {
            if (TabletOsType.IOS.equals(t)) return GdOsType.IOS;
            if (TabletOsType.ANDROID.equals(t)) return GdOsType.ANDROID;
            return null;
        });
    }

    @Nullable
    static GdAdjustmentAdType toGdAdjustmentAdType(@Nullable BannerType type) {
        if (type == null) {
            return null;
        }
        switch (type) {
            case CPM_BANNER:
                return GdAdjustmentAdType.CPM_BANNER;
            case CPM_VIDEO:
            case CPM_OUTDOOR:
            default:
                throw new IllegalArgumentException("Unknown or unsupported banner type: " + type);
        }
    }

    @Nullable
    public static GdInventoryType toGdInventoryType(@Nullable InventoryType type) {
        if (type == null) {
            return null;
        }
        switch (type) {
            case INSTREAM_WEB:
                return GdInventoryType.INSTREAM_WEB;
            case PREROLL:
                return GdInventoryType.PREROLL;
            case MIDROLL:
                return GdInventoryType.MIDROLL;
            case POSTROLL:
                return GdInventoryType.POSTROLL;
            case PAUSEROLL:
                return GdInventoryType.PAUSEROLL;
            case OVERLAY:
                return GdInventoryType.OVERLAY;
            case POSTROLL_OVERLAY:
                return GdInventoryType.POSTROLL_OVERLAY;
            case POSTROLL_WRAPPER:
                return GdInventoryType.POSTROLL_WRAPPER;
            case INROLL_OVERLAY:
                return GdInventoryType.INROLL_OVERLAY;
            case INROLL:
                return GdInventoryType.INROLL;
            case INPAGE:
                return GdInventoryType.INPAGE;
            case INAPP:
                return GdInventoryType.INAPP;
            case INTERSTITIAL:
                return GdInventoryType.INTERSTITIAL;
            case FULLSCREEN:
                return GdInventoryType.FULLSCREEN;
            case INBANNER:
                return GdInventoryType.INBANNER;
            case REWARDED:
                return GdInventoryType.REWARDED;
            default:
                throw new IllegalArgumentException("Unknown or unsupported inventory type: " + type);
        }
    }

    @Nullable
    public static GdTrafaretPosition toGdTrafaretPosition(@Nullable TrafaretPosition trafaretPosition) {
        if (trafaretPosition == null) {
            return null;
        }
        switch (trafaretPosition) {
            case ALONE:
                return GdTrafaretPosition.ALONE;
            case SUGGEST:
                return GdTrafaretPosition.SUGGEST;
            default:
                throw new IllegalArgumentException("Unknown or unsupported trafaret position: " + trafaretPosition);
        }
    }

    public static GdBidModifierType toGdBidModifierType(BidModifierType type) {
        switch (type) {
            case AB_SEGMENT_MULTIPLIER:
                return AB_SEGMENT_MULTIPLIER;
            case BANNER_TYPE_MULTIPLIER:
                return GdBidModifierType.AD_TYPE_MULTIPLIER;
            case DEMOGRAPHY_MULTIPLIER:
                return GdBidModifierType.DEMOGRAPHY_MULTIPLIER;
            case DESKTOP_MULTIPLIER:
                return GdBidModifierType.DESKTOP_MULTIPLIER;
            case DESKTOP_ONLY_MULTIPLIER:
                return GdBidModifierType.DESKTOP_ONLY_MULTIPLIER;
            case SMARTTV_MULTIPLIER:
                return GdBidModifierType.SMARTTV_MULTIPLIER;
            case DISTANCE_MULTIPLIER:
                return GdBidModifierType.DISTANCE_MULTIPLIER;
            case GEO_MULTIPLIER:
                return GdBidModifierType.GEO_MULTIPLIER;
            case INVENTORY_MULTIPLIER:
                return GdBidModifierType.INVENTORY_MULTIPLIER;
            case MOBILE_MULTIPLIER:
                return GdBidModifierType.MOBILE_MULTIPLIER;
            case TABLET_MULTIPLIER:
                return GdBidModifierType.TABLET_MULTIPLIER;
            case PERFORMANCE_TGO_MULTIPLIER:
                return GdBidModifierType.SMART_TGO_MULTIPLIER;
            case RETARGETING_MULTIPLIER:
                return GdBidModifierType.RETARGETING_MULTIPLIER;
            case VIDEO_MULTIPLIER:
                return GdBidModifierType.VIDEO_MULTIPLIER;
            case WEATHER_MULTIPLIER:
                return GdBidModifierType.WEATHER_MULTIPLIER;
            case EXPRESS_TRAFFIC_MULTIPLIER:
                return GdBidModifierType.EXPRESS_TRAFFIC_MULTIPLIER;
            case EXPRESS_CONTENT_DURATION_MULTIPLIER:
                return GdBidModifierType.EXPRESS_CONTENT_DURATION_MULTIPLIER;
            case TRAFARET_POSITION_MULTIPLIER:
                return GdBidModifierType.TRAFARET_POSITION_MULTIPLIER;
            case PRISMA_INCOME_GRADE_MULTIPLIER:
                return GdBidModifierType.PRISMA_INCOME_GRADE_MULTIPLIER;
            default:
                throw new IllegalArgumentException("Unknown core bid modifier type: " + type);
        }
    }

    public static ComplexBidModifier toComplexBidModifier(
            @Nullable List<GdUpdateAdGroupRetargetingItem> updateRetargetingItems,
            @Nullable GdUpdateBidModifiers gdUpdateBidModifiers,
            @Nullable Long campaignId,
            @Nullable Long adGroupId) {
        var complexBidModifier = toComplexBidModifier(gdUpdateBidModifiers);
        if (updateRetargetingItems == null || updateRetargetingItems.isEmpty()) {
            return complexBidModifier;
        }
        return Objects.requireNonNullElseGet(complexBidModifier, ComplexBidModifier::new)
                .withRetargetingFilterModifier(
                        toBidModifierFromRetargeting(updateRetargetingItems, campaignId, adGroupId));
    }

    public static ComplexBidModifier toComplexBidModifier(
            @Nullable GdUpdateBidModifiers gdUpdateBidModifiers) {
        if (gdUpdateBidModifiers == null) {
            return null;
        }
        return new ComplexBidModifier()
                .withAbSegmentModifier(
                        (BidModifierABSegment) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierABSegment()))
                .withBannerTypeModifier(
                        (BidModifierBannerType) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierAdType()))
                .withDemographyModifier(
                        (BidModifierDemographics) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierDemographics()))
                .withDesktopModifier(
                        (BidModifierDesktop) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierDesktop()))
                .withDesktopOnlyModifier(
                        (BidModifierDesktopOnly) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierDesktopOnly()))
                .withSmartTVModifier(
                        (BidModifierSmartTV) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierSmartTV()))
                .withGeoModifier(
                        (BidModifierGeo) toBidModifierImplementation(gdUpdateBidModifiers.getBidModifierGeo()))
                .withInventoryModifier(
                        (BidModifierInventory) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierInventory()))
                .withMobileModifier(
                        (BidModifierMobile) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierMobile()))
                .withTabletModifier(
                        (BidModifierTablet) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierTablet()))
                .withPerformanceTgoModifier(
                        (BidModifierPerformanceTgo) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierSmartTgo()))
                .withRetargetingModifier(
                        (BidModifierRetargeting) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierRetargeting()))
                .withVideoModifier(
                        (BidModifierVideo) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierVideo()))
                .withWeatherModifier(
                        (BidModifierWeather) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierWeather()))
                .withTrafaretPositionModifier(
                        (BidModifierTrafaretPosition) toBidModifierImplementation(
                                gdUpdateBidModifiers.getBidModifierTrafaretPosition()))
                .withExpressionModifiers(
                        mapList(
                                gdUpdateBidModifiers.getBidModifierExpress(),
                                BidModifierDataConverter::toCoreBidModifierExpress
                        )
                );
    }

    @Nullable
    public static List<BidModifier> toBidModifiers(@Nullable GdUpdateBidModifiers gdUpdateBidModifiers) {
        if (gdUpdateBidModifiers == null) {
            return null;
        }
        BidModifier abSegmentBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierABSegment());
        BidModifier adTypeBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierAdType());
        BidModifier demographicsBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierDemographics());
        BidModifier desktopBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierDesktop());
        BidModifier desktopOnlyBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierDesktopOnly());
        BidModifier inventoryBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierInventory());
        BidModifier mobileBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierMobile());
        BidModifier tabletBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierTablet());
        BidModifier smartTgoBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierSmartTgo());
        BidModifier retargetingBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierRetargeting());
        BidModifier videoBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierVideo());
        BidModifier weatherBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierWeather());
        BidModifier geoBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierGeo());
        BidModifier trafaretPositionBidModifier = toBidModifierImplementation(
                gdUpdateBidModifiers.getBidModifierTrafaretPosition());
        List<BidModifierExpression> expressionBidModifiers = mapList(
                gdUpdateBidModifiers.getBidModifierExpress(),
                BidModifierDataConverter::toCoreBidModifierExpress
        );
        List<BidModifier> result = new ArrayList<>();
        addModifierIfNotNull(result, abSegmentBidModifier);
        addModifierIfNotNull(result, adTypeBidModifier);
        addModifierIfNotNull(result, demographicsBidModifier);
        addModifierIfNotNull(result, desktopBidModifier);
        addModifierIfNotNull(result, desktopOnlyBidModifier);
        addModifierIfNotNull(result, inventoryBidModifier);
        addModifierIfNotNull(result, mobileBidModifier);
        addModifierIfNotNull(result, tabletBidModifier);
        addModifierIfNotNull(result, smartTgoBidModifier);
        addModifierIfNotNull(result, retargetingBidModifier);
        addModifierIfNotNull(result, videoBidModifier);
        addModifierIfNotNull(result, weatherBidModifier);
        addModifierIfNotNull(result, geoBidModifier);
        addModifierIfNotNull(result, trafaretPositionBidModifier);
        if (expressionBidModifiers != null) {
            result.addAll(expressionBidModifiers);
        }
        return result;
    }

    private static void addModifierIfNotNull(List<BidModifier> bidBodifiers, @Nullable BidModifier bidModifierToAdd) {
        if (bidModifierToAdd != null) {
            bidBodifiers.add(bidModifierToAdd);
        }
    }

    public static BidModifierRetargetingFilter toBidModifierFromRetargeting(List<GdUpdateAdGroupRetargetingItem> items,
                                                                            @Nullable Long campaignId,
                                                                            @Nullable Long adGroupId) {
        return new BidModifierRetargetingFilter()
                .withType(BidModifierType.RETARGETING_FILTER)
                .withCampaignId(campaignId)
                .withAdGroupId(adGroupId)
                .withRetargetingAdjustments(mapList(items,
                        BidModifierDataConverter::toBidModifierRetargetingFilterAdjustmentFromRetargeting));
    }


    private static BidModifier toBidModifierImplementation(@Nullable GdUpdateBidModifiersItem item) {
        if (item == null) {
            return null;
        }

        BidModifier bidModifier;
        switch (item.getType()) {
            case AB_SEGMENT_MULTIPLIER:
                bidModifier = new BidModifierABSegment().withAbSegmentAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierABSegment) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierABSegmentAdjustment));
                break;
            case AD_TYPE_MULTIPLIER:
                bidModifier = new BidModifierBannerType().withBannerTypeAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierAdType) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierBannerTypeAdjustment));
                break;
            case DEMOGRAPHY_MULTIPLIER:
                bidModifier = new BidModifierDemographics().withDemographicsAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierDemographics) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierDemographicsAdjustment));
                break;
            case DESKTOP_MULTIPLIER:
                bidModifier = new BidModifierDesktop().withDesktopAdjustment(
                        convertToCore(((GdUpdateBidModifierDesktop) item).getAdjustment(),
                                BidModifierDataConverter::toBidModifierDesktopAdjustment));
                break;
            case DESKTOP_ONLY_MULTIPLIER:
                bidModifier = new BidModifierDesktopOnly().withDesktopOnlyAdjustment(
                        convertToCore(((GdUpdateBidModifierDesktopOnly) item).getAdjustment(),
                                BidModifierDataConverter::toBidModifierDesktopOnlyAdjustment));
                break;
            case SMARTTV_MULTIPLIER:
                bidModifier = new BidModifierSmartTV().withSmartTVAdjustment(
                        convertToCore(((GdUpdateBidModifierSmartTV) item).getAdjustment(),
                                BidModifierDataConverter::toBidModifierSmartTVAdjustment));
                break;
            case GEO_MULTIPLIER:
                bidModifier = new BidModifierGeo().withRegionalAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierGeo) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierRegionalAdjustment));
                break;
            case INVENTORY_MULTIPLIER:
                bidModifier = new BidModifierInventory().withInventoryAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierInventory) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierInventoryAdjustment));
                break;
            case MOBILE_MULTIPLIER:
                bidModifier = new BidModifierMobile().withMobileAdjustment(
                        convertToCore(((GdUpdateBidModifierMobile) item).getAdjustment(),
                                BidModifierDataConverter::toBidModifierMobileAdjustment));
                break;
            case TABLET_MULTIPLIER:
                bidModifier = new BidModifierTablet().withTabletAdjustment(
                        convertToCore(((GdUpdateBidModifierTablet) item).getAdjustment(),
                                BidModifierDataConverter::toBidModifierTabletAdjustment));
                break;
            case RETARGETING_MULTIPLIER:
                bidModifier = new BidModifierRetargeting().withRetargetingAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierRetargeting) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierRetargetingAdjustment));
                break;
            case SMART_TGO_MULTIPLIER:
                bidModifier = new BidModifierPerformanceTgo().withPerformanceTgoAdjustment(
                        convertToCore(((GdUpdateBidModifierSmartTgo) item).getAdjustment(),
                                BidModifierDataConverter::toBidModifierPerformanceTgoAdjustment));
                break;
            case VIDEO_MULTIPLIER:
                bidModifier = new BidModifierVideo().withVideoAdjustment(
                        convertToCore(((GdUpdateBidModifierVideo) item).getAdjustment(),
                                BidModifierDataConverter::toBidModifierVideoAdjustment));
                break;
            case WEATHER_MULTIPLIER:
                bidModifier = new BidModifierWeather().withWeatherAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierWeather) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierWeatherAdjustment));
                break;
            case TRAFARET_POSITION_MULTIPLIER:
                bidModifier = new BidModifierTrafaretPosition().withTrafaretPositionAdjustments(
                        toCoreAdjustments((((GdUpdateBidModifierTrafaretPosition) item).getAdjustments()),
                                BidModifierDataConverter::toBidModifierTrafaretPositionAdjustment));
                break;
            case DISTANCE_MULTIPLIER:
                throw new IllegalArgumentException("Unsupported graphql bid modifier type: " + item.getType());
            default:
                throw new IllegalArgumentException("Unknown graphql bid modifier type: " + item.getType());
        }
        return bidModifier.withId(item.getId())
                .withCampaignId(item.getCampaignId())
                .withAdGroupId(item.getAdGroupId())
                .withType(toBidModifierType(item.getType()))
                .withEnabled(item.getEnabled());
    }

    private static BidModifierExpression toCoreBidModifierExpress(GdUpdateBidModifierExpress item) {
        var type = BidModifierType.valueOf(item.getType().toString());
        if (BidModifierExpressionFactory.notExpressionBidModifierType(type)) {
            throw new IllegalArgumentException("Bid modifier type " + type + " is not expression bid modifier type");
        }
        BidModifierExpression bidModifier = BidModifierExpressionFactory.createBidModifierByType(type);
        bidModifier
                .withId(item.getId())
                .withCampaignId(item.getCampaignId())
                .withAdGroupId(item.getAdGroupId())
                .withEnabled(item.getEnabled())
                .withType(type);

        bidModifier.setExpressionAdjustments(toCoreAdjustments(
                item.getAdjustments(),
                adj -> toCoreBidModifierExpressAdjustment(adj, type)
        ));

        return bidModifier;
    }

    private static BidModifierExpressionAdjustment toCoreBidModifierExpressAdjustment(
            GdUpdateBidModifierExpressAdjustmentItem expressAdjustment,
            BidModifierType type
    ) {
        BidModifierExpressionAdjustment adj = BidModifierExpressionFactory.createBidModifierAdjustmentByType(type);
        return adj
                .withCondition(
                        mapList(
                                expressAdjustment.getCondition(),
                                list -> mapList(list, lit -> toCoreBidModifierExpressLiteral(lit, type))
                        )
                );
    }

    private static BidModifierExpressionLiteral toCoreBidModifierExpressLiteral(
            GdUpdateBidModifierExpressLiteral gdLiteral,
            BidModifierType type
    ) {
        var literal = new BidModifierExpressionLiteral()
                .withParameter(gdLiteral.getParameter())
                .withOperation(gdLiteral.getOperation());

        var param = literal.getParameter();
        if (Constants.ALL_PARAMETERS.containsKey(type) && Constants.ALL_PARAMETERS.get(type).containsKey(param)) {
            var paramInfo = Constants.ALL_PARAMETERS.get(type).get(param);
            if (paramInfo.getType() == ParameterType.INTEGER) {
                int value;
                try {
                    value = Integer.parseInt(gdLiteral.getValue());
                } catch (NumberFormatException ex) {
                    throw new IllegalArgumentException("Expected integer value for parameter " + param +
                            ", got '" + gdLiteral.getValue() + "'");
                }
                literal.withValueInteger(value);
            } else {
                literal.withValueString(gdLiteral.getValue());
            }
        }

        return literal;
    }

    private static <T extends BidModifierAdjustment, Y extends GdUpdateBidModifierAdjustmentItem>
    List<T> toCoreAdjustments(List<Y> adjustments, Function<Y, T> function) {
        return mapList(adjustments, adjustment -> convertToCore(adjustment, function));
    }


    private static <T extends BidModifierAdjustment, Y extends GdUpdateBidModifierAdjustmentItem> T convertToCore(
            Y adjustment, Function<Y, T> function) {
        T converted = function.apply(adjustment);
        converted.setId(adjustment.getId());
        converted.setPercent(adjustment.getPercent());
        return converted;
    }

    private static BidModifierABSegmentAdjustment toBidModifierABSegmentAdjustment(
            GdUpdateBidModifierABSegmentAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierABSegmentAdjustment()
                .withAbSegmentRetargetingConditionId(gdBidModifierAdjustment.getAbSegmentRetargetingConditionId())
                .withSectionId(gdBidModifierAdjustment.getSectionId())
                .withSegmentId(gdBidModifierAdjustment.getSegmentId());
    }

    private static BidModifierBannerTypeAdjustment toBidModifierBannerTypeAdjustment(
            GdUpdateBidModifierAdTypeAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierBannerTypeAdjustment().withBannerType(toBannerType(gdBidModifierAdjustment.getAdType()));
    }

    private static BidModifierDemographicsAdjustment toBidModifierDemographicsAdjustment(
            GdUpdateBidModifierDemographicsAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierDemographicsAdjustment()
                .withAge(toAge(gdBidModifierAdjustment.getAge()))
                .withGender(toGender(gdBidModifierAdjustment.getGender()));
    }

    private static BidModifierDesktopAdjustment toBidModifierDesktopAdjustment(
            @SuppressWarnings("unused") GdUpdateBidModifierDesktopAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierDesktopAdjustment();
    }

    private static BidModifierDesktopOnlyAdjustment toBidModifierDesktopOnlyAdjustment(
            @SuppressWarnings("unused") GdUpdateBidModifierDesktopOnlyAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierDesktopOnlyAdjustment();
    }

    private static BidModifierSmartTVAdjustment toBidModifierSmartTVAdjustment(
            @SuppressWarnings("unused") GdUpdateBidModifierSmartTVAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierSmartTVAdjustment();
    }

    private static BidModifierRegionalAdjustment toBidModifierRegionalAdjustment(
            GdUpdateBidModifierRegionalAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierRegionalAdjustment()
                .withRegionId(gdBidModifierAdjustment.getRegionId().longValue())
                .withHidden(nvl(gdBidModifierAdjustment.getHidden(), false));
    }

    private static BidModifierInventoryAdjustment toBidModifierInventoryAdjustment(
            GdUpdateBidModifierInventoryAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierInventoryAdjustment()
                .withInventoryType(toInventoryType(gdBidModifierAdjustment.getInventoryType()));
    }

    private static BidModifierMobileAdjustment toBidModifierMobileAdjustment(
            GdUpdateBidModifierMobileAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierMobileAdjustment().withOsType(toOsType(gdBidModifierAdjustment.getOsType()));
    }

    private static BidModifierTabletAdjustment toBidModifierTabletAdjustment(
            GdUpdateBidModifierTabletAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierTabletAdjustment().withOsType(toTabletOsType(gdBidModifierAdjustment.getOsType()));
    }

    private static BidModifierRetargetingAdjustment toBidModifierRetargetingAdjustment(
            GdUpdateBidModifierRetargetingAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierRetargetingAdjustment()
                .withRetargetingConditionId(gdBidModifierAdjustment.getRetargetingConditionId());
    }

    private static BidModifierRetargetingFilterAdjustment toBidModifierRetargetingFilterAdjustmentFromRetargeting(
            GdUpdateAdGroupRetargetingItem retargetingItem
    ) {
        return new BidModifierRetargetingFilterAdjustment()
                .withId(retargetingItem.getId())
                .withRetargetingConditionId(retargetingItem.getRetCondId())
                .withPercent(0);
    }

    private static BidModifierPerformanceTgoAdjustment toBidModifierPerformanceTgoAdjustment(
            @SuppressWarnings("unused") GdUpdateBidModifierSmartTgoAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierPerformanceTgoAdjustment();
    }

    private static BidModifierVideoAdjustment toBidModifierVideoAdjustment(
            @SuppressWarnings("unused") GdUpdateBidModifierVideoAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierVideoAdjustment();
    }

    private static BidModifierWeatherAdjustment toBidModifierWeatherAdjustment(
            GdUpdateBidModifierWeatherAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierWeatherAdjustment()
                .withExpression(mapList(gdBidModifierAdjustment.getExpression(),
                        BidModifierDataConverter::toGdBidModifierWeatherLiteralList));
    }

    private static BidModifierTrafaretPositionAdjustment toBidModifierTrafaretPositionAdjustment(
            GdUpdateBidModifierTrafaretPositionAdjustmentItem gdBidModifierAdjustment) {
        return new BidModifierTrafaretPositionAdjustment()
                .withTrafaretPosition(toTrafaretPosition(gdBidModifierAdjustment.getTrafaretPosition()));
    }

    private static <T extends GdBidModifierAdjustment, Y extends BidModifierAdjustment> List<T> toGdAdjustments(
            List<Y> adjustments, Function<Y, T> function) {
        return mapList(adjustments, adjustment -> convertToGd(adjustment, function));
    }

    private static <T extends GdBidModifierAdjustment, Y extends BidModifierAdjustment> T convertToGd(
            Y adjustment, Function<Y, T> function) {
        T converted = function.apply(adjustment);
        converted.setId(adjustment.getId());
        converted.setPercent(adjustment.getPercent());
        return converted;
    }

    private static GdBidModifierABSegmentAdjustment toGdBidModifierABSegmentAdjustment(
            BidModifierABSegmentAdjustment bidModifierAdjustment) {
        return new GdBidModifierABSegmentAdjustment()
                .withAbSegmentRetargetingConditionId(bidModifierAdjustment.getAbSegmentRetargetingConditionId())
                .withAccessible(bidModifierAdjustment.getAccessible())
                .withSectionId(bidModifierAdjustment.getSectionId())
                .withSegmentId(bidModifierAdjustment.getSegmentId());
    }

    private static GdBidModifierAdTypeAdjustment toGdBidModifierAdTypeAdjustment(
            BidModifierBannerTypeAdjustment bidModifierAdjustment) {
        return new GdBidModifierAdTypeAdjustment()
                .withAdType(toGdAdjustmentAdType(bidModifierAdjustment.getBannerType()));
    }

    private static GdBidModifierDemographicsAdjustment toGdBidModifierDemographicsAdjustment(
            BidModifierDemographicsAdjustment bidModifierAdjustment) {
        return new GdBidModifierDemographicsAdjustment()
                .withAge(toGdAge(bidModifierAdjustment.getAge()))
                .withGender(toGdGender(bidModifierAdjustment.getGender()));
    }

    private static GdBidModifierDesktopAdjustment toGdBidModifierDesktopAdjustment(
            @SuppressWarnings("unused") BidModifierDesktopAdjustment bidModifierAdjustment) {
        return new GdBidModifierDesktopAdjustment();
    }

    private static GdBidModifierDesktopOnlyAdjustment toGdBidModifierDesktopOnlyAdjustment(
            @SuppressWarnings("unused") BidModifierDesktopOnlyAdjustment bidModifierAdjustment) {
        return new GdBidModifierDesktopOnlyAdjustment();
    }

    private static GdBidModifierSmartTVAdjustment toGdBidModifierSmartTVAdjustment(
            @SuppressWarnings("unused") BidModifierSmartTVAdjustment bidModifierAdjustment) {
        return new GdBidModifierSmartTVAdjustment();
    }

    private static GdBidModifierRegionalAdjustment toGdBidModifierRegionalAdjustment(
            BidModifierRegionalAdjustment bidModifierAdjustment) {
        return new GdBidModifierRegionalAdjustment()
                .withRegionId(bidModifierAdjustment.getRegionId().intValue())
                .withHidden(bidModifierAdjustment.getHidden());
    }

    private static GdBidModifierInventoryAdjustment toGdBidModifierInventoryAdjustment(
            BidModifierInventoryAdjustment bidModifierAdjustment) {
        return new GdBidModifierInventoryAdjustment()
                .withInventoryType(toGdInventoryType(bidModifierAdjustment.getInventoryType()));
    }

    private static GdBidModifierMobileAdjustment toGdBidModifierMobileAdjustment(
            BidModifierMobileAdjustment bidModifierAdjustment) {
        return new GdBidModifierMobileAdjustment().withOsType(toGdOsType(bidModifierAdjustment.getOsType()));
    }

    private static GdBidModifierTabletAdjustment toGdBidModifierTabletAdjustment(
            BidModifierTabletAdjustment bidModifierAdjustment) {
        return new GdBidModifierTabletAdjustment().withOsType(toGdOsType(bidModifierAdjustment.getOsType()));
    }

    private static GdBidModifierRetargetingAdjustment toGdBidModifierRetargetingAdjustment(
            AbstractBidModifierRetargetingAdjustment bidModifierAdjustment) {
        return new GdBidModifierRetargetingAdjustment()
                .withRetargetingConditionId(bidModifierAdjustment.getRetargetingConditionId())
                .withAccessible(bidModifierAdjustment.getAccessible());
    }

    private static GdBidModifierSmartTgoAdjustment toGdBidModifierSmartTgoAdjustment(
            @SuppressWarnings("unused") BidModifierPerformanceTgoAdjustment bidModifierAdjustment) {
        return new GdBidModifierSmartTgoAdjustment();
    }

    private static GdBidModifierVideoAdjustment toGdBidModifierVideoAdjustment(
            @SuppressWarnings("unused") BidModifierVideoAdjustment bidModifierAdjustment) {
        return new GdBidModifierVideoAdjustment();
    }

    private static GdBidModifierWeatherAdjustment toGdBidModifierWeatherAdjustment(
            BidModifierWeatherAdjustment bidModifierAdjustment) {
        return new GdBidModifierWeatherAdjustment()
                .withExpression(mapList(bidModifierAdjustment.getExpression(),
                        BidModifierDataConverter::toGdBidModifierWeatherExpressionList));
    }

    private static GdBidModifierTrafaretPositionAdjustment toGdBidModifierTrafaretPositionAdjustment(
            BidModifierTrafaretPositionAdjustment bidModifierAdjustment) {
        return new GdBidModifierTrafaretPositionAdjustment()
                .withTrafaretPosition(toGdTrafaretPosition(bidModifierAdjustment.getTrafaretPosition()));
    }

    private static List<BidModifierWeatherLiteral> toGdBidModifierWeatherLiteralList(
            @GraphQLNonNull List<@GraphQLNonNull GdUpdateBidModifierWeatherExpression> expressionList) {
        return mapList(expressionList, BidModifierDataConverter::toBidModifierWeatherLiteral);
    }

    private static BidModifierWeatherLiteral toBidModifierWeatherLiteral(
            @GraphQLNonNull GdUpdateBidModifierWeatherExpression gdBidModifierWeatherExpression) {
        return new BidModifierWeatherLiteral()
                .withOperation(toOperationType(gdBidModifierWeatherExpression.getOperation()))
                .withParameter(toWeatherType(gdBidModifierWeatherExpression.getParameter()))
                .withValue(gdBidModifierWeatherExpression.getValue());
    }

    public static List<GdRetargeting> toGdRetargetingsFromRetargetingModifier(
            AbstractBidModifierRetargeting bidModifier) {
        return mapList(bidModifier.getRetargetingAdjustments(), adj -> toGdRetargetingFromRetargetingAdjustment(
                adj,
                bidModifier.getCampaignId(),
                bidModifier.getAdGroupId())
        );
    }

    private static GdRetargeting toGdRetargetingFromRetargetingAdjustment(
            AbstractBidModifierRetargetingAdjustment bidModifierRetargetingAdjustment,
            Long campaignId, Long adGroupId
    ) {
        return new GdRetargeting()
                .withCampaignId(campaignId)
                .withAdGroupId(adGroupId)
                .withRetargetingConditionId(bidModifierRetargetingAdjustment.getRetargetingConditionId())
                .withRetargetingId(bidModifierRetargetingAdjustment.getId())
                .withPriceContext(BigDecimal.ZERO);
    }

    private static WeatherType toWeatherType(GdWeatherType gdWeatherType) {
        return GdWeatherType.toSource(gdWeatherType);
    }

    private static OperationType toOperationType(GdOperationType gdOperation) {
        return GdOperationType.toSource(gdOperation);
    }


    @Nullable
    private static GenderType toGender(@Nullable GdGenderType gender) {
        return ifNotNull(gender, t -> GenderType.fromSource(GdGenderType.toSource(t)));
    }

    @Nullable
    private static AgeType toAge(@Nullable GdAgeType age) {
        return ifNotNull(age, t -> AgeType.fromSource(GdAgeType.toSource(t)));
    }

    @Nullable
    public static OsType toOsType(@Nullable GdOsType type) {
        return ifNotNull(type, t -> OsType.fromSource(GdOsType.toSource(t)));
    }

    @Nullable
    public static TabletOsType toTabletOsType(@Nullable GdOsType type) {
        return ifNotNull(type, t -> {
            if (GdOsType.ANDROID.equals(t)) return TabletOsType.ANDROID;
            if (GdOsType.IOS.equals(t)) return TabletOsType.IOS;
            return null;
        });
    }

    @Nullable
    private static BannerType toBannerType(@Nullable GdAdjustmentAdType type) {
        if (type == null) {
            return null;
        }
        if (type == GdAdjustmentAdType.CPM_BANNER) {
            return BannerType.CPM_BANNER;
        }
        throw new IllegalArgumentException("Unknown or unsupported banner type: " + type);
    }

    @Nullable
    public static InventoryType toInventoryType(@Nullable GdInventoryType type) {
        if (type == null) {
            return null;
        }
        switch (type) {
            case INSTREAM_WEB:
                return InventoryType.INSTREAM_WEB;
            case PREROLL:
                return InventoryType.PREROLL;
            case MIDROLL:
                return InventoryType.MIDROLL;
            case POSTROLL:
                return InventoryType.POSTROLL;
            case PAUSEROLL:
                return InventoryType.PAUSEROLL;
            case OVERLAY:
                return InventoryType.OVERLAY;
            case POSTROLL_OVERLAY:
                return InventoryType.POSTROLL_OVERLAY;
            case POSTROLL_WRAPPER:
                return InventoryType.POSTROLL_WRAPPER;
            case INROLL_OVERLAY:
                return InventoryType.INROLL_OVERLAY;
            case INROLL:
                return InventoryType.INROLL;
            case INPAGE:
                return InventoryType.INPAGE;
            case INAPP:
                return InventoryType.INAPP;
            case INTERSTITIAL:
                return InventoryType.INTERSTITIAL;
            case FULLSCREEN:
                return InventoryType.FULLSCREEN;
            case INBANNER:
                return InventoryType.INBANNER;
            case REWARDED:
                return InventoryType.REWARDED;
            default:
                throw new IllegalArgumentException("Unknown or unsupported inventory type: " + type);
        }
    }

    @Nullable
    public static TrafaretPosition toTrafaretPosition(@Nullable GdTrafaretPosition trafaretPosition) {
        if (trafaretPosition == null) {
            return null;
        }
        switch (trafaretPosition) {
            case ALONE:
                return TrafaretPosition.ALONE;
            case SUGGEST:
                return TrafaretPosition.SUGGEST;
            default:
                throw new IllegalArgumentException("Unknown or unsupported trafaret position: " + trafaretPosition);
        }
    }

    public static BidModifierType toBidModifierType(GdBidModifierType type) {
        switch (type) {
            case AB_SEGMENT_MULTIPLIER:
                return BidModifierType.AB_SEGMENT_MULTIPLIER;
            case AD_TYPE_MULTIPLIER:
                return BidModifierType.BANNER_TYPE_MULTIPLIER;
            case DEMOGRAPHY_MULTIPLIER:
                return BidModifierType.DEMOGRAPHY_MULTIPLIER;
            case DESKTOP_MULTIPLIER:
                return BidModifierType.DESKTOP_MULTIPLIER;
            case DESKTOP_ONLY_MULTIPLIER:
                return BidModifierType.DESKTOP_ONLY_MULTIPLIER;
            case SMARTTV_MULTIPLIER:
                return BidModifierType.SMARTTV_MULTIPLIER;
            case DISTANCE_MULTIPLIER:
                return BidModifierType.DISTANCE_MULTIPLIER;
            case GEO_MULTIPLIER:
                return BidModifierType.GEO_MULTIPLIER;
            case INVENTORY_MULTIPLIER:
                return BidModifierType.INVENTORY_MULTIPLIER;
            case MOBILE_MULTIPLIER:
                return BidModifierType.MOBILE_MULTIPLIER;
            case TABLET_MULTIPLIER:
                return BidModifierType.TABLET_MULTIPLIER;
            case RETARGETING_MULTIPLIER:
                return BidModifierType.RETARGETING_MULTIPLIER;
            case SMART_TGO_MULTIPLIER:
                return BidModifierType.PERFORMANCE_TGO_MULTIPLIER;
            case VIDEO_MULTIPLIER:
                return BidModifierType.VIDEO_MULTIPLIER;
            case WEATHER_MULTIPLIER:
                return BidModifierType.WEATHER_MULTIPLIER;
            case TRAFARET_POSITION_MULTIPLIER:
                return BidModifierType.TRAFARET_POSITION_MULTIPLIER;
            default:
                throw new IllegalArgumentException("Unknown graphql bid modifier type: " + type);
        }
    }
}
