package ru.yandex.direct.grid.processing.service.bidmodifier;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilterAdjustment;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdBidModifier;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargeting;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingOrderBy;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingsContainer;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.group.GroupDataService;

import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_TYPES;
import static ru.yandex.direct.grid.model.Order.DESC;
import static ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingOrderByField.GROUP_ID;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.addAdGroupAndAccessInfo;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

;

@Service
@ParametersAreNonnullByDefault
public class BidModifierDataService {

    private final BidModifierService bidModifierService;
    private final CampaignRepository campaignRepository;
    private final FeatureService featureService;
    private final CampaignInfoService campaignInfoService;
    private final GroupDataService groupDataService;

    private static final Map<Function<GdRetargetingFilter, Set<Long>>,
            Function<BidModifierRetargetingFilterAdjustment, Long>> RETARGETING_FILTER_FIELDS_IN_MAPPER = Map.of(
                    GdRetargetingFilter::getRetargetingIdIn,
                    BidModifierRetargetingFilterAdjustment::getId,
                    GdRetargetingFilter::getRetargetingConditionIdIn,
                    BidModifierRetargetingFilterAdjustment::getRetargetingConditionId
            );

    private static final Map<Function<GdRetargetingFilter, Set<Long>>,
            Function<BidModifierRetargetingFilterAdjustment, Long>> RETARGETING_FILTER_FIELDS_NOT_IN_MAPPER = Map.of(
                    GdRetargetingFilter::getRetargetingIdNotIn,
                    BidModifierRetargetingFilterAdjustment::getId,
                    GdRetargetingFilter::getRetargetingConditionIdNotIn,
                    BidModifierRetargetingFilterAdjustment::getRetargetingConditionId
            );

    @Autowired
    public BidModifierDataService(BidModifierService bidModifierService,
                                  CampaignRepository campaignRepository,
                                  FeatureService featureService,
                                  CampaignInfoService campaignInfoService,
                                  GroupDataService groupDataService) {
        this.bidModifierService = bidModifierService;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
        this.campaignInfoService = campaignInfoService;
        this.groupDataService = groupDataService;
    }

    Map<Long, List<GdBidModifier>> getBidModifiersByAdGroupIds(int shard, Set<Long> adGroupIds) {
        List<BidModifier> byAdGroupIds = bidModifierService.getByAdGroupIds(
                shard,
                adGroupIds,
                ALL_TYPES,
                Collections.singleton(BidModifierLevel.ADGROUP)
        );

        return StreamEx.of(byAdGroupIds)
                .mapToEntry(BidModifier::getAdGroupId, BidModifierDataConverter::toGdBidModifierImplementation)
                .grouping();
    }

    Map<Long, List<GdBidModifier>> getBidModifiersByCampaignIds(int shard, Set<Long> campaignIds) {
        List<BidModifier> byCampaignIds = bidModifierService.getByCampaignIds(
                shard,
                campaignIds,
                ALL_TYPES,
                Collections.singleton(BidModifierLevel.CAMPAIGN)
        );

        Map<Long, CampaignType> campaignTypeMap = campaignRepository.getCampaignsTypeMap(shard, campaignIds);
        return StreamEx.of(byCampaignIds)
                .mapToEntry(BidModifier::getCampaignId,
                        bidModifier -> BidModifierDataConverter.toGdBidModifierImplementation(
                                bidModifier, campaignTypeMap.get(bidModifier.getCampaignId())))
                .grouping();
    }

    public List<GdRetargeting> getSearchRetargetingsFromRetargetingFilterModifiers(GridGraphQLContext context,
                                                                                   GdRetargetingsContainer input) {
        GdClientInfo clientInfo = context.getQueriedClient();
        ClientId clientId = ClientId.fromLong(clientInfo.getId());

        boolean isNewRetargetingEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.SEARCH_RETARGETING_ENABLED);
        if (!isNewRetargetingEnabled) {
            return Collections.emptyList();
        }

        var filter = input.getFilter();
        List<BidModifierRetargetingFilter> bidModifiersResult =
                bidModifierService.getRetargetingFilterModifier(
                        clientId,
                        filter.getRetargetingIdIn(),
                        filter.getAdGroupIdIn(),
                        filter.getCampaignIdIn(),
                        context.getOperator().getUid()
                );
        bidModifiersResult = applyRemainingFilters(filter, bidModifiersResult);

        var gdRetargetingsByModifiers =
                mapList(bidModifiersResult, BidModifierDataConverter::toGdRetargetingsFromRetargetingModifier);
        List<GdRetargeting> retargetingsResult = new ArrayList<>();
        gdRetargetingsByModifiers.forEach(retargetingsResult::addAll);

        //получаем информацию о существующих кампаниях и группах
        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService
                .getTruncatedCampaigns(clientId, listToSet(retargetingsResult, GdRetargeting::getCampaignId));
        var adGroups = groupDataService.getTruncatedAdGroups(
                clientInfo.getShard(), clientInfo.getCountryRegionId(), clientId, context.getOperator(),
                listToSet(retargetingsResult, GdRetargeting::getAdGroupId), campaignsById);
        Map<Long, GdAdGroupTruncated> adGroupsById = listToMap(adGroups, GdAdGroupTruncated::getId);
        retargetingsResult = filterList(retargetingsResult,
                retargeting -> campaignsById.containsKey(retargeting.getCampaignId())
                        && adGroupsById.containsKey(retargeting.getAdGroupId()));
        retargetingsResult.forEach(r ->
                addAdGroupAndAccessInfo(r, campaignsById.get(r.getCampaignId()), adGroupsById.get(r.getAdGroupId())));

        //сортируем если возможно(на данный момент только по GROUP_ID)
        sortByGroupIdIfPossible(input.getOrderBy(), retargetingsResult);

        return retargetingsResult;
    }

    private List<BidModifierRetargetingFilter> applyRemainingFilters(GdRetargetingFilter filter,
                                                                     List<BidModifierRetargetingFilter> bidModifiers) {
        bidModifiers.forEach(b -> b.setRetargetingAdjustments(filterList(
                b.getRetargetingAdjustments(), adj ->
                        checkFieldsFilters(filter, (BidModifierRetargetingFilterAdjustment) adj))
        ));
        return filterList(bidModifiers, bidModifier -> !bidModifier.getRetargetingAdjustments().isEmpty());
    }

    private boolean checkFieldsFilters(GdRetargetingFilter filter,
                                       BidModifierRetargetingFilterAdjustment adjustment) {
        return RETARGETING_FILTER_FIELDS_IN_MAPPER.keySet().stream().allMatch((key) -> {
                    var fieldFilter = key.apply(filter);
                    var value = RETARGETING_FILTER_FIELDS_IN_MAPPER.get(key);
                    return fieldFilter == null || fieldFilter.isEmpty()
                            || fieldFilter.contains(value.apply(adjustment));
                })
                &&
                RETARGETING_FILTER_FIELDS_NOT_IN_MAPPER.keySet().stream().allMatch((key) -> {
                    var fieldFilter = key.apply(filter);
                    var value = RETARGETING_FILTER_FIELDS_IN_MAPPER.get(key);
                    return fieldFilter == null || fieldFilter.isEmpty()
                            || !fieldFilter.contains(value.apply(adjustment));
                });
    }

    private void sortByGroupIdIfPossible(List<GdRetargetingOrderBy> orderByList,
                                         List<GdRetargeting> retargetings) {
        if (orderByList.size() != 1) return;
        var orderBy = orderByList.get(0);
        if (orderBy.getField().equals(GROUP_ID)) {
            retargetings.sort(orderBy.getOrder().equals(DESC)
                    ? Comparator.comparingLong(GdRetargeting::getAdGroupId).reversed()
                    : Comparator.comparingLong(GdRetargeting::getAdGroupId));
        }
    }

}
