package ru.yandex.direct.grid.processing.service.brandlift;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections4.ListUtils;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.brandSurvey.BrandSurvey;
import ru.yandex.direct.core.entity.brandlift.repository.BrandSurveyRepository;
import ru.yandex.direct.core.entity.campaign.model.BasicUplift;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignBudgetReach;
import ru.yandex.direct.core.entity.campaign.model.SurveyStatus;
import ru.yandex.direct.core.entity.campaign.repository.CampaignBrandSurveyYtRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignBudgetReachDailyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.model.campaign.GdCampaignBrandSurvey;
import ru.yandex.direct.grid.model.campaign.GdCampaignBrandSurveyInfo;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdGetBrandSurveys;
import ru.yandex.direct.grid.processing.model.campaign.GdGetBrandSurveysFilter;
import ru.yandex.direct.grid.processing.model.campaign.GdGetBrandSurveysPayload;

import static ru.yandex.direct.feature.FeatureName.BRAND_LIFT_HIDDEN;
import static ru.yandex.direct.grid.core.entity.campaign.repository.GridCampaignRepository.convertBrandSurveyStatus;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Получение информации об опросах взгляда
 */
@Service
public class BrandSurveysInfoService {

    private final BrandSurveyRepository brandSurveyRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignBrandSurveyYtRepository brandSurveyYtRepository;
    private final ShardHelper shardHelper;
    private final CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository;
    private final FeatureService featureService;
    private final PpcPropertiesSupport ppcPropertySupport;

    private final static Set<SurveyStatus> STATUSES_CAN_ADD_CAMP = Set.of(
            SurveyStatus.ACTIVE,
            SurveyStatus.DRAFT,
            SurveyStatus.MODERATION
    );
    private final static BrandSurveyStatus UNKNOWN_BRAND_SURVEY_STATUS =
            new BrandSurveyStatus()
                    .withSurveyStatusDaily(SurveyStatus.DRAFT)
                    .withReasonIds(Collections.emptyList())
                    .withBrandSurveyStopReasonsDaily(Collections.emptySet())
                    .withSumSpentByTotalPeriod(BigDecimal.ZERO)
                    .withSumSpentByDay(BigDecimal.ZERO)
                    .withBasicUplift(new BasicUplift());


    public BrandSurveysInfoService(BrandSurveyRepository brandSurveyRepository,
                                   CampaignRepository campaignRepository,
                                   CampaignBrandSurveyYtRepository brandSurveyYtRepository,
                                   ShardHelper shardHelper,
                                   CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository,
                                   FeatureService featureService,
                                   PpcPropertiesSupport ppcPropertySupport) {
        this.brandSurveyRepository = brandSurveyRepository;
        this.campaignRepository = campaignRepository;
        this.brandSurveyYtRepository = brandSurveyYtRepository;
        this.shardHelper = shardHelper;
        this.campaignBudgetReachDailyRepository = campaignBudgetReachDailyRepository;
        this.featureService = featureService;
        this.ppcPropertySupport = ppcPropertySupport;
    }

    public GdGetBrandSurveysPayload getBrandSurveysWithCampaigns(ClientId clientId, GridGraphQLContext context,
                                                                 GdGetBrandSurveys input) {
        boolean brandLiftHiddenEnabledForOperator = featureService.isEnabled(context.getOperator().getUid(),
                BRAND_LIFT_HIDDEN);
        var shard = shardHelper.getShardByClientId(clientId);
        List<GdCampaignBrandSurvey> brandSurveys = getBrandSurveysWithCampaignsForShard(clientId,
                input, shard, brandLiftHiddenEnabledForOperator);

        var operatorShard = shardHelper.getShardByClientId(context.getOperator().getClientId());
        List<GdCampaignBrandSurvey> brandSurveysForOperatorShard = new ArrayList<>();
        if (operatorShard != shard) {
            brandSurveysForOperatorShard = getBrandSurveysWithCampaignsForShard(clientId,
                    input, operatorShard, brandLiftHiddenEnabledForOperator);
        }

        List<GdCampaignBrandSurvey> rowset = ListUtils.union(brandSurveys, brandSurveysForOperatorShard);

        return new GdGetBrandSurveysPayload()
                .withRowset(rowset)
                .withTotalCount(rowset.size());

    }

    private List<GdCampaignBrandSurvey> getBrandSurveysWithCampaignsForShard(ClientId clientId,
                                                                             GdGetBrandSurveys input, Integer shard,
                                                                             Boolean brandLiftHiddenEnabledForOperator) {
        var brandSurveys = filterList(
                brandSurveyRepository.getClientBrandSurveys(shard, clientId.asLong()),
                bl -> !bl.getIsBrandLiftHidden() || brandLiftHiddenEnabledForOperator);
        List<String> brandSurveyIds = brandSurveys.stream()
                .map(BrandSurvey::getBrandSurveyId)
                .collect(Collectors.toList());
        var campaigns = campaignRepository
                .getCampaignsForBrandSurveys(shard, clientId, brandSurveyIds);
        LocalDate brandSurveyBudgetDatePropVal = ppcPropertySupport.get(PpcPropertyNames.BRAND_SURVEY_BUDGET_DATE).get();
        // старые и открепленные от всех РК Brand Lift нельзя выбрать в списке запущенных
        // кроме Brand Lift, прикрепленного к РК из input.
        brandSurveys = filterList(brandSurveys,
                bs -> {
                    var camps = campaigns.get(bs.getBrandSurveyId());
                    if (camps == null) return false;
                    return brandSurveyBudgetDatePropVal == null
                            || brandSurveyBudgetDatePropVal.atStartOfDay().isAfter(LocalDateTime.now())
                            || camps.stream().anyMatch(
                                    camp -> camp.getCreateTime().isAfter(brandSurveyBudgetDatePropVal.atStartOfDay()) ||
                                            (camp.getId().equals(input.getCurrentCampaignId())));
                });

        var statuses = brandSurveyYtRepository.getStatusForBrandSurveys(brandSurveyIds);
        List<Long> cids = campaigns.values().stream()
                .flatMap(List::stream)
                .map(Campaign::getId)
                .collect(Collectors.toList());
        Map<Long, CampaignBudgetReach> campaignBudgets = campaignBudgetReachDailyRepository
                .getCampaignBudgetReachForCampaigns(shard, cids);
        Map<BrandSurvey, List<CampaignBudgetReach>> campaignBudgetsForBs = brandSurveys.stream()
                .collect(Collectors.toMap(
                        i -> i,
                        i -> nvl(campaigns.get(i.getBrandSurveyId()), Collections.<Campaign>emptyList()).stream()
                                .map(Campaign::getId)
                                .map(campaignBudgets::get)
                                .filter(Objects::nonNull)
                                .collect(Collectors.toList())
                ));

        return convertToGd(filterBrandSurveys(brandSurveys, statuses, input.getFilter()),
                campaigns, statuses, campaignBudgetsForBs);
    }

    private static List<BrandSurvey> filterBrandSurveys(List<BrandSurvey> brandSurveys,
                                                        Map<String, BrandSurveyStatus> statuses,
                                                        GdGetBrandSurveysFilter filter) {
        return nvl(filter.getAddCampMode(), false) ?
                brandSurveys.stream()
                        .filter(brandSurvey -> STATUSES_CAN_ADD_CAMP
                                .contains(nvl(
                                        statuses.get(brandSurvey.getBrandSurveyId()),
                                        UNKNOWN_BRAND_SURVEY_STATUS
                                ).getSurveyStatusDaily())
                        )
                        .collect(Collectors.toList())
                : brandSurveys;
    }

    private static List<GdCampaignBrandSurvey> convertToGd(List<BrandSurvey> brandSurveys,
                                                           Map<String, List<Campaign>> campaigns,
                                                           Map<String, BrandSurveyStatus> statuses,
                                                           Map<BrandSurvey, List<CampaignBudgetReach>> campaignBudgetsForBs) {
        return brandSurveys.stream()
                .map(brandSurvey -> {
                            var sumSpentByDay = campaignBudgetsForBs.get(brandSurvey).stream()
                                    .map(CampaignBudgetReach::getBudgetSpent)
                                    .reduce(BigDecimal::add)
                                    .orElse(BigDecimal.ZERO);
                            var gdBrandSurveyStatus = convertBrandSurveyStatus(
                                    nvl(statuses.get(brandSurvey.getBrandSurveyId()), UNKNOWN_BRAND_SURVEY_STATUS))
                                    .withSumSpentByDay(sumSpentByDay);

                            return new GdCampaignBrandSurvey()
                                    .withBrandSurveyId(brandSurvey.getBrandSurveyId())
                                    .withName(brandSurvey.getName())
                                    .withStatus(gdBrandSurveyStatus)
                                    .withCampaigns(nvl(campaigns.get(brandSurvey.getBrandSurveyId()),
                                            Collections.<Campaign>emptyList()).stream()
                                            .map(campaign -> new GdCampaignBrandSurveyInfo()
                                                    .withId(campaign.getId())
                                                    .withName(campaign.getName()))
                                            .collect(Collectors.toList())
                                    );
                        }
                )
                .collect(Collectors.toList());
    }

}
