package ru.yandex.direct.grid.processing.service.cache;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.grid.processing.service.cache.storage.GridCacheStorageHelper;
import ru.yandex.direct.grid.processing.service.cache.util.CacheKeyGenerator;
import ru.yandex.direct.grid.processing.service.cache.util.CacheUtils;
import ru.yandex.direct.grid.processing.service.cache.util.ChunkRef;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toList;

/**
 * Сервис, осуществляющий запись/чтение из кеша, если это требуется, концептуальное описание см. в README.md
 */
@Component
@ParametersAreNonnullByDefault
public class GridCacheService {
    private static final Logger logger = LoggerFactory.getLogger(GridCacheService.class);

    /**
     * для кеширования разбивать массив строк на кусочки такого размера
     */
    private static final int ROWSET_CHUNK_SIZE = 500;

    private final GridCacheStorageHelper storageHelper;
    private final CacheKeyGenerator cacheKeyGenerator = new CacheKeyGenerator(20);

    public GridCacheService(GridCacheStorageHelper storageHelper) {
        this.storageHelper = storageHelper;
    }

    /**
     * Получить из кеша готовый объект, уже с выбранным диапазоном строк.
     */
    public <R, F extends CacheFilterData, D extends CachedGridData<R>> Optional<D> getFromCache(
            CacheRecordInfo<R, F, D> recordInfo,
            LimitOffset limitOffset
    ) {
        checkNotNull(recordInfo.getFilter());
        checkArgument(recordInfo.getClientId() > 0);
        if (recordInfo.getKey() == null) {
            return Optional.empty();
        }

        Optional<? extends CacheRecordInfo> infoOpt =
                storageHelper.getObject(recordInfo.infoKey(), recordInfo.getClass());
        if (!infoOpt.isPresent()) {
            return Optional.empty();
        }
        CacheRecordInfo cachedInfo = infoOpt.get();

        if (!Objects.equals(recordInfo.getFilter(), cachedInfo.getFilter())) {
            logger.warn("Incorrect cached filter, current: " + recordInfo.getFilter() + ", cached: " + cachedInfo
                    .getFilter());
            return Optional.empty();
        }

        List<R> ret = new ArrayList<>();
        for (ChunkRef ch : CacheUtils
                .calculateChunkRefs(limitOffset, cachedInfo.getChunkSize(), cachedInfo.getTotalSize())) {
            @SuppressWarnings("unchecked")
            Optional<List<R>> chunk = (Optional<List<R>>)
                    storageHelper.getObjectList(cachedInfo.chunkKey(ch.idx()), cachedInfo.getRowClass(), ch.begin(),
                            ch.end());
            if (!chunk.isPresent()) {
                logger.warn("Chunk not found: " + ch.idx());
                return Optional.empty();
            }
            ret.addAll(chunk.get());
        }

        @SuppressWarnings("unchecked")
        D data = (D) cachedInfo.getData();
        data.setRowset(ret);

        return Optional.of(data);
    }

    /**
     * Записать в кеш данные
     *
     * @return - cacheKey, для дальнейшего доступа к данным в кеше
     */
    private <R, F extends CacheFilterData, D extends CachedGridData<R>> String saveToCache(
            CacheRecordInfo<R, F, D> recordInfo,
            D data,
            List<R> rowsetFull) {
        checkArgument(recordInfo.getClientId() > 0, "Invalid clientId");
        checkArgument(recordInfo.getFilter() != null, "Filter must be not null");

        String cacheKey = cacheKeyGenerator.generate();
        data.setCacheKey(cacheKey);
        recordInfo.setKey(cacheKey);
        recordInfo.setTotalSize(rowsetFull.size());
        recordInfo.setChunkSize(ROWSET_CHUNK_SIZE);
        recordInfo.setData(data);

        storageHelper.saveObject(recordInfo.infoKey(), recordInfo);

        List<List<R>> chunks = Lists.partition(rowsetFull, ROWSET_CHUNK_SIZE);
        for (int i = 0; i < chunks.size(); i++) {
            storageHelper.saveObjectList(recordInfo.chunkKey(i), recordInfo.getRowClass(), chunks.get(i));
        }

        return cacheKey;
    }

    /**
     * Вырезать из rowsetFull нужный диапазон строк и сохранить в кеш, если это требуется.
     * Переопределяет у recordInfo поля data и rowset
     *
     * @param saveToCache сохранять ли в кеш данные
     * @return - параметр data, в который записан нужный диапазон строк и cacheKey
     */
    public <R, F extends CacheFilterData, D extends CachedGridData<R>> D getResultAndSaveToCacheIfRequested(
            CacheRecordInfo<R, F, D> recordInfo,
            D data,
            List<R> rowsetFull,
            LimitOffset limitOffset,
            boolean saveToCache) {
        if (saveToCache) {
            data.setCacheKey(saveToCache(recordInfo, data, rowsetFull));
        }
        List<R> rowset = rowsetFull.stream()
                .skip(limitOffset.offset()).limit(limitOffset.limit())
                .collect(toList());
        data.setRowset(rowset);
        return data;
    }

    /**
     * Вырезать из rowsetFull нужный диапазон строк и сохранить в кеш.
     *
     * @return - параметр data, в который записан нужный диапазон строк и cacheKey
     */
    public <R, F extends CacheFilterData, D extends CachedGridData<R>> D getResultAndSaveToCache(
            CacheRecordInfo<R, F, D> recordInfo,
            D data,
            List<R> rowsetFull,
            LimitOffset limitOffset) {
        return getResultAndSaveToCacheIfRequested(recordInfo, data, rowsetFull, limitOffset, true);
    }
}
