package ru.yandex.direct.grid.processing.service.cache.storage;

import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.CollectionType;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

/**
 * Сериализатор для сохранения объектов в кеш
 */
public class CacheSerializer {
    private static final ObjectMapper MAPPER = new ObjectMapper()
            .registerModule(new JavaTimeModule())
            .registerModule(new KotlinModule())
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private CacheSerializer() {}

    public static <T> byte[] serialize(T obj) {
        try (TraceProfile ignored = Trace.current().profile("cache:serialize")) {
            return MAPPER.writeValueAsBytes(obj);
        } catch (JsonProcessingException ex) {
            throw new IllegalArgumentException("Serialization error for " + obj.getClass().getCanonicalName(), ex);
        }
    }

    public static <T> byte[] serializeList(List<T> list, Class<T> itemClass) {
        try (TraceProfile ignored = Trace.current().profile("cache:serialize")) {
            CollectionType listType = MAPPER.getTypeFactory().constructCollectionType(List.class, itemClass);
            return MAPPER.writerFor(listType).writeValueAsBytes(list);
        } catch (JsonProcessingException ex) {
            throw new IllegalArgumentException("Serialization error for list of " + itemClass.getCanonicalName(), ex);
        }
    }

    public static <T> T deserialize(String data, Class<T> cls) {
        try (TraceProfile ignored = Trace.current().profile("cache:deserialize")) {
            return MAPPER.readValue(data, cls);
        } catch (IOException ex) {
            throw new IllegalArgumentException(
                    String.format("Deserialization error for class: %s, data: %s", cls.getCanonicalName(), data),
                    ex
            );
        }
    }

    public static <T> List<T> deserializeList(String data, Class<T> itemClass) {
        try (TraceProfile ignored = Trace.current().profile("cache:deserialize")) {
            CollectionType listType = MAPPER.getTypeFactory().constructCollectionType(List.class, itemClass);
            return MAPPER.readValue(data, listType);
        } catch (IOException ex) {
            throw new IllegalArgumentException(
                    String.format("Deserialization error for list of: %s, data: %s",
                            itemClass.getCanonicalName(), data),
                    ex
            );
        }
    }
}
