package ru.yandex.direct.grid.processing.service.cache.storage;

import java.time.Duration;
import java.util.List;
import java.util.Optional;

import org.springframework.stereotype.Component;

/**
 * Надстройка над GridCacheStorage:
 * - добавляет к ключам префик (в частности, для смены схемы)
 * - умеет сериализовывать объекты и списки объектов
 */
@Component
public class GridCacheStorageHelper {

    // Если меняется формат хранения - нужно менять ключ
    private static final String KEY_PREFIX = "gridcache2.";

    private final Duration cacheTtl = Duration.ofMinutes(10);

    private final GridCacheStorage storage;

    public GridCacheStorageHelper(GridCacheStorage storage) {
        this.storage = storage;
    }

    public <T> Optional<T> getObject(String key, Class<T> cls) {
        return storage.get(KEY_PREFIX + key).map(s -> CacheSerializer.deserialize(s, cls));
    }

    public <T> void saveObject(String key, T obj) {
        storage.set(KEY_PREFIX + key, CacheSerializer.serialize(obj), cacheTtl);
    }

    public <R> void saveObjectList(String key, Class<R> itemClass, List<R> list) {
        storage.set(KEY_PREFIX + key, CacheSerializer.serializeList(list, itemClass), cacheTtl);
    }

    public <R> Optional<List<R>> getObjectList(String key, Class<R> itemClass, int begin, int end) {
        Optional<List<R>> objectsOpt =
                storage.get(KEY_PREFIX + key).map(s -> CacheSerializer.deserializeList(s, itemClass));
        if (objectsOpt.isEmpty()) {
            return Optional.empty();
        }
        List<R> objects = objectsOpt.get();
        if (objects.size() < end) {
            throw new IllegalArgumentException(
                    String.format("Incorrect range: %d-%d, list size: %d", begin, end, objects.size()));
        }
        return Optional.of(objects.subList(begin, end));
    }
}
