package ru.yandex.direct.grid.processing.service.cache.storage;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Optional;

import com.google.common.primitives.Ints;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.compress.LZ4Utils;

/**
 * Сохранение кеша грида в Redis-Cluster.
 * <p>
 * Перед сохранение cтроки жмутся gzip-ом
 */
public class LettuceGridCacheStorage implements GridCacheStorage {
    private static final Logger logger = LoggerFactory.getLogger(LettuceGridCacheStorage.class);

    private final LettuceConnectionProvider connectionProvider;

    public LettuceGridCacheStorage(LettuceConnectionProvider connectionProvider) {
        this.connectionProvider = connectionProvider;
    }

    @Override
    public Optional<String> get(String key) {
        try (TraceProfile ignored = Trace.current().profile("redis:get")) {
            return Optional.ofNullable(connectionProvider.getBinaryConnection().sync().get(string2bytes(key)))
                    .map(LZ4Utils::decompressString);
        }
    }

    @Override
    public void set(String key, byte[] data, Duration expire) {
        byte[] compressedData;
        try (TraceProfile ignored = Trace.current().profile("lz4:compress")) {
            compressedData = LZ4Utils.compress(data);
        }
        logger.debug("Set key={}, value length={}, compressed size={}, expire={}",
                key, data.length, compressedData.length, expire);
        try (TraceProfile ignored = Trace.current().profile("redis:setEx:async")) {
            // можем не ждать окончания запроса - это же кэш
            connectionProvider.getBinaryConnection().async().setex(
                    string2bytes(key),
                    Ints.saturatedCast(expire.toMillis() / 1000),
                    compressedData
            );
        }
    }

    private byte[] string2bytes(String key) {
        return key.getBytes(StandardCharsets.UTF_8);
    }
}
