package ru.yandex.direct.grid.processing.service.campaign;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPromoExtension;
import ru.yandex.direct.core.entity.campaign.model.MobileContentInfo;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithPricePackagePermissionUtils;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.grid.model.campaign.GdCampaignAccess;
import ru.yandex.direct.grid.model.campaign.GdCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName;
import ru.yandex.direct.grid.model.campaign.GdiWalletAction;
import ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorFacade;
import ru.yandex.direct.grid.processing.model.campaign.GdWallet;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.Predicates;

import static ru.yandex.direct.core.entity.campaign.CampaignUtils.CPM_TYPES;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepositoryConstants.CAMPAIGN_CLASS_TO_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithPricePackagePermissionUtils.canResetFlightStatusApprove;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignSource.EDA;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignSource.UAC;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.isCpmPriceCampaignReadyToStart;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignValidationService.isDisableWeeklyBudgetForbidden;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignValidationService.isEditWeeklyBudgetForbidden;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignAccessHelper {

    public static final Set<CampaignType> MANAGE_PROMOACTION_CAMPAIGN_TYPES =
            EntryStream.of(CAMPAIGN_CLASS_TO_TYPE)
                    .filterKeys(CampaignWithPromoExtension.class::isAssignableFrom)
                    .values()
                    .toSet();

    private static final Set<GdiCampaignSource> NONSTOPPABLE_SOURCES = Set.of(EDA);

    static class ActionCheckNode {
        private final GdiCampaignAction action;
        private final User operator;
        private final GdiBaseCampaign campaign;

        @Nullable
        private final GdWallet wallet;
        private final boolean isDeletable;
        private final boolean isCpmBannerDisabled;

        ActionCheckNode(GdiCampaignAction action, User operator,
                        GdiBaseCampaign campaign, @Nullable GdWallet wallet, boolean isDeletable,
                        boolean isCpmBannerDisabled) {
            this.action = action;
            this.operator = operator;
            this.campaign = campaign;
            this.wallet = wallet;
            this.isDeletable = isDeletable;
            this.isCpmBannerDisabled = isCpmBannerDisabled;
        }
    }

    private static class WalletActionCheckNode {
        private final GdiWalletAction action;
        private final User operator;
        private final GdiBaseCampaign wallet;
        private final Collection<GdiBaseCampaign> campaignsUnderWallet;

        WalletActionCheckNode(GdiWalletAction action, User operator, GdiBaseCampaign wallet,
                              Collection<GdiBaseCampaign> campaignsUnderWallet) {
            this.action = action;
            this.operator = operator;
            this.wallet = wallet;
            this.campaignsUnderWallet = campaignsUnderWallet;
        }
    }

    private final static Set<GdiCampaignStrategyName> PERFORMANCE_SET_BIDS_STRATEGY_NAMES = ImmutableSet.of(
            GdiCampaignStrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP,
            GdiCampaignStrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER,
            GdiCampaignStrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
            GdiCampaignStrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER);
    private final static Set<CampaignType> SET_BIDS_FORBIDDEN_TYPES = StreamEx.of(CampaignTypeKinds.INTERNAL)
            .append(CampaignType.CPM_PRICE)
            .toSet();


    //Сколько минут не даем архивировать кампанию с момента остановки и последнего показа
    static final Duration DURATION_AFTER_LAST_SHOW_AND_STOP_TIME_FOR_ARCHIVE_CAMPAIGN = Duration.ofMinutes(60);

    static final Duration DURATION_BEFORE_CAN_AUTO_DELETE = Duration.ofDays(14);

    private final Map<GdiWalletAction, Predicate<WalletActionCheckNode>> walletActionPredicates;

    /**
     * Мапа, в котором каждому из расчитываемых здесь действий оператора ставится в соответствие предикат,
     * который должен выполняться для того, чтобы мы добавили действие в набор доступных
     */
    private final Map<GdiCampaignAction, Predicate<ActionCheckNode>> actionPredicates;

    /**
     * Мапа, условия добавления дополнительных действий, которые достпуны только операторам с правом на редактирование
     */
    private final Map<GdiCampaignAction, Predicate<ActionCheckNode>> editActionPredicates;

    /**
     * Мапа, в котором каждому из расчитываемых здесь псевдодействий оператора ставится в соответствие предикат,
     * который должен выполняться для того, чтобы мы добавили действие в набор доступных
     */
    private final Map<GdiCampaignAction, Predicate<ActionCheckNode>> pseudoActionPredicates;

    private final GdStrategyExtractorFacade gdStrategyExtractorFacade;

    @Autowired
    public CampaignAccessHelper(GdStrategyExtractorFacade gdStrategyExtractorFacade) {
        this.gdStrategyExtractorFacade = gdStrategyExtractorFacade;
        this.actionPredicates = getActionPredicates();
        this.editActionPredicates = getEditActionPredicates();
        this.pseudoActionPredicates = getPseudoActionPredicates();
        this.walletActionPredicates = getWalletActionsPredicates();
    }

    /**
     * Можно ли вообще оплачивать кампанию.
     *
     * @param user     — пользователь, по факту подходит любой с ролью «клиент»
     * @param campaign — кампания
     * @return можно или нет
     */
    public boolean isCampaignPayable(User user, GdiBaseCampaign campaign) {
        if (isValidId(campaign.getWalletId())) {
            return false;
        }
        var node = new ActionCheckNode(GdiCampaignAction.PAY, user, campaign, null, false, false);
        return actionPredicates.get(GdiCampaignAction.PAY).test(node);
    }

    public GdCampaignAccess getCampaignAccess(User operator, GdiBaseCampaign campaign,
                                              @Nullable GdWallet wallet, boolean isDeletable,
                                              boolean isCpmBannerDisabled) {
        List<ActionCheckNode> nodes = mapList(campaign.getActions().getActions(),
                a -> new ActionCheckNode(a, operator, campaign, wallet, isDeletable, isCpmBannerDisabled));

        Set<GdiCampaignAction> effectiveActions = nodes.stream()
                .filter(n -> actionPredicates.containsKey(n.action))
                .filter(n -> actionPredicates.get(n.action).test(n))
                .map(n -> n.action)
                .collect(Collectors.toSet());

        if (campaign.getActions().getActions().contains(GdiCampaignAction.EDIT_CAMP)) {
            ActionCheckNode node = new ActionCheckNode(
                    GdiCampaignAction.EDIT_CAMP, operator, campaign, wallet, isDeletable, isCpmBannerDisabled);

            Set<GdiCampaignAction> additionActions = EntryStream.of(editActionPredicates)
                    .mapKeyValue((a, p) -> p.test(node) ? a : null)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());

            effectiveActions.addAll(additionActions);
        }

        Set<GdCampaignAction> effectivePseudoActions = nodes.stream()
                .filter(n -> pseudoActionPredicates.containsKey(n.action))
                .filter(n -> pseudoActionPredicates.get(n.action).test(n))
                .map(n -> n.action)
                .map(CampaignDataConverter::toGdCampaignAction)
                .collect(Collectors.toSet());

        return new GdCampaignAccess()
                .withActions(mapSet(effectiveActions, CampaignDataConverter::toGdCampaignAction))
                .withPseudoActions(effectivePseudoActions)
                .withCanEdit(effectiveActions.contains(GdiCampaignAction.EDIT_CAMP))
                .withNoActions(effectiveActions.isEmpty())
                .withServicedState(CampaignDataConverter.extractCampaignServicedState(effectivePseudoActions));
    }

    static boolean checkAllowPay(User operator, GdiBaseCampaign wallet,
                                 Collection<GdiBaseCampaign> campaignsUnderWallet) {
        WalletActionCheckNode node = new WalletActionCheckNode(GdiWalletAction.PAY, operator,
                wallet, campaignsUnderWallet);
        return allowPay(node);
    }

    private ImmutableMap<GdiCampaignAction, Predicate<ActionCheckNode>> getEditActionPredicates() {
        Predicate<ActionCheckNode> setAutoPricePredicate =
                n -> CampaignTypeKinds.WEB_EDIT_BASE.contains(n.campaign.getType()) &&
                        !gdStrategyExtractorFacade.isAutoBudget(n.campaign) && (
                        n.campaign.getSum().compareTo(BigDecimal.ZERO) > 0 ||
                                hasOneOfRoles(n.operator, RbacRole.SUPER, RbacRole.SUPPORT));
        Predicate<ActionCheckNode> setBidsPredicate =
                n -> !SET_BIDS_FORBIDDEN_TYPES.contains(n.campaign.getType()) &&
                        (!gdStrategyExtractorFacade.isAutoBudget(n.campaign) ||
                                (n.campaign.getType() == CampaignType.PERFORMANCE &&
                                        PERFORMANCE_SET_BIDS_STRATEGY_NAMES.contains(n.campaign.getStrategyName()))) &&
                        !n.campaign.getArchived();
        return ImmutableMap.<GdiCampaignAction, Predicate<ActionCheckNode>>builder()
                .put(GdiCampaignAction.SET_AUTO_PRICE, setAutoPricePredicate)
                .put(GdiCampaignAction.AJAX_SET_AUTO_RESOURCES, n -> !n.campaign.getArchived() &&
                        CampaignTypeKinds.AUTO_RESOURCES.contains(n.campaign.getType()))
                .put(GdiCampaignAction.EDIT_METRICA_COUNTERS,
                        n -> !n.campaign.getArchived() && n.campaign.getType() != CampaignType.MOBILE_CONTENT)
                .put(GdiCampaignAction.PAY_BY_CASH, n -> CampaignAccessHelper.allowPay(n)
                        && n.campaign.getStatusPostModerate() == CampaignStatusPostmoderate.ACCEPTED)
                .put(GdiCampaignAction.SET_BIDS, setBidsPredicate)
                .put(GdiCampaignAction.MODERATE_CAMP, CampaignAccessHelper::allowModerate)
                .build();
    }

    private ImmutableMap<GdiCampaignAction, Predicate<ActionCheckNode>> getActionPredicates() {
        return ImmutableMap.<GdiCampaignAction, Predicate<ActionCheckNode>>builder()
                .put(GdiCampaignAction.COPY_CAMP_CLIENT,
                        n -> !n.campaign.getArchived() && n.campaign.getHasBanners()
                                && CampaignTypeKinds.COPYABLE_BY_CLIENT.contains(n.campaign.getType())
                                && isCampaignEditAllowed(n.campaign, n.isCpmBannerDisabled))
                .put(GdiCampaignAction.RESUME_CAMP,
                        n -> !n.campaign.getArchived()
                                && !n.campaign.getShowing()
                                &&
                                (!isClient(n.operator)
                                        || CampaignTypeKinds.WEB_EDIT_BASE.contains(n.campaign.getType()))
                                && canResumeCampaign(n.operator, n.campaign)
                                && !NONSTOPPABLE_SOURCES.contains(n.campaign.getSource())
                                && isCampaignEditAllowed(n.campaign, n.isCpmBannerDisabled))
                .put(GdiCampaignAction.STOP_CAMP,
                        n -> !n.campaign.getArchived()
                                && n.campaign.getShowing()
                                && (!isDraft(n) || draftUacOnModeration(n))
                                &&
                                (!isClient(n.operator)
                                        || CampaignTypeKinds.WEB_EDIT_BASE.contains(n.campaign.getType()))
                                && (canStopCampaign(n.operator, n.campaign)
                                && !NONSTOPPABLE_SOURCES.contains(n.campaign.getSource()))
                )
                .put(GdiCampaignAction.SHOW_CAMP_STAT, n -> isValidId(n.campaign.getOrderId()))
                .put(GdiCampaignAction.EDIT_CAMP, n -> !n.campaign.getArchived()
                        && isCampaignEditAllowed(n.campaign, n.isCpmBannerDisabled))
                .put(GdiCampaignAction.SHOW_CAMP_SETTINGS, Predicates.ignore())
                .put(GdiCampaignAction.PAY, CampaignAccessHelper::allowPay)
                .put(GdiCampaignAction.ARCHIVE_CAMP, CampaignAccessHelper::allowArchive)
                .put(GdiCampaignAction.UNARCHIVE_CAMP, n -> n.campaign.getArchived()
                        && isCampaignEditAllowed(n.campaign, n.isCpmBannerDisabled))
                .put(GdiCampaignAction.LOOKUP, Predicates.ignore())
                .put(GdiCampaignAction.DELETE_CAMP, n -> n.isDeletable)
                .put(GdiCampaignAction.REMODERATE_CAMP, n -> !n.campaign.getArchived()
                        && n.campaign.getStatusModerate() != CampaignStatusModerate.NEW
                        && !isCpmPriceCampaignReadyToStart(n.campaign))
                .put(GdiCampaignAction.SHOW_BS_LINK, CampaignAccessHelper::allowShowBsLink)
                .put(GdiCampaignAction.EXPORT_IN_EXCEL, n -> isCampaignCanBeExported(n.campaign))
                .put(GdiCampaignAction.CAN_BE_AUTODELETED, CampaignAccessHelper::canBeAutodeleted)
                .put(GdiCampaignAction.RESET_FLIGHT_STATUS_APPROVE, n -> n.campaign.getType() == CampaignType.CPM_PRICE
                        && n.campaign.getFlightStatusApprove() != PriceFlightStatusApprove.NEW
                        && canResetFlightStatusApprove(n.operator))
                .put(GdiCampaignAction.VIEW_OFFLINE_REPORTS, n -> isValidId(n.campaign.getOrderId())
                        && CampaignTypeKinds.PDF_REPORT.contains(n.campaign.getType()))
                .put(GdiCampaignAction.MANAGE_VCARDS, n -> !n.campaign.getArchived()
                        && n.campaign.getType() == CampaignType.TEXT)
                .put(GdiCampaignAction.BAN_PAY, n -> !n.campaign.getArchived() && !n.campaign.getNoPay()
                        && CampaignTypeKinds.PAYABLE.contains(n.campaign.getType()))
                .put(GdiCampaignAction.UNBAN_PAY, n -> !n.campaign.getArchived() && n.campaign.getNoPay()
                        && CampaignTypeKinds.PAYABLE.contains(n.campaign.getType()))
                .put(GdiCampaignAction.SERVICING_STOPPED, Predicates.ignore())
                .put(GdiCampaignAction.REDIRECT_VIEW, Predicates.ignore())
                .put(GdiCampaignAction.REDIRECT_EDIT, Predicates.ignore())
                .put(GdiCampaignAction.MANAGE_PROMO_EXTENSION, n -> !n.campaign.getArchived() &&
                        MANAGE_PROMOACTION_CAMPAIGN_TYPES.contains(n.campaign.getType()))
                .put(GdiCampaignAction.EDIT_WEEKLY_BUDGET, n -> isEditWeeklyBudgetAllowed(n.campaign,
                        gdStrategyExtractorFacade.isAutoBudget(n.campaign)))
                .put(GdiCampaignAction.DISABLE_WEEKLY_BUDGET, n -> isDisableWeeklyBudgetAllowed(n.campaign))
                .build();
    }

    private static boolean draftUacOnModeration(ActionCheckNode n) {
        if (n.campaign.getSource() != UAC) {
            return false;
        }

        // возможно нужно еще проверить, что кампанию действительно переключили в started
        return true;
    }

    private static boolean isDraft(ActionCheckNode n) {
        return n.campaign.getStatusModerate() == CampaignStatusModerate.NEW;
    }

    private boolean canResumeCampaign(User operator, GdiBaseCampaign campaign) {
        if (operator.getIsReadonlyRep()) {
            return false;
        }

        if (campaign.getType() == CampaignType.CPM_PRICE) {
            return CampaignWithPricePackagePermissionUtils.canResumeCampaign(
                    operator.getRole(), campaign.getFinishDate());
        }
        return true;
    }

    private static boolean canStopCampaign(User operator, GdiBaseCampaign campaign) {
        if (campaign.getType() == CampaignType.CPM_PRICE) {
            return CampaignWithPricePackagePermissionUtils.canStopCampaign(
                    operator.getRole(), campaign.getFinishDate());
        }
        return true;
    }

    private static ImmutableMap<GdiCampaignAction, Predicate<ActionCheckNode>> getPseudoActionPredicates() {
        return ImmutableMap.<GdiCampaignAction, Predicate<ActionCheckNode>>builder()
                .put(GdiCampaignAction.OFFER_SERVICING, CampaignAccessHelper::canOfferServicing)
                .put(GdiCampaignAction.ACCEPT_SERVICING, Predicates.ignore())
                .put(GdiCampaignAction.ALLOW_TRANSFER_MONEY_SUBCLIENT, Predicates.ignore())
                .put(GdiCampaignAction.SERVICED, Predicates.ignore())
                .put(GdiCampaignAction.OTHER_MANAGER_SERVICED, Predicates.ignore())
                .put(GdiCampaignAction.AGENCY_SERVICED, Predicates.ignore())
                .build();
    }

    private static ImmutableMap<GdiWalletAction, Predicate<WalletActionCheckNode>> getWalletActionsPredicates() {
        return ImmutableMap.<GdiWalletAction, Predicate<WalletActionCheckNode>>builder()
                .put(GdiWalletAction.EDIT, Predicates.ignore())
                .put(GdiWalletAction.PAY, CampaignAccessHelper::allowPay)
                .build();
    }

    /**
     * Возвращает список доступных действий для конкретного кошелька с учетом состояния кампаний клиента
     * В старом директе часть условий содержится во фронте: data3/desktop.blocks/b-wallet-rest/b-wallet-rest.bemtree.js
     *
     * @param operator             оператор
     * @param wallet               общий счет клиента
     * @param campaignsUnderWallet все кампании относящиеся к данному кошельку.
     */
    Set<GdiWalletAction> getWalletActions(User operator, GdiBaseCampaign wallet,
                                          Collection<GdiBaseCampaign> campaignsUnderWallet) {
        List<WalletActionCheckNode> nodes = mapList(wallet.getWalletActions().getActions(),
                action -> new WalletActionCheckNode(action, operator, wallet, campaignsUnderWallet));

        return StreamEx.of(nodes)
                .filter(n -> walletActionPredicates.containsKey(n.action))
                .filter(n -> walletActionPredicates.get(n.action).test(n))
                .map(n -> n.action)
                .toSet();
    }

    private static List<GdiBaseCampaign> getCampsNotForbiddenToPay(Collection<GdiBaseCampaign> campaigns) {
        return filterList(campaigns, campaign -> !campaign.getArchived() && !campaign.getNoPay());
    }

    private static boolean hasModeratedCamps(List<GdiBaseCampaign> campaigns) {
        return campaigns.stream().anyMatch(c -> c.getStatusModerate() == CampaignStatusModerate.YES);
    }

    private static boolean hasServicingOrAgencyCamps(List<GdiBaseCampaign> campaigns) {
        return campaigns.stream()
                .anyMatch(c -> isValidId(c.getAgencyId()) || isValidId(c.getManagerUserId()));
    }

    private static boolean canOfferServicing(ActionCheckNode node) {
        return !node.campaign.getArchived()
                && !node.campaign.getActions().getHasManager()
                && hasOneOfRoles(node.operator, RbacRole.MANAGER, RbacRole.SUPER);
    }

    private static boolean allowPay(WalletActionCheckNode node) {
        List<GdiBaseCampaign> campaignsNotForbiddenToPay = getCampsNotForbiddenToPay(node.campaignsUnderWallet);
        return node.wallet.getWalletCanPayBeforeModeration() || (
                !campaignsNotForbiddenToPay.isEmpty() && (
                        hasModeratedCamps(campaignsNotForbiddenToPay) ||
                                hasServicingOrAgencyCamps(campaignsNotForbiddenToPay) ||
                                //оператор может класть деньги на ОС, при отсутствии промодерированных кампаний
                                hasOneOfRoles(node.operator, RbacRole.SUPER, RbacRole.AGENCY, RbacRole.MANAGER)));
    }

    private static boolean allowPay(ActionCheckNode node) {
        return !node.campaign.getNoPay()
                && !isValidId(node.campaign.getWalletId())
                && !node.campaign.getArchived()
                && !CampaignTypeKinds.INTERNAL.contains(node.campaign.getType())

                && (node.campaign.getStatusModerate() == CampaignStatusModerate.YES
                || hasOneOfRoles(node.operator, RbacRole.MANAGER, RbacRole.SUPER)
                || CampaignTypeKinds.WEB_EDIT_BASE.contains(node.campaign.getType()))

                && (node.campaign.getStatusModerate() == CampaignStatusModerate.YES
                || isValidId(node.campaign.getManagerUserId())
                || isValidId(node.campaign.getAgencyUserId()));
    }

    /**
     * Проверяет, может ли кампания быть экспортирована в Excel.
     * Имеет смысл подумать о переносе этой бизнес-логики в общее место для переиспользования в других приложениях и/или
     * в коде валидации в сервисе экспорта кампаний.
     */
    private static boolean isCampaignCanBeExported(GdiBaseCampaign campaign) {
        return !campaign.getArchived() &&
                (campaign.getType() == CampaignType.TEXT || campaign.getType() == CampaignType.MOBILE_CONTENT);
    }

    static boolean allowArchive(ActionCheckNode node) {
        return !node.campaign.getArchived()
                && (!node.campaign.getShowing() || isDraft(node))
                && checkLastShowAndStopTimeForArchiveAction(node.campaign)
                && NumberUtils.isZero(node.campaign.getSumRest());
    }

    static boolean canBeAutodeleted(ActionCheckNode node) {
        LocalDateTime now = LocalDateTime.now();
        var campaign = node.campaign;
        return CampaignTypeKinds.UNDER_WALLET.contains(campaign.getType())
                && !isValidId(campaign.getOrderId())
                && (node.wallet == null || NumberUtils.isZero(node.wallet.getSum()))
                && NumberUtils.isZero(campaign.getSum())
                && NumberUtils.isZero(campaign.getSumToPay())
                && NumberUtils.isZero(campaign.getSumSpent())
                && NumberUtils.isZero(campaign.getSumLast())
                && (campaign.getShows() == null || campaign.getShows() == 0)
                && campaign.getCreateTime() != null
                && campaign.getCreateTime().plus(DURATION_BEFORE_CAN_AUTO_DELETE).isBefore(now);
    }

    /**
     * Для архивации кампания должна быть остановлена и с момента остановки и последнего показа должно пройти не менее:
     * {@link #DURATION_AFTER_LAST_SHOW_AND_STOP_TIME_FOR_ARCHIVE_CAMPAIGN}
     */
    private static boolean checkLastShowAndStopTimeForArchiveAction(GdiBaseCampaign campaign) {
        LocalDateTime now = LocalDateTime.now();

        return (campaign.getLastShowTime() == null
                || campaign.getLastShowTime().plus(DURATION_AFTER_LAST_SHOW_AND_STOP_TIME_FOR_ARCHIVE_CAMPAIGN).isBefore(now))
                && (!isValidId(campaign.getOrderId()) || campaign.getStopTime() == null
                || campaign.getStopTime().plus(DURATION_AFTER_LAST_SHOW_AND_STOP_TIME_FOR_ARCHIVE_CAMPAIGN).isBefore(now));
    }

    private static boolean allowModerate(ActionCheckNode node) {
        return !hasOneOfRoles(node.operator, RbacRole.MEDIA)
                && !node.campaign.getArchived()
                && node.campaign.getStatusModerate() == CampaignStatusModerate.NEW
                && node.campaign.getHasBanners();
    }

    static boolean allowShowBsLink(ActionCheckNode node) {
        return isValidId(node.campaign.getOrderId()) && node.operator.getRole().anyOf(RbacRole.SUPER,
                RbacRole.SUPERREADER, RbacRole.SUPPORT, RbacRole.LIMITED_SUPPORT, RbacRole.PLACER,
                RbacRole.INTERNAL_AD_ADMIN, RbacRole.INTERNAL_AD_MANAGER, RbacRole.INTERNAL_AD_SUPERREADER);
    }

    /**
     * Является ли кампания активной РМП кампанией, рекламирующей Ios приложение
     */
    public static boolean isActiveIosMobileCampaign(GdiBaseCampaign campaign) {
        return !campaign.getArchived()
                && (campaign.getType() == CampaignType.MOBILE_CONTENT)
                && (ifNotNull(campaign.getMobileContentInfo(), MobileContentInfo::getOsType) == OsType.IOS);
    }

    private static boolean isEditWeeklyBudgetAllowed(GdiBaseCampaign campaign, boolean isAutobudget) {
        return !campaign.getArchived()
                && !isEditWeeklyBudgetForbidden(isAutobudget, campaign.getType(),
                        Boolean.TRUE.equals(campaign.getIsGoods()));
    }

    private static boolean isDisableWeeklyBudgetAllowed(GdiBaseCampaign campaign){
        var source = CampaignSource.fromSource(GdiCampaignSource.toSource(campaign.getSource()));
        var strategyName = StrategyName.fromSource(GdiCampaignStrategyName.toSource(campaign.getStrategyName()));

        return !isDisableWeeklyBudgetForbidden(source, strategyName);
    }

    private static boolean isCampaignEditAllowed(GdiBaseCampaign campaign, boolean isCpmBannerDisabled) {
        return !CPM_TYPES.contains(campaign.getType()) || !isCpmBannerDisabled;
    }
}
