package ru.yandex.direct.grid.processing.service.campaign;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bs.export.queue.service.BsExportQueueService;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.core.AllowedTypesCampaignAccessibilityChecker;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaignAccess;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdiWalletAction;
import ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter;
import ru.yandex.direct.grid.processing.model.campaign.GdWallet;
import ru.yandex.direct.grid.processing.util.GdEntitySet;
import ru.yandex.direct.utils.NumberUtils;

import static java.util.function.Predicate.not;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_COLLECTIONS_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_EDA_INTERFACE;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_SERVICES_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_VIDEO_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.CPM_YNDX_FRONTPAGE_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_BANNER_CAMPAIGNS_IN_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_PRICE_CAMPAIGNS_IN_GRID;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

/**
 * Сервис, предназначенный для расчета прав доступа оператора
 */
@Service
@ParametersAreNonnullByDefault
public class CampaignAccessService {
    private static final Logger logger = LoggerFactory.getLogger(CampaignAccessService.class);

    public static final GdEntitySet<GdCampaignType> GRID_VISIBLE_TYPES = GdEntitySet.<GdCampaignType>newBuilder()
            .add(GdCampaignType.TEXT)
            .add(GdCampaignType.PERFORMANCE)
            .add(GdCampaignType.MOBILE_CONTENT)
            .add(GdCampaignType.DYNAMIC)
            .add(GdCampaignType.MCBANNER)
            .add(GdCampaignType.INTERNAL_AUTOBUDGET)
            .add(GdCampaignType.INTERNAL_DISTRIB)
            .add(GdCampaignType.INTERNAL_FREE)
            .add(GdCampaignType.CPM_DEALS)
            .addIf(GdCampaignType.CPM_BANNER, SHOW_CPM_BANNER_CAMPAIGNS_IN_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_VIDEO_ON_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_COLLECTIONS_ON_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_SERVICES_ON_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_EDA_INTERFACE)
            .addIf(GdCampaignType.CPM_YNDX_FRONTPAGE, CPM_YNDX_FRONTPAGE_ON_GRID)
            .addIf(GdCampaignType.CPM_PRICE, SHOW_CPM_PRICE_CAMPAIGNS_IN_GRID)
            .build();

    public static final GdEntitySet<GdCampaignType> GRID_SUPPORTED_TYPES = GdEntitySet.<GdCampaignType>newBuilder()
            .add(GdCampaignType.TEXT)
            .add(GdCampaignType.PERFORMANCE)
            .add(GdCampaignType.MOBILE_CONTENT)
            .add(GdCampaignType.DYNAMIC)
            .add(GdCampaignType.MCBANNER)
            .add(GdCampaignType.INTERNAL_AUTOBUDGET)
            .add(GdCampaignType.INTERNAL_DISTRIB)
            .add(GdCampaignType.INTERNAL_FREE)
            .add(GdCampaignType.CPM_DEALS)
            .addIf(GdCampaignType.CPM_BANNER, SHOW_CPM_BANNER_CAMPAIGNS_IN_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_VIDEO_ON_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_COLLECTIONS_ON_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_SERVICES_ON_GRID)
            .addIf(GdCampaignType.CONTENT_PROMOTION, CONTENT_PROMOTION_EDA_INTERFACE)
            .addIf(GdCampaignType.CPM_YNDX_FRONTPAGE, CPM_YNDX_FRONTPAGE_ON_GRID)
            .addIf(GdCampaignType.CPM_PRICE, SHOW_CPM_PRICE_CAMPAIGNS_IN_GRID)
            .build();

    public static final GdEntitySet<CampaignSource> GRID_VISIBLE_SOURCES = GdEntitySet.<CampaignSource>newBuilder()
            .add(CampaignSource.DIRECT)
            .add(CampaignSource.API)
            .add(CampaignSource.UAC)
            .add(CampaignSource.DC)
            .add(CampaignSource.XLS)
            .add(CampaignSource.GEO)
            .addIf(CampaignSource.EDA, CONTENT_PROMOTION_EDA_INTERFACE)
            .addIf(CampaignSource.USLUGI, CONTENT_PROMOTION_SERVICES_ON_GRID)
            .add(CampaignSource.ZEN)
            .add(CampaignSource.WIDGET)
            .build();

    public static final GdEntitySet<CampaignSource> GRID_EDITABLE_SOURCES = GdEntitySet.<CampaignSource>newBuilder()
            .add(CampaignSource.DIRECT)
            .add(CampaignSource.API)
            .add(CampaignSource.UAC)
            .add(CampaignSource.DC)
            .add(CampaignSource.XLS)
            .add(CampaignSource.GEO)
            .add(CampaignSource.WIDGET)
            .build();

    private final BsExportQueueService bsExportQueueService;
    private final CampaignAccessHelper campaignAccessHelper;
    private final FeatureService featureService;
    private final PricePackageService pricePackageService;

    @Autowired
    public CampaignAccessService(BsExportQueueService bsExportQueueService, CampaignAccessHelper campaignAccessHelper,
                                 FeatureService featureService, PricePackageService pricePackageService) {
        this.bsExportQueueService = bsExportQueueService;
        this.campaignAccessHelper = campaignAccessHelper;
        this.featureService = featureService;
        this.pricePackageService = pricePackageService;
    }

    /**
     * Получить мапу, в котором идентификатору кампании ставится в соответствие объект с правами оператора на кампанию.
     * Аналог перлового Common::camps_add_rbac_actions. Здесь права рассчитываются, основываясь на предварительных
     * данных,
     * полученных ранее, и реальном текущем состоянии кампании. Например, если кампания уже архивна, оператору будет
     * недоступно действие ее архивации, даже если у него есть на это право
     *
     * @param operator        описание оператора
     * @param clientId        клиент
     * @param campaigns       список кампаний (без кошельков), для которых производится расчет
     * @param campaignWallets мапа id кошелька -> его описание
     */
    public <T extends GdiBaseCampaign> Map<Long, GdCampaignAccess> getCampaignsAccess(
            User operator,
            ClientId clientId,
            Collection<T> campaigns,
            Map<Long, GdWallet> campaignWallets
    ) {
        List<GdiBaseCampaign> campaignsWithDelete = filterList(campaigns,
                c -> c.getActions().getActions().contains(GdiCampaignAction.DELETE_CAMP));
        Set<Long> deletableCampaignIds = getDeletableCampaignIds(clientId, campaignsWithDelete);
        boolean isCpmBannerDisabled = featureService.isEnabledForClientId(
                clientId, FeatureName.IS_CPM_BANNER_CAMPAIGN_DISABLED);

        return campaigns.stream()
                .collect(Collectors.toMap(GdiBaseCampaign::getId,
                        c -> campaignAccessHelper
                                .getCampaignAccess(operator, c, campaignWallets.get(c.getWalletId()),
                                        deletableCampaignIds.contains(c.getId()), isCpmBannerDisabled)));
    }

    private Set<Long> getDeletableCampaignIds(ClientId clientId, Collection<GdiBaseCampaign> campaigns) {
        List<Long> campaignIds = mapList(campaigns, GdiBaseCampaign::getId);
        Map<Long, PricePackage> pricePackageByCampaignId = pricePackageService.getPricePackageByCampaignIds(clientId,
                campaignIds);
        List<Long> preDeletableCampaignIds = filterAndMapList(campaigns,
                c -> isDeletableCampaign(c, pricePackageByCampaignId.get(c.getId())),
                GdiBaseCampaign::getId);
        Set<Long> campsInBsExport = bsExportQueueService.getCampaignIdsGoingToBeSent(preDeletableCampaignIds);

        return filterToSet(preDeletableCampaignIds, not(campsInBsExport::contains));
    }

    /**
     * Менять сихнронно с методом
     * {@link ru.yandex.direct.core.entity.campaign.service.validation.DeleteCampaignValidationService#isDeletableCampaign}
     * и методом perl/protected/Direct/Validation/Campaigns.pm::validate_delete_campaigns.
     * Нахождение кампаний в очереди на отправку в БК проверяется отдельно - campsInBsExport
     */
    private static boolean isDeletableCampaign(GdiBaseCampaign campaign, @Nullable PricePackage pricePackage) {
        return NumberUtils.isZero(campaign.getSum())
                && NumberUtils.isZero(campaign.getSumToPay())
                && NumberUtils.isZero(campaign.getSumLast())
                && !isValidId(campaign.getOrderId())
                && !(campaign.getCurrencyConverted() && campaign.getCurrencyCode() == CurrencyCode.YND_FIXED)
                && (!CampaignServiceUtils.isApprovedCpmPriceCampaign(campaign) ||
                    ifNotNullOrDefault(pricePackage, p -> nvl(p.getCampaignAutoApprove(), false), false));
    }

    public AllowedTypesCampaignAccessibilityChecker getCampaignAccessibilityChecker(ClientId clientId) {
        Set<String> availableFeatures = featureService.getEnabledForClientId(clientId);

        return new AllowedTypesCampaignAccessibilityChecker(
                mapSet(GRID_SUPPORTED_TYPES.getAll(availableFeatures), CampaignDataConverter::toCampaignType),
                mapSet(GRID_VISIBLE_TYPES.getAll(availableFeatures), CampaignDataConverter::toCampaignType),
                GRID_EDITABLE_SOURCES.getAll(availableFeatures),
                GRID_VISIBLE_SOURCES.getAll(availableFeatures));
    }

    Set<GdCampaignType> getVisibleTypes(ClientId clientId) {
        Set<String> availableFeatures = featureService.getEnabledForClientId(clientId);
        return GRID_VISIBLE_TYPES.getAll(availableFeatures);
    }

    /**
     * Возвращает список доступных действий для конкретного кошелька с учетом состояния кампаний клиента
     * В старом директе часть условий содержится во фронте: data3/desktop.blocks/b-wallet-rest/b-wallet-rest.bemtree.js
     *
     * @param operator             оператор
     * @param wallet               общий счет клиента
     * @param campaignsUnderWallet все кампании относящиеся к данному кошельку.
     */
    public Set<GdiWalletAction> getWalletActions(User operator, GdiBaseCampaign wallet,
                                                 Collection<GdiBaseCampaign> campaignsUnderWallet) {
        if (wallet.getWalletActions() == null || wallet.getWalletActions().getActions() == null) {
            logger.error("Wallet actions can not be null. Returning wrong wallet actions. walletId = {}",
                    wallet.getId());
            return Collections.emptySet();
        }
        return campaignAccessHelper.getWalletActions(operator, wallet, campaignsUnderWallet);
    }

}
