package ru.yandex.direct.grid.processing.service.campaign;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.YandexOffice;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ContactPhone;
import ru.yandex.direct.core.entity.client.service.ClientOfficeService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.grid.model.campaign.GdCampaignAgencyInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignManagerInfo;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter;
import ru.yandex.direct.grid.processing.service.operator.OperatorDataService;
import ru.yandex.direct.utils.CommonUtils;

import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toCampaignGdAgencyInfo;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;

/**
 * получение информации об агентстве и менеджере кампании
 */
@Service
@ParametersAreNonnullByDefault
public class CampaignAgencyManagerInfoHelper {
    private final ClientOfficeService clientOfficeService;
    private final UserService userService;
    private final ClientService clientService;
    private final OperatorDataService operatorDataService;

    @Autowired
    public CampaignAgencyManagerInfoHelper(ClientOfficeService clientOfficeService,
                                           UserService userService,
                                           ClientService clientService,
                                           OperatorDataService operatorDataService) {
        this.clientOfficeService = clientOfficeService;
        this.userService = userService;
        this.clientService = clientService;
        this.operatorDataService = operatorDataService;
    }

    public Map<Long, GdCampaignManagerInfo> getCampaignManagerInfoByUid(Collection<Long> managerUserIds) {
        if (CollectionUtils.isEmpty(managerUserIds)) {
            return Map.of();
        }
        var managersByUid = listToMap(userService.massGetUser(managerUserIds), User::getId);
        var officeIds = mapAndFilterToSet(managersByUid.values(), User::getManagerOfficeId, CommonUtils::isValidId);
        var officesById = clientOfficeService.getOfficesByIds(officeIds);

        return EntryStream.of(managersByUid).mapValues(manager -> {
            YandexOffice office = ifNotNull(manager.getManagerOfficeId(), officesById::get);
            List<ContactPhone> contactPhones = clientOfficeService.getManagerContactPhones(manager, office);
            return CommonCampaignConverter.toGdManagerInfo(manager, contactPhones);
        }).toMap();
    }

    public Map<Long, GdCampaignAgencyInfo> getCampaignAgencyInfoByUid(Set<Long> agencyUserIds) {
        Map<Long, Client> agencyClientsByUid = clientService.massGetClientsByUids(agencyUserIds);
        Map<Long, Client> agencyClients = StreamEx.of(agencyClientsByUid.values())
                .distinct(Client::getId)
                .toMap(Client::getId, Function.identity());

        Set<Long> agencyChiefUserIds = listToSet(agencyClients.values(), Client::getChiefUid);

        Set<Long> userIdsToCollect = new HashSet<>(agencyUserIds);
        userIdsToCollect.addAll(agencyChiefUserIds);

        Map<Long, GdUserInfo> usersByIds = operatorDataService.getUsersByIds(userIdsToCollect);

        // Проверяем, стоит ли у представителей опция "не показывать контактные данные"
        Map<Long, Boolean> showAgencyContactsByUserIds =
                userService.getUsersAgencyShowAgencyContacts(userIdsToCollect);

        return StreamEx.of(agencyUserIds)
                .mapToEntry(uid -> getCampaignGdAgencyInfo(uid, usersByIds, agencyClients, showAgencyContactsByUserIds))
                .nonNullValues()
                .toMap();
    }

    static GdCampaignAgencyInfo getCampaignGdAgencyInfo(Long agencyUserId,
                                                        Map<Long, GdUserInfo> usersByIds,
                                                        Map<Long, Client> relatedClientsByIds,
                                                        Map<Long, Boolean> showAgencyContactsByUserIds) {
        GdUserInfo representativeInfo = usersByIds.get(agencyUserId);
        if (representativeInfo != null) {
            Client agencyClient = relatedClientsByIds.get(representativeInfo.getClientId());
            GdUserInfo chiefInfo = usersByIds.get(agencyClient.getChiefUid());

            boolean showRepresentativeContacts = showAgencyContactsByUserIds
                    .getOrDefault(representativeInfo.getUserId(), false);
            boolean showChiefContacts = showAgencyContactsByUserIds
                    .getOrDefault(chiefInfo.getUserId(), false);

            /*  Если у представителя showContacts == true, показываем его информацию
                Иначе, если у главного представителя showContacts == true, показываем его информацию
                Если у обоих showContacts == false, показываем имя главного представителя, без другой информации */
            if (showRepresentativeContacts) {
                return toCampaignGdAgencyInfo(agencyClient, chiefInfo, representativeInfo, true);
            } else if (showChiefContacts) {
                return toCampaignGdAgencyInfo(agencyClient, chiefInfo, chiefInfo, true);
            } else {
                return toCampaignGdAgencyInfo(agencyClient, chiefInfo, chiefInfo, false);
            }
        }
        return null;
    }
}
