package ru.yandex.direct.grid.processing.service.campaign;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.container.UpdateCampMetrikaCountersRequest;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWarnPlaceInterval;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBannerHrefParams;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignsPromotions;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomDayBudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPromoExtension;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPromotion;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdGoalStats;
import ru.yandex.direct.grid.model.GdTotalCampaigns;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaignBrandSafety;
import ru.yandex.direct.grid.model.campaign.GdCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignCheckPositionInterval;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignEmailEvent;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignEmailSettings;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignNotification;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignSmsEvent;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignSmsEventInfo;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignSmsSettings;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudget;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudgetPeriod;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.model.utils.GridTimeUtils;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsContainer;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsContext;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsPromotion;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignMetrikaCounters;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignMetrikaCountersPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBannerHrefParams;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBrandSurvey;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsDayBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsOrganization;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsPromotions;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsStartDate;
import ru.yandex.direct.grid.processing.model.promoextension.GdUpdateCampaignsPromoExtension;
import ru.yandex.direct.grid.processing.service.campaign.container.CampaignsCacheFilterData;
import ru.yandex.direct.grid.processing.service.campaign.container.CampaignsCacheRecordInfo;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.libs.timetarget.WeekdayType;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.NumberUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.CampaignNotificationUtils.getAvailableEmailEvents;
import static ru.yandex.direct.core.entity.campaign.CampaignNotificationUtils.getAvailableSmsFlags;
import static ru.yandex.direct.core.entity.crypta.utils.CryptaSegmentBrandSafetyUtils.extractBrandSafetyAdditionalCategories;
import static ru.yandex.direct.core.entity.crypta.utils.CryptaSegmentBrandSafetyUtils.extractBrandSafetyIsEnabled;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignFeatureCalculator.FEATURE_CALCULATOR;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.setScaleForCore;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toCampaignDayBudgetShowMode;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalGoalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.recalcTotalStatsForUnitedGrid;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.ListUtils.integerToLongList;

@ParametersAreNonnullByDefault
public class CampaignDataConverter {

    static CampaignsCacheRecordInfo toCampaignsCacheRecordInfo(long clientId, GdCampaignsContainer input) {
        return new CampaignsCacheRecordInfo(clientId, input.getCacheKey(),
                new CampaignsCacheFilterData()
                        .withFilter(input.getFilter())
                        .withOrderBy(input.getOrderBy())
                        .withStatRequirements(input.getStatRequirements()));
    }

    static GdCampaignsContext toGdCampaignsContext(List<GdCampaign> rowsetFull) {
        return toGdCampaignsContext(rowsetFull, false);
    }

    static GdCampaignsContext toGdCampaignsContext(List<GdCampaign> rowsetFull, boolean cpcAndCpmOnOneGridEnabled) {

        List<GdEntityStats> stats = StreamEx.of(rowsetFull)
                .remove(campaign -> Boolean.TRUE.equals(campaign.getStatsIsRestrictedByUnavailableGoals()))
                .map(GdCampaign::getStats)
                .toList();
        var totalStats = calcTotalStats(stats);
        if (cpcAndCpmOnOneGridEnabled) {
            Map<GdCampaignType, List<GdEntityStats>> campaignTypeToStats = StreamEx.of(rowsetFull)
                    .mapToEntry(GdCampaign::getType, GdCampaign::getStats)
                    .grouping();
            recalcTotalStatsForUnitedGrid(totalStats, campaignTypeToStats);
        }

        List<List<GdGoalStats>> goalStats = StreamEx.of(rowsetFull)
                .remove(campaign -> Boolean.TRUE.equals(campaign.getStatsIsRestrictedByUnavailableGoals()))
                .map(GdCampaign::getGoalStats)
                .toList();
        return new GdCampaignsContext()
                .withTotalCount(rowsetFull.size())
                .withFeatures(FEATURE_CALCULATOR.apply(rowsetFull))
                .withCampaignIds(rowsetFull.stream()
                        .filter(c -> !AvailableCampaignSources.INSTANCE.isUC(GdCampaignSource.toSource(c.getSource())) && c.getRedirectInfo() == null)
                        .map(GdCampaign::getId)
                        .collect(Collectors.toSet()))
                .withTotalStats(totalStats)
                .withTotalGoalStats(calcTotalGoalStats(totalStats, goalStats))
                .withTotalCampaigns(getTotalCampaigns(rowsetFull))
                .withCampaignTypes(calculateCampaignTypes(rowsetFull));
    }

    private static Map<GdCampaignType, Long> calculateCampaignTypes(List<GdCampaign> rowsetFull) {
        return StreamEx.of(rowsetFull)
                .groupingBy(GdCampaign::getType, Collectors.counting());
    }

    static GdCampaignSmsEvent toGdCampaignSmsEvent(SmsFlag smsFlag) {
        return GdCampaignSmsEvent.fromTypedValue(smsFlag.name());
    }

    static GdTotalCampaigns getTotalCampaigns(List<GdCampaign> rowset) {
        var totalBudgets = StreamEx.of(rowset)
                .map(gdCampaign -> gdCampaign.getStrategy().getBudget())
                .nonNull()
                .mapToEntry(GdCampaignBudget::getPeriod, GdCampaignBudget::getSum)
                .filterValues(NumberUtils::greaterThanZero)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys(BigDecimal::add)
                .map(entry -> new GdCampaignBudget()
                        .withPeriod(entry.getKey())
                        .withSum(entry.getValue())
                        .withShowMode(GdCampaignBudgetShowMode.DEFAULT) //он non-null, но смотреть на него не надо
                ).toList();

        var totalWeeklyBudget = StreamEx.of(rowset)
                .filter(Predicate.not(CampaignDataConverter::isCpmCampaign))
                .filter(CampaignDataConverter::hasDayOrWeeklyBudget)
                .map(c -> GdCampaignBudgetPeriod.WEEK == c.getStrategy().getBudget().getPeriod() ?
                                c.getStrategy().getBudget().getSum() : convertDayBudgetToWeekly(c))
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        var totalSumRest = StreamEx.of(rowset)
                .map(GdCampaign::getSumRest)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        return new GdTotalCampaigns()
                .withTotalSumRest(totalSumRest)
                .withTotalBudgets(nvl(totalBudgets, emptyList()))
                .withTotalWeeklyBudget(totalWeeklyBudget);
    }

    private static BigDecimal convertDayBudgetToWeekly(GdCampaign gdCampaign){
        var dayBudget = gdCampaign.getStrategy().getBudget().getSum();
        return dayBudget.multiply(new BigDecimal(getActiveDaysCount(gdCampaign.getTimeTarget())));
    }

    private static int getActiveDaysCount(GdTimeTarget timeTarget){
        var timeBoard = timeTarget.getTimeBoard();
        return IntStream.rangeClosed(WeekdayType.MONDAY.getInternalNum(), WeekdayType.SUNDAY.getInternalNum())
                .map(day -> hasAnyWorkingHour(timeBoard.get(day - 1)) ? 1 : 0)
                .reduce(0, Integer::sum);
    }

    private static boolean hasAnyWorkingHour(List<Integer> workingHours){
        return workingHours.stream().anyMatch(hourCoeff -> Objects.equals(TimeTarget.PredefinedCoefs.USUAL.getValue(),
                hourCoeff));
    }

    private static boolean hasDayOrWeeklyBudget(GdCampaign gdCampaign){
        var budget = gdCampaign.getStrategy().getBudget();
        return budget != null && NumberUtils.greaterThanZero(budget.getSum()) &&
                (GdCampaignBudgetPeriod.DAY == budget.getPeriod() || GdCampaignBudgetPeriod.WEEK == budget.getPeriod());
    }

    private static boolean isCpmCampaign(GdCampaign gdCampaign){
        var type = gdCampaign.getType();
        return GdCampaignType.CPM_BANNER == type || GdCampaignType.CPM_DEALS == type ||
                GdCampaignType.CPM_PRICE == type || GdCampaignType.CPM_YNDX_FRONTPAGE == type;
    }

    private static GdCampaignSmsSettings extractGdSmsSettings(GdiCampaign campaign) {
        Set<SmsFlag> availableSmsFlags = getAvailableSmsFlags(isValidId(campaign.getWalletId()), campaign.getType());
        var smsTime = GridTimeUtils.toGdTimeInterval(campaign.getSmsTime());

        return new GdCampaignSmsSettings()
                .withEvents(mapSet(availableSmsFlags, flag -> toGdCampaignSmsEventInfo(flag, campaign.getSmsFlags())))
                .withSmsTime(smsTime);
    }

    static GdCampaignEmailSettings extractGdEmailSettings(GdiCampaign campaign) {
        var allowedEvents =
                getAvailableEmailEvents(campaign.getWalletId(), campaign.getType(), campaign.getManagerUserId());
        return new GdCampaignEmailSettings()
                .withEmail(campaign.getEmail())
                .withAllowedEvents(mapSet(allowedEvents, GdCampaignEmailEvent::fromSource))
                .withWarningBalance(campaign.getWarningBalance())
                .withSendAccountNews(campaign.getEnableSendAccountNews())
                .withXlsReady(campaign.getEnableOfflineStatNotice())
                .withStopByReachDailyBudget(campaign.getEnablePausedByDayBudgetEvent())
                .withCheckPositionInterval(campaign.getEnableCheckPositionEvent()
                        ? toGdCampaignCheckPositionInterval(campaign.getCheckPositionInterval())
                        : null);
    }

    static GdCampaignNotification extractGdNotificationData(GdiCampaign campaign) {
        return new GdCampaignNotification()
                .withEmailSettings(extractGdEmailSettings(campaign))
                .withSmsSettings(extractGdSmsSettings(campaign));
    }

    public static GdCampaignSmsEventInfo toGdCampaignSmsEventInfo(SmsFlag smsFlag,
                                                                  Set<SmsFlag> campaignEnableSmsFlags) {
        return toGdCampaignSmsEventInfo(smsFlag, campaignEnableSmsFlags.contains(smsFlag));
    }

    public static GdCampaignSmsEventInfo toGdCampaignSmsEventInfo(SmsFlag smsFlag, Boolean checked) {
        return new GdCampaignSmsEventInfo()
                .withEvent(toGdCampaignSmsEvent(smsFlag))
                .withChecked(checked);
    }

    static GdCampaignBrandSafety extractGdBrandSafetyData(GdiCampaign campaign) {
        return new GdCampaignBrandSafety()
                .withIsEnabled(extractBrandSafetyIsEnabled(campaign.getBrandSafetyCategories()))
                .withAdditionalCategories(extractBrandSafetyAdditionalCategories(campaign.getBrandSafetyCategories()));
    }

    public static GdCampaignCheckPositionInterval toGdCampaignCheckPositionInterval(CampaignWarnPlaceInterval checkPositionInterval) {
        return GdCampaignCheckPositionInterval.fromTypedValue(checkPositionInterval.name());
    }

    static UpdateCampMetrikaCountersRequest toUpdateCampMetrikaCountersRequest(
            GdUpdateCampaignMetrikaCounters input) {
        List<Long> metrikaCounters = integerToLongList(input.getMetrikaCounters());

        return new UpdateCampMetrikaCountersRequest()
                .withCids(input.getCampaignIds())
                .withMetrikaCounters(metrikaCounters);
    }

    static GdUpdateCampaignMetrikaCountersPayload toGdUpdateCampaignMetrikaCountersPayloadWithOnlyValidationResult(
            GdValidationResult preValidationResult) {
        return new GdUpdateCampaignMetrikaCountersPayload()
                .withUpdatedCampaigns(emptyList())
                .withSkippedCampaignIds(emptySet())
                .withValidationResult(preValidationResult);
    }

    public static List<? extends ModelChanges<? extends BaseCampaign>> toModelChangesStartDate(
            GdUpdateCampaignsStartDate input) {
        return StreamEx.of(input.getCampaignIds())
                .map(cid -> new ModelChanges<>(cid, CommonCampaign.class)
                        .processNotNull(input.getStartDate(), CommonCampaign.START_DATE)
                        .processNotNull(input.getFinishDate(), CommonCampaign.END_DATE))
                .toList();
    }

    /**
     * @param input содержит в поле getCampaignIds() только кампании типа: TEXT и DYNAMIC
     */
    public static List<? extends ModelChanges<? extends BaseCampaign>> toModelPromoExtensionChanges(
            GdUpdateCampaignsPromoExtension input) {
        return StreamEx.of(input.getCampaignIds())
                .map(cid -> new ModelChanges<>(cid, CampaignWithPromoExtension.class)
                        .process(input.getPromoExtensionId(), CampaignWithPromoExtension.PROMO_EXTENSION_ID))
                .toList();
    }

    public static List<? extends ModelChanges<? extends BaseCampaign>> toModelChangesDayBudget(
            GdUpdateCampaignsDayBudget input) {
        return StreamEx.of(input.getCampaignIds())
                .map(cid -> new ModelChanges<>(cid, CampaignWithCustomDayBudget.class)
                        .processNotNull(setScaleForCore(input.getDayBudget()), CampaignWithCustomDayBudget.DAY_BUDGET)
                        .processNotNull(toCampaignDayBudgetShowMode(input.getDayBudgetShowMode()),
                                CampaignWithCustomDayBudget.DAY_BUDGET_SHOW_MODE))
                .toList();
    }

    public static List<? extends ModelChanges<? extends BaseCampaign>> toModelChangesOrganization(
            GdUpdateCampaignsOrganization input) {
        return StreamEx.of(input.getCampaignIds())
                .map(cid -> new ModelChanges<>(cid, TextCampaign.class)
                        .process(input.getDefaultChainId(), TextCampaign.DEFAULT_CHAIN_ID)
                        .process(input.getDefaultPermalinkId(), TextCampaign.DEFAULT_PERMALINK_ID)
                        .process(input.getDefaultTrackingPhoneId(), TextCampaign.DEFAULT_TRACKING_PHONE_ID))
                .toList();
    }

    public static List<? extends ModelChanges<? extends BaseCampaign>> toModelChangesCampaignsPromotions(
            GdUpdateCampaignsPromotions input) {
        return StreamEx.of(input.getCampaignIds())
                .map(cid -> new ModelChanges<>(cid, CampaignWithCampaignsPromotions.class)
                        .processNotNull(
                                toCampaignsPromotions(
                                        filterList(input.getCampaignsPromotions(), cp -> cp.getCid().equals(cid))
                                ),
                                CampaignWithCampaignsPromotions.CAMPAIGNS_PROMOTIONS))
                .toList();
    }

    public static List<CampaignsPromotion> toCampaignsPromotions(List<GdCampaignsPromotion> gdCampaignsPromotions) {
        return StreamEx.of(gdCampaignsPromotions)
                .map(cp -> new CampaignsPromotion()
                        .withCid(cp.getCid())
                        .withPromotionId(cp.getPromotionId())
                        .withStart(cp.getStart())
                        .withFinish(cp.getFinish())
                        .withPercent(cp.getPercent())
                ).toList();
    }

    public static List<? extends ModelChanges<? extends BaseCampaign>> toModelChangesBrandSurvey(
            GdUpdateCampaignsAddBrandSurvey input) {
        return StreamEx.of(input.getCampaignIds())
                .map(cid -> new ModelChanges<>(cid, CampaignWithBrandLift.class)
                        .processNotNull(input.getBrandSurveyId(), CampaignWithBrandLift.BRAND_SURVEY_ID)
                        .processNotNull(input.getBrandSurveyName(), CampaignWithBrandLift.BRAND_SURVEY_NAME))
                .toList();
    }

    public static List<? extends ModelChanges<? extends BaseCampaign>> toModelChangesBannerHrefParams(
            GdUpdateCampaignsAddBannerHrefParams input) {
        return StreamEx.of(input.getCampaignIds())
                .map(cid -> new ModelChanges<>(cid, CampaignWithBannerHrefParams.class)
                        .process(input.getBannerHrefParams(), CampaignWithBannerHrefParams.BANNER_HREF_PARAMS))
                .toList();
    }

}
