package ru.yandex.direct.grid.processing.service.campaign;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateMeaningfulGoals;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdGoalsForUpdateCampaigns;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdGoalsForUpdateCampaignsPayload;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdMetrikaGoalsByCounterPayload;
import ru.yandex.direct.grid.processing.service.goal.GoalMutationService;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class CampaignGoalsMassGraphQLService {

    private final CampaignMutationService campaignMutationService;
    private final GoalMutationService goalMutationService;

    @Autowired
    public CampaignGoalsMassGraphQLService(CampaignMutationService campaignMutationService,
                                           GoalMutationService goalMutationService) {
        this.campaignMutationService = campaignMutationService;
        this.goalMutationService = goalMutationService;
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateMeaningfulGoals")
    public GdUpdateCampaignPayload updateMeaningfulGoals(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input")GdUpdateMeaningfulGoals input) {
        return campaignMutationService.updateMeaningfulGoals(context, input);
    }

    /**
     * Получение целей, которые можно использовать для обновления кампаний.
     * Ответ содержит цели, доступные клиенту, а также цели со счетчиков, которые привязаны к кампаниям.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "getGoalsForUpdateCampaigns")
    public GdGoalsForUpdateCampaignsPayload getGoalsForUpdateCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGoalsForUpdateCampaigns input
    ) {
        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = context.getSubjectUser().getClientId();

        GdMetrikaGoalsByCounterPayload payload = goalMutationService.getMetrikaGoalsByAvailableAndCampaignCounters(
                operatorUid, clientId, input.getCampaignIds());
        return new GdGoalsForUpdateCampaignsPayload()
                .withRowset(payload.getGoals())
                .withIsMetrikaAvailable(payload.getIsMetrikaAvailable())
                .withTop1GoalId(payload.getTop1GoalId());
    }
}
