package ru.yandex.direct.grid.processing.service.campaign;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.assertj.core.util.Lists;
import org.jooq.tools.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem;
import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.brandlift.service.targetestimation.TargetEstimation;
import ru.yandex.direct.core.entity.brandlift.service.targetestimation.TargetEstimationsService;
import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStopReason;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheckDefaultImpl;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAllowedPageIds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithShows;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatusInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.BrandSurveyConditionsService;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyService;
import ru.yandex.direct.core.entity.campaign.service.TimeTargetStatusService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.core.AllowedTypesCampaignAccessibilityChecker;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramDetails;
import ru.yandex.direct.core.entity.cashback.service.CashbackClientsService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.measurers.repository.CampMeasurersRepository;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageBase;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.statistics.service.OrderStatService;
import ru.yandex.direct.core.entity.tag.model.CampaignTag;
import ru.yandex.direct.core.entity.tag.repository.TagRepository;
import ru.yandex.direct.core.entity.timetarget.service.GeoTimezoneMappingService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.VcardHelper;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.campaign.repository.GridMobileContentSuggestInfoRepository;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignAccessService;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.CampaignFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.core.entity.model.client.GdiClientInfo;
import ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdGoalStats;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdCampaignAggregatedStatusCountersInfo;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdCampaignAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdAutobudgetProblem;
import ru.yandex.direct.grid.model.campaign.GdBrandSurveyStatus;
import ru.yandex.direct.grid.model.campaign.GdBrandSurveyStatusCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaignAccess;
import ru.yandex.direct.grid.model.campaign.GdCampaignAgencyInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignManagerInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignMeasurer;
import ru.yandex.direct.grid.model.campaign.GdCampaignPrimaryStatus;
import ru.yandex.direct.grid.model.campaign.GdCampaignServicedState;
import ru.yandex.direct.grid.model.campaign.GdCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdCampaignStatus;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdCpmBannerCampaign;
import ru.yandex.direct.grid.model.campaign.GdDynamicCampaign;
import ru.yandex.direct.grid.model.campaign.GdMeaningfulGoal;
import ru.yandex.direct.grid.model.campaign.GdMobileContentSuggestInfo;
import ru.yandex.direct.grid.model.campaign.GdTextCampaign;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdiDayBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.GdiWalletAction;
import ru.yandex.direct.grid.model.campaign.facelift.GdCampaignAdditionalData;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudget;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudgetPeriod;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyAvgCpm;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyType;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.model.entity.campaign.GdCampWithAgencyInfo;
import ru.yandex.direct.grid.model.entity.campaign.GdCampWithManagerInfo;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractor;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorFacade;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.model.utils.GridModerationUtils;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignFilter;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignSourceAvailability;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsContainer;
import ru.yandex.direct.grid.processing.model.campaign.GdConversionStrategyLearningStatusData;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmBudgetLimit;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmBudgetLimitPayload;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignBrandSurveyCheckData;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignBrandSurveyCheckRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignBrandSurveyCheckResponseItem;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignDayBudgetLimitsRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignDayBudgetLimitsRequestItem;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmPriceCampaignBrandSurveyCheckData;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmPriceCampaignBrandSurveyCheckRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdMultipleCampaignsBrandSurveyCheckRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdMultipleCampaignsBrandSurveyCheckResponse;
import ru.yandex.direct.grid.processing.model.campaign.GdPayForConversionInfo;
import ru.yandex.direct.grid.processing.model.campaign.GdWallet;
import ru.yandex.direct.grid.processing.model.campaign.GdWalletAction;
import ru.yandex.direct.grid.processing.model.campaign.GdWalletBudget;
import ru.yandex.direct.grid.processing.model.campaign.GdWalletBudgetShowMode;
import ru.yandex.direct.grid.processing.model.campaign.GdWalletBudgetType;
import ru.yandex.direct.grid.processing.model.campaign.GdWalletCashBackBonus;
import ru.yandex.direct.grid.processing.model.campaign.GdWalletStatus;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdCampaignVcard;
import ru.yandex.direct.grid.processing.model.placement.GdPiPage;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendation;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionWithMetrikaFlag;
import ru.yandex.direct.grid.processing.model.tag.GdTag;
import ru.yandex.direct.grid.processing.service.aggregatedstatuses.CampaignPrimaryStatusCalculator;
import ru.yandex.direct.grid.processing.service.campaign.loader.BannerGeoLegalFlagsInAggrStatusDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignAdsCountDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignAgencyInfoByAgencyUidDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignBrandSurveyStatusDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignDomainsDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignHrefIsTurboDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignManagerInfoByManagerUidDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignPayForConversionInfoDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignsGroupsCountDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignsHasRunningUnmoderatedAdsDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignsLastChangedVcardIdDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.GetBrandSurveyStatusKey;
import ru.yandex.direct.grid.processing.service.client.ClientDataService;
import ru.yandex.direct.grid.processing.service.client.converter.VcardDataConverter;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.grid.processing.service.placement.PlacementDataService;
import ru.yandex.direct.grid.processing.service.showcondition.retargeting.RetargetingConditionDataLoader;
import ru.yandex.direct.grid.processing.service.tags.TagConverter;
import ru.yandex.direct.grid.processing.util.GeoTreeUtils;
import ru.yandex.direct.grid.processing.util.GoalHelper;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.Counter;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.lang3.BooleanUtils.isTrue;
import static ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupStatesEnum.HAS_NO_EFFECTIVE_GEO;
import static ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupStatesEnum.HAS_RESTRICTED_GEO;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID;
import static ru.yandex.direct.core.entity.cashback.service.CashbackClientsService.sumValues;
import static ru.yandex.direct.core.entity.retargeting.model.Goal.MOBILE_GOAL_IDS;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.currency.CurrencyCode.DONT_SHOW_WALLET_NDS_IN_INTERFACE;
import static ru.yandex.direct.feature.FeatureName.ENABLE_REWORKED_RECOMMENDATIONS;
import static ru.yandex.direct.feature.FeatureName.HIDE_OLD_SHOW_CAMPS_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.REDESIGN_BUDGET_LIMIT_EDITOR;
import static ru.yandex.direct.feature.FeatureName.SHOW_AGGREGATED_STATUS_OPEN_BETA;
import static ru.yandex.direct.feature.FeatureName.SHOW_DNA_BY_DEFAULT;
import static ru.yandex.direct.grid.core.entity.campaign.repository.GridCampaignRepository.convertBrandSurveyStatus;
import static ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService.GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.getPromocodeRestrictedDomainViews;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignImplementation;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignMeasurer;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignType;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdTimeTarget;
import static ru.yandex.direct.grid.processing.service.autooverdraft.converter.AutoOverdraftDataConverter.toClientAutoOverdraftInfo;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.extractGdBrandSafetyData;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.extractGdNotificationData;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.CAMPAIGN_FILTER_PROCESSOR;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.CAMPAIGN_FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.INT_CAMPAIGN_FILTER_PROCESSOR;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.INT_CAMPAIGN_FILTER_PROCESSOR_WITHOUT_ARCHIVED_FILTER;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.calculateCampaignSourcesAvailability;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.getComparator;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.getRedirectInfo;
import static ru.yandex.direct.grid.processing.service.campaign.converter.AutobudgetProblemConverter.toGdAutobudgetProblem;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toCampaignStrategy;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toGdPayForConversionInfo;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdiClientInfo;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdTimeTarget;
import static ru.yandex.direct.grid.processing.service.group.AvailableAdGroupTypesCalculator.getAvailableAdGroupTypesForCampaign;
import static ru.yandex.direct.grid.processing.util.StatHelper.applyGoalsStatFilters;
import static ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.regions.Region.KAZAKHSTAN_REGION_ID;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.Predicates.notNull;

/**
 * Получение информации о кампаниях клиентов из различных источников
 */
@Service
@ParametersAreNonnullByDefault
public class CampaignInfoService {
    private static final Logger logger = LoggerFactory.getLogger(CampaignInfoService.class);

    public static final GdCampaignAccess DEFAULT_ACCESS = new GdCampaignAccess()
            .withActions(Collections.emptySet())
            .withPseudoActions(Collections.emptySet())
            .withServicedState(GdCampaignServicedState.SELF_SERVICED)
            .withNoActions(true)
            .withCanEdit(false);

    public static final Set<GdCampaignStrategyType> STRATEGY_TYPES_WITH_CUSTOM_PERIOD_FOR_CALCULATING_BUDGET_LIMITS =
            Set.of(
                    GdCampaignStrategyType.CPM_MAX_IMPRESSIONS_CUSTOM_PERIOD,
                    GdCampaignStrategyType.CPM_MAX_REACH_CUSTOM_PERIOD
            );

    public static final Set<GdCampaignStrategyType> STRATEGY_TYPES_WITH_WEEK_PERIOD_FOR_CALCULATING_BUDGET_LIMITS =
            Set.of(
                    GdCampaignStrategyType.CPM_MAX_IMPRESSIONS,
                    GdCampaignStrategyType.CPM_MAX_REACH
            );
    public static final Set<GdiRecommendationType> RECOMMENDATION_REWORKED_BUDGET_TYPES =
            Set.of(
                    GdiRecommendationType.increaseStrategyWeeklyBudget,
                    GdiRecommendationType.dailyBudget
            );

    private static final int LATEST_CAMPAIGNS_LOOKUP_COUNT = 5;

    private final GridCampaignService gridCampaignService;
    private final GridCampaignAggregationFieldsService gridCampaignAggregationFieldsService;
    private final GridCampaignStrategyService gridCampaignStrategyService;
    private final CampaignStrategyService campaignStrategyService;
    private final CampaignAccessService campaignAccessService;
    private final CampaignValidationService campaignValidationService;
    private final GridRecommendationService gridRecommendationService;
    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ClientService clientService;
    private final ShardHelper shardHelper;
    private final CampaignsLastChangedVcardIdDataLoader campaignsLastChangedVcardIdDataLoader;
    private final CampaignAdsCountDataLoader campaignAdsCountDataLoader;
    private final CampaignsGroupsCountDataLoader campaignsGroupsCountDataLoader;
    private final CampaignsHasRunningUnmoderatedAdsDataLoader campaignsHasRunningUnmoderatedBannersDataLoader;
    private final CampaignPayForConversionInfoDataLoader campaignPayForConversionInfoDataLoader;
    private final RetargetingConditionDataLoader retargetingConditionDataLoader;
    private final CampaignHrefIsTurboDataLoader campaignHrefIsTurboDataLoader;
    private final CampaignDomainsDataLoader campaignDomainsDataLoader;
    private final BannerGeoLegalFlagsInAggrStatusDataLoader bannerGeoLegalFlagsInAggrStatusDataLoader;
    private final CampaignAgencyInfoByAgencyUidDataLoader campaignAgencyInfoByAgencyUidDataLoader;
    private final CampaignManagerInfoByManagerUidDataLoader campaignManagerInfoByManagerUidDataLoader;
    private final CampaignBrandSurveyStatusDataLoader campaignBrandSurveyStatusDataLoader;
    private final VcardHelper vcardHelper;
    private final VcardDataLoader vcardDataLoader;
    private final ConversionStrategyLearningStatusDataLoader conversionStrategyLearningStatusDataLoader;
    private final FeatureService featureService;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;
    private final AdGroupRepository adGroupRepository;
    private final TagRepository tagRepository;
    private final MetrikaGoalsService metrikaGoalsService;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageService pricePackageService;
    private final RegionDescriptionLocalizer regionDescriptionLocalizer;
    private final GridContextProvider gridContextProvider;
    private final GdStrategyExtractorFacade strategyExtractorFacade;
    private final BrandSurveyConditionsService brandSurveyConditionsService;
    private final TimeTargetStatusService timeTargetStatusService;
    private final GeoTimezoneMappingService geoTimezoneMappingService;
    private final BannerCommonRepository bannerCommonRepository;
    private final PricePackageRepository pricePackageRepository;
    private final OrderStatService orderStatService;
    private final AutobudgetAlertService autobudgetAlertService;
    private final GridMobileContentSuggestInfoRepository mobileContentInfoRepository;
    private final CashbackClientsService cashbackClientsService;
    private final ClientDataService clientDataService;
    private final PlacementDataService placementDataService;
    private final RbacService rbacService;
    private final CampMeasurersRepository measurersRepository;
    private final TargetEstimationsService targetEstimationsService;
    private final DirectConfig brandLiftConfig;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public CampaignInfoService(GridCampaignService gridCampaignService,
                               GridCampaignAggregationFieldsService gridCampaignAggregationFieldsService,
                               GridCampaignStrategyService gridCampaignStrategyService,
                               CampaignStrategyService campaignStrategyService,
                               CampaignAccessService campaignAccessService,
                               CampaignValidationService campaignValidationService,
                               GridRecommendationService gridRecommendationService,
                               CampaignRepository campaignRepository,
                               CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                               ClientService clientService,
                               ShardHelper shardHelper,
                               CampaignsLastChangedVcardIdDataLoader campaignsLastChangedVcardIdDataLoader,
                               CampaignAdsCountDataLoader campaignAdsCountDataLoader,
                               CampaignsGroupsCountDataLoader campaignsGroupsCountDataLoader,
                               CampaignsHasRunningUnmoderatedAdsDataLoader campaignsHasRunningUnmoderatedBannersDataLoader,
                               CampaignPayForConversionInfoDataLoader campaignPayForConversionInfoDataLoader,
                               RetargetingConditionDataLoader retargetingConditionDataLoader,
                               CampaignHrefIsTurboDataLoader campaignHrefIsTurboDataLoader,
                               CampaignDomainsDataLoader campaignDomainsDataLoader,
                               BannerGeoLegalFlagsInAggrStatusDataLoader bannerGeoLegalFlagsInAggrStatusDataLoader,
                               CampaignAgencyInfoByAgencyUidDataLoader campaignAgencyInfoByAgencyUidDataLoader,
                               CampaignManagerInfoByManagerUidDataLoader campaignManagerInfoByManagerUidDataLoader,
                               CampaignBrandSurveyStatusDataLoader campaignBrandSurveyStatusDataLoader,
                               VcardHelper vcardHelper,
                               VcardDataLoader vcardDataLoader,
                               ConversionStrategyLearningStatusDataLoader conversionStrategyLearningStatusDataLoader,
                               FeatureService featureService,
                               AggregatedStatusesViewService aggregatedStatusesViewService,
                               AdGroupRepository adGroupRepository,
                               TagRepository tagRepository,
                               MetrikaGoalsService metrikaGoalsService,
                               CampaignTypedRepository campaignTypedRepository,
                               PricePackageService pricePackageService,
                               RegionDescriptionLocalizer regionDescriptionLocalizer,
                               GridContextProvider gridContextProvider,
                               GdStrategyExtractorFacade strategyExtractorFacade,
                               BrandSurveyConditionsService brandSurveyConditionsService,
                               TimeTargetStatusService timeTargetStatusService,
                               GeoTimezoneMappingService geoTimezoneMappingService,
                               BannerCommonRepository bannerCommonRepository,
                               PricePackageRepository pricePackageRepository,
                               OrderStatService orderStatService,
                               AutobudgetAlertService autobudgetAlertService,
                               GridMobileContentSuggestInfoRepository mobileContentInfoRepository,
                               CashbackClientsService cashbackClientsService,
                               ClientDataService clientDataService,
                               PlacementDataService placementDataService,
                               RbacService rbacService,
                               CampMeasurersRepository measurersRepository,
                               TargetEstimationsService targetEstimationsService,
                               DirectConfig directConfig, PpcPropertiesSupport ppcPropertiesSupport) {
        this.gridCampaignService = gridCampaignService;
        this.gridCampaignAggregationFieldsService = gridCampaignAggregationFieldsService;
        this.gridCampaignStrategyService = gridCampaignStrategyService;
        this.campaignStrategyService = campaignStrategyService;
        this.campaignAccessService = campaignAccessService;
        this.campaignValidationService = campaignValidationService;
        this.gridRecommendationService = gridRecommendationService;
        this.campaignRepository = campaignRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.clientService = clientService;
        this.shardHelper = shardHelper;
        this.campaignsLastChangedVcardIdDataLoader = campaignsLastChangedVcardIdDataLoader;
        this.campaignAdsCountDataLoader = campaignAdsCountDataLoader;
        this.campaignsGroupsCountDataLoader = campaignsGroupsCountDataLoader;
        this.campaignsHasRunningUnmoderatedBannersDataLoader = campaignsHasRunningUnmoderatedBannersDataLoader;
        this.campaignPayForConversionInfoDataLoader = campaignPayForConversionInfoDataLoader;
        this.retargetingConditionDataLoader = retargetingConditionDataLoader;
        this.campaignHrefIsTurboDataLoader = campaignHrefIsTurboDataLoader;
        this.campaignDomainsDataLoader = campaignDomainsDataLoader;
        this.bannerGeoLegalFlagsInAggrStatusDataLoader = bannerGeoLegalFlagsInAggrStatusDataLoader;
        this.campaignAgencyInfoByAgencyUidDataLoader = campaignAgencyInfoByAgencyUidDataLoader;
        this.campaignManagerInfoByManagerUidDataLoader = campaignManagerInfoByManagerUidDataLoader;
        this.campaignBrandSurveyStatusDataLoader = campaignBrandSurveyStatusDataLoader;
        this.vcardHelper = vcardHelper;
        this.vcardDataLoader = vcardDataLoader;
        this.conversionStrategyLearningStatusDataLoader = conversionStrategyLearningStatusDataLoader;
        this.featureService = featureService;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.adGroupRepository = adGroupRepository;
        this.tagRepository = tagRepository;
        this.metrikaGoalsService = metrikaGoalsService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageService = pricePackageService;
        this.regionDescriptionLocalizer = regionDescriptionLocalizer;
        this.gridContextProvider = gridContextProvider;
        this.strategyExtractorFacade = strategyExtractorFacade;
        this.brandSurveyConditionsService = brandSurveyConditionsService;
        this.timeTargetStatusService = timeTargetStatusService;
        this.geoTimezoneMappingService = geoTimezoneMappingService;
        this.bannerCommonRepository = bannerCommonRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.orderStatService = orderStatService;
        this.autobudgetAlertService = autobudgetAlertService;
        this.mobileContentInfoRepository = mobileContentInfoRepository;
        this.cashbackClientsService = cashbackClientsService;
        this.clientDataService = clientDataService;
        this.placementDataService = placementDataService;
        this.rbacService = rbacService;
        this.measurersRepository = measurersRepository;
        this.targetEstimationsService = targetEstimationsService;
        this.brandLiftConfig = directConfig.getBranch("brand_lift");
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    public Map<Long, GdiCampaign> getAllCampaignsMap(ClientId clientId) {
        ImmutableList<GdiCampaign> allCampaigns = getAllCampaigns(clientId);
        return listToMap(allCampaigns, GdiCampaign::getId);
    }

    /**
     * Получить список всех кампаний заданного клиента во внутреннем представлении
     * Необходимые параметры берутся из контекста
     * {@link ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider}
     */
    public ImmutableList<GdiCampaign> getAllCampaigns(ClientId clientId) {
        GridGraphQLContext context = gridContextProvider.getGridContext();
        GdiClientInfo clientInfo = toGdiClientInfo(context.getQueriedClient());
        checkState(Objects.equals(clientId.asLong(), clientInfo.getId()),
                "clientId from params not equals clientId from context");

        if (context.getClientGdiCampaigns() == null) {
            //noinspection ConstantConditions
            List<GdiCampaign> campaigns = gridCampaignService.getAllCampaigns(clientInfo.getShard(),
                    context.getSubjectUser(), clientInfo, context.getOperator());
            context.setClientGdiCampaigns(ImmutableList.copyOf(campaigns));
        }
        return context.getClientGdiCampaigns();
    }

    public Map<Long, GdiBaseCampaign> getAllBaseCampaignsMap(ClientId clientId) {
        List<GdiBaseCampaign> allCampaigns = getAllBaseCampaigns(clientId);
        return listToMap(allCampaigns, GdiBaseCampaign::getId);
    }
    /**
     * Получить список всех кампаний заданного клиента во внутреннем представлении.
     * Если в контексте есть полная информация о кампаниях, то вернется она.
     * Иначе вернется базовая информация о кампаниях.
     * Необходимые параметры берутся из контекста
     * {@link ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider}
     */
    public List<GdiBaseCampaign> getAllBaseCampaigns(ClientId clientId) {
        GridGraphQLContext context = gridContextProvider.getGridContext();
        GdiClientInfo clientInfo = toGdiClientInfo(context.getQueriedClient());
        checkState(Objects.equals(clientId.asLong(), clientInfo.getId()),
                "clientId from params not equals clientId from context");

        if (context.getClientGdiCampaigns() != null) {
            return (List<GdiBaseCampaign>) (List<? extends GdiBaseCampaign>) context.getClientGdiCampaigns();
        }
        if (context.getClientGdiBaseCampaigns() == null) {
            //noinspection ConstantConditions
            List<GdiBaseCampaign> campaigns = gridCampaignService.getAllBaseCampaigns(clientInfo.getShard(),
                    context.getSubjectUser(), clientInfo, context.getOperator());
            context.setClientGdiBaseCampaigns(campaigns);
        }
        return context.getClientGdiBaseCampaigns();
    }

    /**
     * Получить список всех кампаний без кошельков заданного клиента во внутреннем представлении
     */
    public List<GdiBaseCampaign> getAllBaseCampaignsWithoutWallets(ClientId clientId) {
        List<GdiBaseCampaign> allCampaigns = getAllBaseCampaigns(clientId);
        return filterList(allCampaigns, not(CampaignServiceUtils::isWallet));
    }

    /**
     * Получить кампании заданного клиента и его кошельки во внутреннем представлении и записать в контекст
     * Необходимые параметры берутся из контекста
     * {@link ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider}
     */
    public Map<Long, GdiCampaign> getCampaignsAndWalletsAndSaveToContext(ClientId clientId, Set<Long> campaignIds) {
        GridGraphQLContext context = gridContextProvider.getGridContext();
        var operator = context.getOperator();
        GdiClientInfo clientInfo = toGdiClientInfo(context.getQueriedClient());
        checkState(Objects.equals(clientId.asLong(), clientInfo.getId()),
                "clientId from params not equals clientId from context");

        if (context.getClientGdiCampaigns() != null) {
            return listToMap(filterList(context.getClientGdiCampaigns(), gdiCampaign ->
                            campaignIds.contains(gdiCampaign.getId()) || CampaignServiceUtils.isWallet(gdiCampaign)),
                    GdiCampaign::getId);
        }
        HashMap<Long, GdiCampaign> result = new HashMap<>();
        Set<Long> campaignIdsRemaining = new HashSet<>(campaignIds);
        if (context.getGdiCampaignsMap() != null) {
            result.putAll(
                    EntryStream.of(context.getGdiCampaignsMap())
                            .filterValues(gdiCampaign -> campaignIds.contains(gdiCampaign.getId()) ||
                                    CampaignServiceUtils.isWallet(gdiCampaign))
                            .toMap()
            );
            campaignIdsRemaining.removeAll(result.keySet());
        }
        if (campaignIdsRemaining.isEmpty()) {
            return result;
        }

        //noinspection ConstantConditions
        List<GdiCampaign> campaignsAndAllWallets = gridCampaignService.getCampaignsAndAllWallets(clientInfo.getShard(),
                campaignIdsRemaining, context.getSubjectUser(), clientInfo, operator);
        if (context.getGdiCampaignsMap() == null) {
            context.setGdiCampaignsMap(listToMap(campaignsAndAllWallets, GdiCampaign::getId));
        } else {
            campaignsAndAllWallets.forEach(gdiCampaign ->
                    context.getGdiCampaignsMap().put(gdiCampaign.getId(), gdiCampaign));
        }
        result.putAll(
                StreamEx.of(campaignsAndAllWallets)
                        .filter(gdiCampaign -> campaignIds.contains(gdiCampaign.getId()) ||
                                CampaignServiceUtils.isWallet(gdiCampaign))
                        .mapToEntry(GdiCampaign::getId, Function.identity())
                        .toMap((a, b) -> a)
        );

        return result;
    }

    /**
     * Получить список кампаний во внешнем формате для выдачи клиенту
     *
     * @param client                        клиент
     * @param operator                      параметры оператора
     * @param campaigns                     список кампаний клиента
     * @param inputContainer                входные параметры для получения кампаний
     * @param currentInstant                момент времени, на который идет расчет
     * @param campaignFetchedFieldsResolver структура содержащая частичную информацию о том, какие поля запрошены на
     *                                      верхнем уровне
     */
    List<GdCampaign> getFilteredCampaigns(GdClientInfo client, User operator,
                                          List<GdiCampaign> campaigns, GdCampaignsContainer inputContainer,
                                          Instant currentInstant,
                                          CampaignFetchedFieldsResolver campaignFetchedFieldsResolver) {
        var filterWithDefaultConditions = addDefaultConditionsToFilter(inputContainer.getFilter(),
                ClientId.fromLong(client.getId()));

        boolean withFilterByAggrStatus = featureService.isEnabledForClientId(operator.getClientId(),
                SHOW_DNA_BY_DEFAULT)
                || featureService.isEnabledForClientId(operator.getClientId(), HIDE_OLD_SHOW_CAMPS_FOR_DNA)
                || featureService.isEnabledForClientId(operator.getClientId(), SHOW_AGGREGATED_STATUS_OPEN_BETA);

        var subCampaignsToMasterCidMap = StreamEx.of(campaigns)
                .filter(Predicates.compose(Objects::isNull, GdiCampaign::getMasterCid))
                .mapToEntry(Function.identity(), campaignKey -> filterList(campaigns,
                        c -> Objects.equals(c.getMasterCid(), campaignKey.getId())))
                .filterValues(subCampaigns -> !CollectionUtils.isEmpty(subCampaigns))
                .mapKeys(GdiCampaign::getId)
                .toMap();

        List<GdiCampaign> filteredCampaigns = filterList(campaigns,
                Predicates.compose(Objects::isNull, GdiCampaign::getMasterCid));

        filteredCampaigns = filterList(filteredCampaigns, c -> (withFilterByAggrStatus ?
                INT_CAMPAIGN_FILTER_PROCESSOR_WITHOUT_ARCHIVED_FILTER :
                INT_CAMPAIGN_FILTER_PROCESSOR).test(filterWithDefaultConditions, c));

        if (withFilterByAggrStatus) {
            // TODO: избавиться от старого фильтра и/или archived в DIRECT-135836 (и в конце метода тоже)
            filteredCampaigns = filterByPrimaryStatuses(filteredCampaigns, inputContainer.getFilter());
        }
        filteredCampaigns = filterByFilterStatuses(filteredCampaigns, inputContainer.getFilter());
        filteredCampaigns = filterByAccess(filteredCampaigns, ClientId.fromLong(client.getId()));
        filteredCampaigns = filterBySource(filteredCampaigns, inputContainer.getFilter());
        filteredCampaigns = filterByWidgetPartnerId(filteredCampaigns, inputContainer.getFilter());

        Set<Long> campaignIds = listToSet(filteredCampaigns, GdiCampaign::getId);

        GdStatRequirements statRequirements = inputContainer.getStatRequirements();

        boolean needStats = campaignFetchedFieldsResolver.getStats()
                || filterWithDefaultConditions.getStats() != null
                || filterWithDefaultConditions.getGoalStats() != null;

        Set<Long> goalIds =
                GoalHelper.combineGoalIds(statRequirements.getGoalIds(), filterWithDefaultConditions.getGoalStats());
        boolean useCampaignGoalIds = Boolean.TRUE.equals(statRequirements.getUseCampaignGoalIds());

        ClientId clientId = ClientId.fromLong(client.getId());
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);

        boolean enabled = enabledFeatures.contains(FeatureName.GOALS_ONLY_WITH_CAMPAIGN_COUNTERS_USED.getName())
                && enabledFeatures.contains(FeatureName.GRID_GOALS_FILTRATION_FOR_STAT.getName());

        Map<Long, GdiCampaignStats> statsInternal = emptyMap();
        if (needStats) {
            Set<Long> availableGoalIds = null;
            if (enabledFeatures.contains(FeatureName.GRID_CAMPAIGN_GOALS_FILTRATION_FOR_STAT.getName()) ||
                    enabledFeatures.contains(FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS.getName())) {
                availableGoalIds = metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                        operator.getUid(), clientId);
            }
            if (useCampaignGoalIds) {
                statsInternal = gridCampaignService.getCampaignStatsForCampaignGoals(filteredCampaigns,
                        statRequirements.getFrom(), statRequirements.getTo(),
                        availableGoalIds, enabledFeatures);
            } else if (!enabled) {
                statsInternal = gridCampaignService.getCampaignStats(filteredCampaigns,
                        statRequirements.getFrom(), statRequirements.getTo(),
                        goalIds, availableGoalIds, enabledFeatures);
            } else {
                // эта ветка deprecated, т.к. фичу grid_goals_filtration_for_stat намеренно не включили в проде
                List<Goal> availableGoals;
                try {
                    availableGoals = metrikaGoalsService.getGoalsWithCounters(operator.getUid(), clientId, null);
                } catch (MetrikaClientException | InterruptedRuntimeException e) {
                    availableGoals = emptyList();
                    logger.warn("Got an exception when querying for metrika counters and goals for clientId: "
                            + clientId, e);
                }

                Set<Long> availableMetrikaGoalIds = listToSet(availableGoals, GoalBase::getId);

                statsInternal = gridCampaignService.getCampaignStatsWithGoalFiltering(
                        campaignIds, statRequirements.getFrom(), statRequirements.getTo(), goalIds,
                        Sets.union(availableMetrikaGoalIds, MOBILE_GOAL_IDS));
            }
        }

        Map<Long, CampaignType> campaignIdToGdiType =
                listToMap(filteredCampaigns, GdiCampaign::getId, GdiCampaign::getType);
        Map<Long, GdCampaignType> campaignTypes = EntryStream.of(campaignIdToGdiType)
                .mapValues(gdiCampaignType -> toGdCampaignType(gdiCampaignType))
                .toMap();

        // Скрываем некоторые поля статистики, в зависимости от типа кампании
        if (enabledFeatures.contains(FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED.getName())) {
            for (Map.Entry<Long, GdiCampaignStats> campaignIdToStats : statsInternal.entrySet()) {
                CampaignType campaignType = campaignIdToGdiType.get(campaignIdToStats.getKey());
                GdiEntityStats stats = campaignIdToStats.getValue().getStat();

                if (CampaignServiceUtils.CPC_CAMPAIGN_TYPES.contains(campaignType)) {
                    stats.withCpmPrice(null);
                } else if (CampaignServiceUtils.CPM_CAMPAIGN_TYPES.contains(campaignType)) {
                    stats.withAvgClickCost(null);
                }
            }
        }

        Map<Long, GdEntityStats> idToStats = listToMap(statsInternal.entrySet(), Map.Entry::getKey,
                e -> StatHelper.internalStatsToOuter(e.getValue().getStat(), campaignTypes.get(e.getKey())));

        Map<Long, List<GdGoalStats>> idToGoalStats = listToMap(statsInternal.entrySet(), Map.Entry::getKey,
                e -> mapList(e.getValue().getGoalStats(), StatHelper::internalGoalStatToOuter));

        Map<Long, Boolean> idToStatIsRestricted = EntryStream.of(statsInternal)
                .mapValues(GdiCampaignStats::getIsRestrictedByUnavailableGoals)
                .nonNullValues()
                .toMap();

        Set<GdiRecommendationType> recommendationTypes = Optional.ofNullable(inputContainer.getFilter())
                .map(GdCampaignFilter::getRecommendations)
                .orElse(emptySet());
        boolean isReworkedRecommendationsEnabled = featureService
                .isEnabledForClientId(clientId, ENABLE_REWORKED_RECOMMENDATIONS);
        boolean addReworkedRecommendationTypes = isReworkedRecommendationsEnabled &&
                //если рекомендации запросили, но фильтр пустой, берём сразу все типы рекомендаций, т. е. список дополнять не надо
                !(campaignFetchedFieldsResolver.getRecommendations() && recommendationTypes.isEmpty());
        Map<Long, List<GdRecommendation>> campaignIdToRecommendations =
                campaignFetchedFieldsResolver.getRecommendations()
                        || !recommendationTypes.isEmpty()  || addReworkedRecommendationTypes
                        ? getRecommendations(client.getId(), operator,
                        addReworkedRecommendationTypes
                                    ? Sets.union(recommendationTypes, RECOMMENDATION_REWORKED_BUDGET_TYPES)
                                    : recommendationTypes,
                        campaignIds)
                        : Collections.emptyMap();
        var campaignIdToFilteredRecommendations = filterCidToRecommendations(
                recommendationTypes, campaignFetchedFieldsResolver.getRecommendations(), campaignIdToRecommendations);

        Counter counter = new Counter();

        Set<Long> contentPromotionCampaignIds = StreamEx.of(filteredCampaigns)
                .filter(gdiCampaign -> gdiCampaign.getType().equals(CampaignType.CONTENT_PROMOTION))
                .map(GdiCampaign::getId)
                .toSet();
        Map<Long, ContentPromotionAdgroupType> contentPromotionTypeByCampaignId =
                adGroupRepository.getContentPromotionAdGroupTypeByCampaignId(client.getShard(),
                        contentPromotionCampaignIds);

        //подгружаем пакеты. От них зависит какие группы доступны на кампании availableAdGroupTypes
        var pricePackageByCampaignId = pricePackageMapByCampaignId(client.getShard(), filteredCampaigns);

        List<Long> cpmBannerCampaignIds =
                filterAndMapList(filteredCampaigns, c -> c.getType() == CampaignType.CPM_BANNER, GdiCampaign::getId);
        Map<Long, List<AdGroupSimple>> adGroupSimpleByCpmCampaignsIds =
                adGroupRepository.getAdGroupSimpleByCampaignsIds(client.getShard(), cpmBannerCampaignIds);
        Map<Long, Boolean> isCpvStrategyEnabledForCpmCampaign = EntryStream.of(adGroupSimpleByCpmCampaignsIds)
                .mapValues(groups -> groups.stream().allMatch(g -> g.getType() == AdGroupType.CPM_VIDEO))
                .toMap();

        Map<Long, AutobudgetAggregatedHourlyProblem> autobudgetProblems = autobudgetAlertService
                .getActiveHourlyProblemsAlerts(client.getShard(), campaignIds);
        Map<Long, GdAutobudgetProblem> gdAutobudgetProblems = StreamEx.of(campaigns)
                .mapToEntry(GdiCampaign::getId, c -> toGdAutobudgetProblem(c, autobudgetProblems.get(c.getId())))
                .filterValues(Objects::nonNull)
                .toMap();

        return getTruncatedCampaignsStream(campaigns, client, toClientAutoOverdraftInfo(client),
                currentInstant, filteredCampaigns, operator, subCampaignsToMasterCidMap
        )
                .map(c -> c.withAvailableAdGroupTypes(getAvailableAdGroupTypesForCampaign(c, enabledFeatures,
                        contentPromotionTypeByCampaignId, pricePackageByCampaignId)))
                // Фильтрация синтетических типов
                .filter(c -> c.getType() != GdCampaignType.CONTENT_PROMOTION || !c.getAvailableAdGroupTypes().isEmpty())
                .map(c -> c.withStats(idToStats.get(c.getId())))
                .map(c -> c.withGoalStats(idToGoalStats.get(c.getId())))
                .map(c -> c.withStatsIsRestrictedByUnavailableGoals(idToStatIsRestricted.get(c.getId())))
                .filter(c -> (withFilterByAggrStatus ? CAMPAIGN_FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS :
                        CAMPAIGN_FILTER_PROCESSOR).test(inputContainer.getFilter(), c))
                .filter(c -> applyGoalsStatFilters(inputContainer.getFilter().getGoalStats(), c))
                .map(c -> isReworkedRecommendationsEnabled && campaignIdToRecommendations.containsKey(c.getId()) ?
                        addReworkedBudgetRecommendations(c, campaignIdToRecommendations.get(c.getId())) : c)
                .filter(c -> recommendationTypes.isEmpty() || campaignIdToFilteredRecommendations.containsKey(c.getId()))
                .map(c -> c.withRecommendations(campaignIdToFilteredRecommendations.get(c.getId())))
                .map(c -> c.withAutobudgetProblem(gdAutobudgetProblems.get(c.getId())))
                .map(c -> c.withRedirectInfo(getRedirectInfo(c)))
                .peek(c -> enrichCpmBannerCampaign(isCpvStrategyEnabledForCpmCampaign, c))
                .sorted(getComparator(inputContainer.getOrderBy()))
                .map(c -> c.withIndex(counter.next()))
                .toList();
    }

    private void enrichCpmBannerCampaign(Map<Long, Boolean> isCpvStrategyEnabledForCpmCampaign, GdCampaign c) {
        if (c.getType() == GdCampaignType.CPM_BANNER) {
            ((GdCpmBannerCampaign) c)
                    .withIsCpvStrategiesEnabled(isCpvStrategyEnabledForCpmCampaign.getOrDefault(c.getId(), true));
        }
    }

    private GdCampaign addReworkedBudgetRecommendations(GdCampaign campaign, List<GdRecommendation> recommendations) {
        var dailyBudgetRecommendation = StreamEx.of(recommendations)
                .filter(c -> c.getType() == GdiRecommendationType.dailyBudget)
                .findFirst();
        var increaseStrategyWeeklyBudgetRecommendation = StreamEx.of(recommendations)
                .filter(c -> c.getType() == GdiRecommendationType.increaseStrategyWeeklyBudget)
                .findFirst();
        dailyBudgetRecommendation.ifPresent(campaign::withDailyBudgetRecommendation);
        increaseStrategyWeeklyBudgetRecommendation.ifPresent(campaign::withIncreaseStrategyWeeklyBudgetRecommendation);
        return campaign;
    }

    /**
     * Если запрашивали фильтрацию типов рекомендаций, то нужно проверить, что лишние типы не возвращаются.
     * Это может произойти при включенной фиче ENABLE_REWORKED_RECOMMENDATIONS, когда мы также получаем рекомендации по
     * увеличению бюджета
     */
    private Map<Long, List<GdRecommendation>> filterCidToRecommendations(
            Set<GdiRecommendationType> recommendationTypes,
            boolean getRecommendations,
            Map<Long, List<GdRecommendation>> cidToRecommendations
    ) {
        //если не запрашивали никакие рекомендации, ничего не возвращаем
        if (!getRecommendations && recommendationTypes.isEmpty()) {
            return emptyMap();
        }
        return EntryStream.of(cidToRecommendations)
                .mapValues(recoms ->
                        filterList(recoms,
                                r -> recommendationTypes.isEmpty() || recommendationTypes.contains(r.getType())
                        )
                ).filterValues(Predicate.not(List::isEmpty))
                .toMap();
    }

    private List<GdiCampaign> filterByPrimaryStatuses(List<GdiCampaign> filteredCampaigns,
                                                      GdCampaignFilter filter) {
        Set<GdCampaignPrimaryStatus> statuses = new HashSet<>();
        if (filter.getStatusIn() != null) {
            statuses.addAll(filter.getStatusIn());
        }
        if (Boolean.TRUE.equals(filter.getArchived())) {
            statuses.add(GdCampaignPrimaryStatus.ARCHIVED);
        }
        // возможные сочетания фильтров: https://st.yandex-team.ru/DIRECT-121619#5ee9c10874ee912e33451070
        return filteredCampaigns.stream()
                .filter(c -> statuses.isEmpty()
                        || statuses.contains(CampaignPrimaryStatusCalculator.convertToPrimaryStatus(c)))
                .filter(c -> !Boolean.FALSE.equals(filter.getArchived())
                        || c.getAggregatedStatus() == null || c.getAggregatedStatus().getStatus().isEmpty()
                        || !Objects.equals(c.getAggregatedStatus().getStatus().get(), GdSelfStatusEnum.ARCHIVED))
                .collect(Collectors.toList());
    }

    private List<GdiCampaign> filterByFilterStatuses(List<GdiCampaign> filteredCampaigns,
                                                     GdCampaignFilter filter) {
        var statuses = filter.getFilterStatusIn();
        if (statuses == null || statuses.isEmpty()) {
            return filteredCampaigns;
        }

        return filteredCampaigns.stream()
                .filter(c -> statuses.contains(CampaignPrimaryStatusCalculator.convertToFilterStatus(c)))
                .collect(Collectors.toList());
    }

    private List<GdiCampaign> filterBySource(List<GdiCampaign> filteredCampaigns, GdCampaignFilter filter) {
        if (filter.getSource() == null) {
            return filteredCampaigns;
        }

        return filteredCampaigns
                .stream()
                .filter(c -> c.getSource().name().equals(filter.getSource().name()))
                .collect(Collectors.toList());
    }

    private List<GdiCampaign> filterByWidgetPartnerId(List<GdiCampaign> filteredCampaigns, GdCampaignFilter filter) {
        if (filter.getWidgetPartnerId() == null) {
            return filteredCampaigns;
        }

        return filteredCampaigns
                .stream()
                .filter(c -> filter.getWidgetPartnerId().equals(c.getWidgetPartnerId()))
                .collect(Collectors.toList());
    }

    private Map<Long, List<GdRecommendation>> getRecommendations(Long clientId, User operator,
                                                                 Set<GdiRecommendationType> recommendationTypes,
                                                                 Set<Long> campaignIds) {
        try (TraceProfile ignore = Trace.current().profile("recommendations:service:campaigns")) {
            return gridRecommendationService
                    .getCampaignRecommendations(clientId, operator, recommendationTypes, campaignIds);

        }
    }

    public CompletableFuture<GdRetargetingConditionWithMetrikaFlag> getRetargetingCondition(
            Long retargetingConditionId) {
        return retargetingConditionDataLoader.get().load(retargetingConditionId);
    }

    public CompletableFuture<Set<String>> getBannerGeoLegalFlagsInAggrStatus(
            GdCampaignAggregatedStatusInfo statusInfo) {
        if (nvl(statusInfo.getHasRestrictedGeo(), false)) {
            Long cid = statusInfo.getCid();
            return bannerGeoLegalFlagsInAggrStatusDataLoader.get().load(cid);
        } else {
            return CompletableFuture.completedFuture(null);
        }
    }

    /**
     * Получить ограниченное представление кампаний
     */
    public Map<Long, GdCampaignTruncated> getTruncatedCampaigns(ClientId clientId, Collection<Long> campaignIds) {
        GridGraphQLContext context = gridContextProvider.getGridContext();
        GdClientInfo client = context.getQueriedClient();
        checkState(Objects.equals(clientId.asLong(), client.getId()),
                "clientId from params not equals clientId from context");

        ImmutableList<GdiCampaign> allCampaigns = getAllCampaigns(clientId);
        List<GdiCampaign> filteredCampaigns = filterList(allCampaigns, c -> campaignIds.contains(c.getId()));

        return getTruncatedCampaignsStream(allCampaigns, client, toClientAutoOverdraftInfo(client),
                context.getInstant(), filteredCampaigns, context.getOperator(),
                emptyMap()).toMap(GdCampaign::getId, GdCampaignTruncated.class::cast);
    }

    private <T extends GdiBaseCampaign> StreamEx<GdCampaign> getTruncatedCampaignsStream(
            List<T> allCampaigns, GdClientInfo clientInfo,
            ClientAutoOverdraftInfo clientAutoOverdraftInfo, Instant currentInstant,
            List<GdiCampaign> filteredCampaigns, User operator,
            Map<Long, List<GdiCampaign>> subCampaignToMasterCidMap) {
        Map<Long, GdWallet> campaignWallets = extractWalletsMap(
                operator, clientInfo, clientAutoOverdraftInfo, (List<GdiBaseCampaign>) allCampaigns, currentInstant);
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        Map<Long, GdCampaignAccess> campaignAccessMap =
                campaignAccessService.getCampaignsAccess(operator, clientId, filteredCampaigns, campaignWallets);

        boolean aggrStatusesDebugEnabled = featureService
                .isEnabledForClientId(operator.getClientId(), FeatureName.SHOW_AGGREGATED_STATUS_DEBUG);
        return StreamEx.of(filteredCampaigns)
                .map(c -> toCampaignImplementation(c, campaignWallets.get(c.getWalletId()),
                        currentInstant, aggrStatusesDebugEnabled, operator,
                        subCampaignToMasterCidMap.get(c.getId())))
                .peek(c -> c.withAccess(campaignAccessMap.getOrDefault(c.getId(), DEFAULT_ACCESS)));
    }

    private List<GdiCampaign> filterByAccess(List<GdiCampaign> filteredCampaigns, ClientId clientId) {
        AllowedTypesCampaignAccessibilityChecker checker =
                campaignAccessService.getCampaignAccessibilityChecker(clientId);

        return StreamEx.of(filteredCampaigns)
                .mapToEntry(CampaignInfoService::toCampaignForAccess)
                .filterValues(checker::isAllowable)
                .keys()
                .toList();
    }

    protected static CampaignForAccessCheckDefaultImpl toCampaignForAccess(GdiCampaign campaign) {
        return new CampaignForAccessCheckDefaultImpl()
                .withId(campaign.getId())
                .withType(campaign.getType())
                .withArchived(campaign.getArchived())
                .withSource(CampaignSource.fromSource(GdiCampaignSource.toSource(campaign.getSource())));
    }

    // в java могут не поддерживаться опредленные фичи, уже используемые внутри директа,
    // чтобы не обрабатывать их некорректно, фильтруем их
    private GdCampaignFilter addDefaultConditionsToFilter(GdCampaignFilter currentFilter, ClientId clientId) {
        //создаем копию т.к. фильтр используется для кеширования
        GdCampaignFilter newFilter = currentFilter.copy();

        Set<GdCampaignType> currentFilteringTypes = newFilter.getTypeIn();

        Set<GdCampaignType> visibleTypes = campaignAccessService.getVisibleTypes(clientId);

        if (currentFilteringTypes == null) {
            return newFilter.withTypeIn(visibleTypes);
        }

        Set<GdCampaignType> resultFilteringTypes = StreamEx.of(currentFilteringTypes)
                .filter(visibleTypes::contains)
                .toSet();

        return newFilter.withTypeIn(resultFilteringTypes);
    }

    private GdCampaign toCampaignImplementation(GdiCampaign gdiCampaign,
                                                GdWallet wallet, Instant currentInstant,
                                                boolean aggrStatusesDebugEnabled,
                                                User operator, List<GdiCampaign> subCampaigns) {
        GdCampaignFlatStrategy campaignFlatStrategy = gridCampaignStrategyService.extractFlatStrategy(gdiCampaign);
        GdCampaignFlatStrategy campaignStrategy = gridCampaignStrategyService.extractStrategy(gdiCampaign);
        var clientId = ClientId.fromNullableLong(gdiCampaign.getClientId());
        processWeeklyBudget(campaignStrategy, campaignFlatStrategy, gdiCampaign,
                clientId);

        GdCampaignStatus campaignStatus =
                gridCampaignAggregationFieldsService.extractStatus(gdiCampaign, wallet, currentInstant);
        if (!CollectionUtils.isEmpty(subCampaigns)) {
            var totalBudget = StreamEx.of(subCampaigns)
                    .map(gridCampaignStrategyService::extractStrategy)
                    .map(strategy -> strategy.getBudget().getSum())
                    .reduce(BigDecimal.ZERO, BigDecimal::add)
                    .add(campaignStrategy.getBudget().getSum());
            campaignStrategy.getBudget().setSum(totalBudget);
        }
        final List<GdRecommendation> recommendations =
                gdiCampaign.getRecommendations() == null ? emptyList()
                        : gdiCampaign.getRecommendations().stream()
                        .map(GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION)
                        .collect(toList());

        List<MeaningfulGoal> meaningfulGoals = gdiCampaign.getMeaningfulGoals();
        //если ключевых целей нет отправляем цель -- вовлчеченные сессии
        if (isEmpty(meaningfulGoals)) {
            gdiCampaign.setMeaningfulGoals(List.of(new MeaningfulGoal().withGoalId(ENGAGED_SESSION_GOAL_ID)));
        }
        var advancedGeoTargeting = featureService.isEnabledForClientId(clientId, FeatureName.ADVANCED_GEOTARGETING);
        GdCampaign gdCampaign = toGdCampaignImplementation(gdiCampaign, campaignStatus, campaignFlatStrategy,
                campaignStrategy, recommendations,
                geo -> GeoTreeUtils.geoToFrontendGeo(geo, pricePackageService.getGeoTree(),
                        regionDescriptionLocalizer),
                advancedGeoTargeting
        );

        AggregatedStatusCampaignData status = gdiCampaign.getAggregatedStatus();
        if (aggrStatusesDebugEnabled) {
            gdCampaign.setAggregatedStatus(aggregatedStatusesViewService.toJson(status));
        }

        gdCampaign.setAggregatedStatusInfo(toAggregatedStatusInfo(gdiCampaign, currentInstant));

        setPromocodeRestrictedDomainFromWallet(gdCampaign, wallet);

        hideBrandLiftIfNeeded(gdCampaign, gdiCampaign, operator);

        var notificationData = extractGdNotificationData(gdiCampaign);
        var brandSafetyData = extractGdBrandSafetyData(gdiCampaign);
        return gdCampaign
                .withNotification(notificationData)
                .withBrandSafety(brandSafetyData);
    }

    /**
     * В рамках фичи новых рекомендаций поменялось представление недельного бюджета на гридах.
     * Ранее при не заданном недельном бюджета в поле budget отправлялся null.
     * При включении фичи в аналогичной ситуации начинаем отправлять непустой объект с нулевым значением поля sum
     * и period: WEEK. На фронте на основе strategy.period определяется тип попапа для отрисовки (дневной или недельный)
     *
     * Метод следует удалить после включения фичи REDESIGN_BUDGET_LIMIT_EDITOR
     */
    private void processWeeklyBudget(GdCampaignFlatStrategy campaignStrategy,
                                     GdCampaignFlatStrategy campaignFlatStrategy,
                                     GdiCampaign gdiCampaign,
                                     ClientId clientId){
        boolean isBudgetLimitEditorEnabled = clientId != null && featureService.isEnabledForClientId(clientId,
                REDESIGN_BUDGET_LIMIT_EDITOR);
        StrategyData strategyData = GdStrategyExtractor.extractStrategyData(gdiCampaign);
        boolean isEmptyWeeklyBudget = campaignStrategy.getBudget() != null &&
                campaignFlatStrategy.getBudget() != null &&
                GdCampaignBudgetPeriod.WEEK.equals(campaignStrategy.getBudget().getPeriod()) &&
                GdCampaignBudgetPeriod.WEEK.equals(campaignFlatStrategy.getBudget().getPeriod()) &&
                (strategyData == null || strategyData.getSum() == null);

        if (!isBudgetLimitEditorEnabled && isEmptyWeeklyBudget){
            campaignStrategy.setBudget(null);
            campaignFlatStrategy.setBudget(null);
        }
    }

    /**
     * Не показываем скрытый Brand Lift клиенту
     */
    private void hideBrandLiftIfNeeded(GdCampaign gdCampaign, GdiCampaign gdiCampaign, User operator) {
        boolean isBrandLiftHiddenEnabled = featureService.isEnabled(operator.getUid(), FeatureName.BRAND_LIFT_HIDDEN);
        if (!isBrandLiftHiddenEnabled && gdiCampaign.getIsBrandLiftHidden()) {
            gdCampaign.withBrandSurveyId(null);
            gdCampaign.withBrandSurveyName(null);
        }
    }

    /**
     * @return кампании с переданными id, а так же кошельки
     */
    public Map<Long, GdiCampaign> getCampaignsByIdsAndAllWallets(Set<Long> campaignIds, User subjectUser,
                                                                 User operator, GdClientInfo clientInfo) {
        return listToMap(filterList(gridCampaignService.getCampaignsAndAllWallets(
                                clientInfo.getShard(), campaignIds, subjectUser, toGdiClientInfo(clientInfo), operator),
                        gdiCampaign ->
                                campaignIds.contains(gdiCampaign.getId()) || CampaignServiceUtils.isWallet(gdiCampaign)),
                GdiCampaign::getId);
    }

    private GdClientInfo getGdClientInfo(User operator, ClientId clientId) {
        return clientDataService
                .getClientInfo(operator, Set.of(clientId.asLong()))
                .stream()
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("cant find client"));
    }

    /**
     * Получение модели доступа по кампании
     * НЕ ИСПОЛЬЗУЕТ контекст GridGraphQLContext, при работе с гридовым запросом лучше не использовать
     */
    public <T extends GdiBaseCampaign> Map<Long, GdCampaignAccess> getCampaignsAccess(
            Set<Long> campaignIds,
            Map<Long, T> gdiCampaignsAndWallets,
            User operator,
            GdClientInfo clientInfo,
            Instant instant
    ) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        Map<Long, GdWallet> campaignWallets = extractWalletsMap(
                operator, clientInfo, toClientAutoOverdraftInfo(clientInfo),
                (Collection<GdiBaseCampaign>) gdiCampaignsAndWallets.values(),
                instant);
        var accessMap = campaignAccessService.getCampaignsAccess(operator, clientId,
                filterList(gdiCampaignsAndWallets.values(), t -> t.getType() != CampaignType.WALLET), campaignWallets);
        return StreamEx.of(campaignIds)
                .mapToEntry(id -> accessMap.getOrDefault(id, DEFAULT_ACCESS))
                .toMap();
    }

    public GdCampaignAggregatedStatusInfo toAggregatedStatusInfo(GdiCampaign gdiCampaign,
                                                                 Instant currentInstant) {
        return toAggregatedStatusInfo(
                gdiCampaign.getAggregatedStatus(),
                gdiCampaign.getTimeTarget(),
                gdiCampaign.getTimezoneId(),
                currentInstant,
                gdiCampaign.getId());
    }

    private GdCampaignAggregatedStatusInfo toAggregatedStatusInfo(
            AggregatedStatusCampaignData aggregatedStatusCampaignData,
            TimeTarget timeTarget,
            Long timezoneId,
            Instant currentInstant,
            Long cid) {
        if (aggregatedStatusCampaignData == null ||
                aggregatedStatusCampaignData.getStatus().isEmpty()) {
            return null;
        }
        TimeTargetStatusInfo timeTargetStatus =
                timeTargetStatusService.getTimeTargetStatus(timeTarget,
                        geoTimezoneMappingService.getRegionIdByTimezoneId(timezoneId),
                        currentInstant);
        var statusInfo = new GdCampaignAggregatedStatusInfo()
                .withStatus(aggregatedStatusCampaignData.getStatus().get())
                .withReasons(aggregatedStatusCampaignData.getReasons())
                .withRejectReasons(GridModerationUtils.toGdRejectReasons(aggregatedStatusCampaignData.getRejectReasons()))
                .withIsObsolete(aggregatedStatusCampaignData.getIsObsolete())
                .withTimeTargetStatus(timeTargetStatus)
                .withCid(cid)
                .withHasRestrictedGeo(false)
                .withMightHaveRejectReasons(aggregatedStatusesViewService.campaignCouldHaveRejectReasons(aggregatedStatusCampaignData));

        if (aggregatedStatusCampaignData.getCounters() != null) {
            statusInfo.withCounters(new GdCampaignAggregatedStatusCountersInfo().withAdgroups(
                    aggregatedStatusCampaignData.getCounters().getStatuses()
            ));
            boolean hasRestrictedGeo =
                    aggregatedStatusCampaignData.getCounters().getStates()
                            .getOrDefault(HAS_RESTRICTED_GEO, 0) > 0
                            ||
                            aggregatedStatusCampaignData.getCounters().getStates()
                                    .getOrDefault(HAS_NO_EFFECTIVE_GEO, 0) > 0;
            statusInfo.withHasRestrictedGeo(hasRestrictedGeo);
        }
        return statusInfo;
    }

    /**
     * Если у кампании есть кошелек, то берем domain к которому привязан промокод из кошелька
     */
    private void setPromocodeRestrictedDomainFromWallet(GdCampaign gdCampaign, @Nullable GdWallet wallet) {
        if (wallet != null) {
            String promocodeRestrictedDomain = wallet.getPromocodeRestrictedDomain();
            gdCampaign.setPromocodeRestrictedDomain(getPromocodeRestrictedDomainViews(promocodeRestrictedDomain));
        }
    }

    private Stream<GdWallet> extractWalletsStream(User operator, GdClientInfo clientInfo,
                                                  ClientAutoOverdraftInfo clientAutoOverdraftInfo,
                                                  Collection<GdiBaseCampaign> campaigns,
                                                  Instant currentInstant) {
        Map<Long, List<GdiBaseCampaign>> walletIdToCampaign = StreamEx.of(campaigns)
                .filter(not(CampaignServiceUtils::isWallet))
                .groupingBy(GdiBaseCampaign::getWalletId);

        Set<Long> clientIdsWithPayBeforeModeration = clientService.clientIdsWithPayBeforeModeration(
                StreamEx.of(campaigns)
                        .filter(CampaignServiceUtils::isWallet)
                        .map(GdiBaseCampaign::getClientId)
                        .collect(toSet())
        );

        Client client = clientService.getClient(ClientId.fromLong(clientInfo.getId()));

        return campaigns.stream()
                .filter(CampaignServiceUtils::isWallet)
                .map(w -> w
                        .withWalletCanPayBeforeModeration(clientIdsWithPayBeforeModeration.contains(w.getClientId())))
                .map(w -> toWallet(operator, clientInfo, clientAutoOverdraftInfo, w,
                        walletIdToCampaign.getOrDefault(w.getId(), emptyList()),
                        currentInstant, client));
    }

    /**
     * Из списка всех кампаний клиента, получить список кошельков этих кампании
     *
     * @param campaigns      список всех кампаний клиента (в т.ч. кошельки)
     * @param currentInstant текущий момент
     * @return List<GdWallet>  список кошельков.
     */
    public List<GdWallet> extractWalletsList(User operator, GdClientInfo clientInfo,
                                             ClientAutoOverdraftInfo clientAutoOverdraftInfo,
                                             List<GdiBaseCampaign> campaigns,
                                             Instant currentInstant) {
        return extractWalletsStream(operator, clientInfo, clientAutoOverdraftInfo, campaigns, currentInstant)
                .collect(toList());
    }

    /**
     * Из списка кампаний клиента найти кошельки этих кампаний.
     *
     * @param campaigns      список всех кампаний клиента (в т.ч и кошельки)
     * @param currentInstant текущий момент
     * @return мапа Id_кошелька : Кошелек
     */
    public Map<Long, GdWallet> extractWalletsMap(User operator, GdClientInfo clientInfo,
                                                 ClientAutoOverdraftInfo clientAutoOverdraftInfo,
                                                 Collection<GdiBaseCampaign> campaigns,
                                                 Instant currentInstant) {
        return extractWalletsStream(operator, clientInfo, clientAutoOverdraftInfo, campaigns, currentInstant)
                .collect(toMap(GdWallet::getId, Function.identity()));
    }

    /**
     * Сконвертировать внутреннее представление кампании с типом WALLET в описание кошелька, доступное пользователям
     *
     * @param clientAutoOverdraftInfo инфо клиента (свойства, касающиеся автоовердрафтов)
     * @param wallet                  внутреннее представление кампании с типом WALLET
     * @param campaignsUnderWallet    все кампании клиента, для которых эта кампания указана как кошелек
     */
    private GdWallet toWallet(User operator,
                              GdClientInfo clientInfo,
                              ClientAutoOverdraftInfo clientAutoOverdraftInfo,
                              GdiBaseCampaign wallet, Collection<GdiBaseCampaign> campaignsUnderWallet,
                              Instant currentInstant, Client client) {
        Money moneyOnCamps = Money.valueOf(BigDecimal.ZERO, wallet.getCurrencyCode());
        Money walletSumDebt = Money.valueOf(BigDecimal.ZERO, wallet.getCurrencyCode());
        for (GdiBaseCampaign campaign : campaignsUnderWallet) {
            if (wallet.getCurrencyCode() != campaign.getCurrencyCode()) {
                continue;
            }
            Money leftOnCampaign =
                    Money.valueOf(campaign.getSum().subtract(campaign.getSumSpent()), campaign.getCurrencyCode());
            if (leftOnCampaign.greaterThanZero()) {
                moneyOnCamps = moneyOnCamps.add(leftOnCampaign);
            } else {
                walletSumDebt = walletSumDebt.add(leftOnCampaign);
            }
        }
        Money moneyOnWallet = Money.valueOf(wallet.getSum(), wallet.getCurrencyCode()).add(walletSumDebt);

        ClientId clientId = ClientId.fromLong(client.getClientId());
        boolean isSocialAdvertising = featureService.isEnabledForClientId(clientId, FeatureName.SOCIAL_ADVERTISING);
        boolean socialPaymentCondition = !isSocialAdvertising || featureService.isEnabledForClientId(clientId,
                FeatureName.SOCIAL_ADVERTISING_PAYABLE);

        // Вычисляем добавку, которую привносит автоовердрафт, сохраним её отдельно на кошелёк,
        // чтобы можно было и отобразить текущий баланс, и определить корректные статусы кампаний
        // (которые крутятся вопреки отрицательному текущему балансу общего счёта)
        BigDecimal autoOverdraftAddition = AutoOverdraftUtils.calculateAutoOverdraftAddition(
                wallet.getCurrencyCode(), wallet.getSum(), walletSumDebt.bigDecimalValue(),
                clientAutoOverdraftInfo);

        Set<GdiWalletAction> walletActions = socialPaymentCondition ?
                campaignAccessService.getWalletActions(operator, wallet, campaignsUnderWallet) : emptySet();

        Boolean enabled = wallet.getWalletCanPayBeforeModeration() || !campaignsUnderWallet.isEmpty();
        Boolean isAgencyWallet = wallet.getAgencyId() != 0 && wallet.getAgencyUserId() != null;
        CurrencyCode currency = wallet.getCurrencyCode();
        BigDecimal gdWalletSum = moneyOnWallet.roundToCentDown().bigDecimalValue();
        BigDecimal gdWalletSumWithNds = addNds(gdWalletSum, clientInfo.getNds(), currency);

        List<GdiBaseCampaign> campaignsNotForbiddenToPay = filterList(campaignsUnderWallet,
                campaign -> !campaign.getArchived() && !campaign.getNoPay() && !socialPaymentCondition);

        // N.B. walletActions.contains(GdiWalletAction.PAY) не подходит,
        // т.к. там содержится результаты двухуровневой проверки:
        // 1. в GridCampaignAccessService и 2. в CampaignAccessHelper
        // Здесь нужна только 2.
        boolean allowPay = socialPaymentCondition && CampaignAccessHelper.checkAllowPay(operator, wallet,
                campaignsUnderWallet);

        // флажок, который говорит, что оплату можно будет сделать сразу после того, как какую-нибудь кампанию
        // промодерируют
        boolean needCampModerationToPay =
                socialPaymentCondition && !campaignsNotForbiddenToPay.isEmpty() && !allowPay;
        Boolean needCalcDaysLeft = ifNotNull(gridContextProvider.getGridContext(),
                t -> t.getFetchedFieldsReslover().getWallet().getPaidDaysLeft());
        Integer paidDaysLeft = Boolean.TRUE.equals(needCalcDaysLeft) ?
                gridCampaignService.getPaidDaysLeft(gdWalletSum, campaignsUnderWallet) : null;

        boolean showOrderId = !GridCampaignAccessService.isCurrencyConverted(wallet)
                && enabled
                && hasOneOfRoles(operator, SUPER, SUPERREADER, SUPPORT, LIMITED_SUPPORT);

        GdWalletCashBackBonus bonus = null;
        if (featureService.isEnabledForClientId(ClientId.fromLong(clientInfo.getId()),
                FeatureName.SHOW_CASHBACK_BONUS)) {
            // Отдаем значения кэшбека, ожидающего кешбэка и начисленного за последний месяц кешбэка
            // Если правишь здесь, то нужно поправить аналогичный код для старого интерфейса в
            // https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Wallet.pm?rev=7439568#L122
            var details = cashbackClientsService.getPreviousMonthDetails(clientId);
            var lastMonthBonus = sumValues(details, CashbackProgramDetails::getReward);
            var lastMonthBonusWithoutNds = sumValues(details, CashbackProgramDetails::getRewardWithoutNds);
            bonus = new GdWalletCashBackBonus()
                    .withSum(subtractNds(clientInfo.getCashBackBonus(), clientInfo.getNds(), currency))
                    .withSumWithNds(clientInfo.getCashBackBonus())
                    .withAwaitingSum(subtractNds(clientInfo.getCashBackAwaitingBonus(), clientInfo.getNds(), currency))
                    .withAwaitingSumWithNds(clientInfo.getCashBackAwaitingBonus())
                    .withLastMonthSum(lastMonthBonusWithoutNds)
                    .withLastMonthSumWithNds(lastMonthBonus);
        }

        return new GdWallet()
                .withId(wallet.getId())
                .withCurrency(wallet.getCurrencyCode())
                .withSum(gdWalletSum)
                .withSumWithNds(gdWalletSumWithNds)
                .withBonus(bonus)
                .withAutoOverdraftAddition(Money.valueOf(autoOverdraftAddition, wallet.getCurrencyCode())
                        .roundToCentDown().bigDecimalValue())
                .withSumOnCampaigns(moneyOnCamps.roundToCentDown().bigDecimalValue())
                .withBudget(getWalletBudget(wallet))
                .withActions(mapSet(walletActions, GdWalletAction::fromSource))
                .withPromocodeRestrictedDomain(wallet.getPromocodeRestrictedDomain())
                .withIsAgencyWallet(isAgencyWallet)
                .withStatus(
                        getWalletStatus(wallet, enabled, needCampModerationToPay, moneyOnWallet.bigDecimalValue(),
                                currentInstant, client))
                .withLastPayDate(wallet.getLastPayDate())
                .withOrderId(showOrderId ? wallet.getOrderId() : null)
                .withPaidDaysLeft(paidDaysLeft);
    }

    private static BigDecimal addNds(BigDecimal value, @Nullable BigDecimal nds, CurrencyCode currency) {
        var percentNds = Objects.nonNull(nds) ? Percent.fromPercent(nds) : null;
        var money = Money.valueOf(value, currency);
        return Objects.isNull(percentNds) ? money.bigDecimalValue() : money.addNds(percentNds).bigDecimalValue();
    }

    private static BigDecimal subtractNds(BigDecimal sum, @Nullable BigDecimal nds, CurrencyCode currency) {
        var percentNds = Objects.nonNull(nds) ? Percent.fromPercent(nds) : null;
        var money = Money.valueOf(sum, currency);
        return Objects.isNull(percentNds) ? money.bigDecimalValue() : money.subtractNds(percentNds).bigDecimalValue();
    }

    private GdWalletBudget getWalletBudget(GdiBaseCampaign wallet) {
        GdWalletBudgetType budgetType =
                wallet.getDayBudget() != null && BigDecimal.ZERO.compareTo(wallet.getDayBudget()) < 0 ?
                        GdWalletBudgetType.DAY_BUDGET : GdWalletBudgetType.NO_BUDGET;
        GdWalletBudget budget = new GdWalletBudget()
                .withType(budgetType);
        if (budgetType != GdWalletBudgetType.NO_BUDGET) {
            budget
                    .withSum(wallet.getDayBudget())
                    .withShowMode(
                            wallet.getDayBudgetShowMode() == GdiDayBudgetShowMode.DEFAULT_ ?
                                    GdWalletBudgetShowMode.DEFAULT :
                                    GdWalletBudgetShowMode.STRETCHED);
        }
        return budget;
    }

    private GdWalletStatus getWalletStatus(GdiBaseCampaign wallet, Boolean enabled,
                                           boolean needCampModerationToPay,
                                           BigDecimal sumTotal, Instant currentInstant, Client client) {
        return new GdWalletStatus()
                .withEnabled(enabled)
                .withMoneyBlocked(wallet.getMoneyBlocked())
                .withNeedsNewPayment(
                        gridCampaignAggregationFieldsService.isCampaignNeedsNewPayment(wallet, sumTotal))
                .withWaitingForPayment(gridCampaignAggregationFieldsService
                        .isCampaignWaitingForPayment(wallet, wallet.getStatusModerate()))
                .withShowNds(getWalletShowNds(wallet.getCurrencyCode(), sumTotal, client))
                .withNeedCampModerationToPay(needCampModerationToPay)
                .withDisabledButHaveMoney(!enabled && NumberUtils.greaterThanZero(sumTotal))
                .withBudgetLimitationStopTime(getWalletBudgetStopTime(wallet, currentInstant));
    }

    private boolean getWalletShowNds(CurrencyCode walletCurrency, BigDecimal sumTotal, Client client) {
        if (NumberUtils.isZero(sumTotal)) {
            return false;
        }
        if (DONT_SHOW_WALLET_NDS_IN_INTERFACE.contains(walletCurrency)) {
            return false;
        }
        if (CurrencyCode.KZT == walletCurrency) {
            boolean isKazakhstanRegion = Long.valueOf(KAZAKHSTAN_REGION_ID).equals(client.getCountryRegionId());
            boolean isUsingQuasiCurrency = isTrue(client.getUsesQuasiCurrency());
            return !(isUsingQuasiCurrency && isKazakhstanRegion);
        }
        return true;
    }

    private LocalDateTime getWalletBudgetStopTime(GdiBaseCampaign wallet, Instant currentInstant) {
        LocalDate today = currentInstant.atZone(MSK).toLocalDate();

        boolean isStopped = wallet.getDayBudget() != null &&
                wallet.getDayBudget().compareTo(BigDecimal.ZERO) > 0 &&
                wallet.getDayBudgetStopTime() != null &&
                today.compareTo(wallet.getDayBudgetStopTime().toLocalDate()) == 0;
        return isStopped ? wallet.getDayBudgetStopTime() : null;
    }

    public CompletableFuture<Long> getLastChangedVcardId(Long campaignId) {
        return campaignsLastChangedVcardIdDataLoader.get().load(campaignId);
    }

    public CompletableFuture<Integer> getAdsCount(Long campaignId) {
        return campaignAdsCountDataLoader.get().load(campaignId);
    }

    public CompletableFuture<Long> getGroupsCount(Long campaignId) {
        return campaignsGroupsCountDataLoader.get().load(campaignId);
    }

    public CompletableFuture<GdPayForConversionInfo> getPayForConversionStatInfo(GridGraphQLContext context,
                                                                                 Long campaignId,
                                                                                 GdCampaignFlatStrategy strategy) {
        var clientId = ClientId.fromLong(context.getQueriedClient().getId());
        if (!featureService.isEnabledForClientId(clientId, FeatureName.CPA_PAY_FOR_CONVERSION_WARNING_ENABLED) ||
                !strategyExtractorFacade.isPayForConversionEnabled(strategy)) {
            return CompletableFuture.completedFuture(toGdPayForConversionInfo(null, null));
        }
        return campaignPayForConversionInfoDataLoader.get().load(campaignId, strategy);
    }

    public CompletableFuture<GdCampaignVcard> getCommonVcard(GdTextCampaign campaign) {
        Vcard contactInfo = campaign.getContactInfo();
        Long campaignId = campaign.getId();
        return getCommonVcard(contactInfo, campaignId, campaign.getHasNotArchivedAds());
    }

    public CompletableFuture<GdCampaignVcard> getCommonVcard(GdDynamicCampaign campaign) {
        Vcard contactInfo = campaign.getContactInfo();
        Long campaignId = campaign.getId();
        return getCommonVcard(contactInfo, campaignId, campaign.getHasNotArchivedAds());
    }

    private CompletableFuture<GdCampaignVcard> getCommonVcard(
            Vcard contactInfo,
            Long campaignId,
            Boolean hasNotArchivedVcard) {
        return vcardDataLoader.get().load(campaignId)
                .thenApply(vcard -> getVcardFromContactInfoAndVcard(vcard, contactInfo, hasNotArchivedVcard))
                .thenApply(VcardDataConverter::toGdCampaignVcard)
                .thenApply(vcard -> ifNotNull(vcard, v -> v.withCampaignId(campaignId)));
    }

    @Nullable
    private Vcard getVcardFromContactInfoAndVcard(@Nullable Vcard vcard, @Nullable Vcard contactInfo, Boolean
            hasAds) {
        //если у кампании нет баннеров, то отдаем контактную визитку
        if (!hasAds) {
            // DIRECT-115428 Заглушка для старых значений contactInfo без phone, country_code или city_code
            if (contactInfo != null) {
                if (contactInfo.getPhone().getCountryCode() == null) {
                    contactInfo.getPhone().setCountryCode("");
                }
                if (contactInfo.getPhone().getCityCode() == null) {
                    contactInfo.getPhone().setCityCode("");
                }
                if (contactInfo.getPhone().getPhoneNumber() == null) {
                    contactInfo.getPhone().setPhoneNumber("");
                }
            }
            return getVcardWithSystemFields(contactInfo);
        }
        return vcard;
    }

    @Nullable
    private Vcard getVcardWithSystemFields(@Nullable Vcard vcard) {
        if (vcard == null) {
            return null;
        }
        vcardHelper.fillVcardsWithRegionIds(List.of(vcard));
        vcardHelper.fillVcardsWithGeocoderData(List.of(vcard));
        return vcard;
    }

    public CompletableFuture<Boolean> getHasRunningUnmoderatedAds(Long campaignId) {
        return campaignsHasRunningUnmoderatedBannersDataLoader.get().load(campaignId);
    }

    public Set<Long> getTouchCampaignIds(int shard, ClientId clientId) {
        return campaignRepository.getTouchCampaignIds(shard, clientId);
    }

    public CompletableFuture<GdConversionStrategyLearningStatusData> getStrategyLearningStatus(
            GdCampaign campaign,
            List<GdMeaningfulGoal> meaningfulGoals) {
        Set<Long> meaningfulGoalIds = listToSet(meaningfulGoals, GdMeaningfulGoal::getGoalId);
        return conversionStrategyLearningStatusDataLoader.get().load(campaign, meaningfulGoalIds);
    }

    public CompletableFuture<Boolean> getCampaignHrefIsTurbo(GdCampaignAdditionalData additionalData) {
        if (additionalData.getHref() == null) {
            return null;
        }
        if (StringUtils.isBlank(additionalData.getHref())) {
            return CompletableFuture.completedFuture(false);
        }
        return campaignHrefIsTurboDataLoader.get().load(additionalData.getHref())
                .thenApply(isTurbo -> nvl(isTurbo, false));
    }

    public CompletableFuture<GdCampaignAgencyInfo> getGdCampaignAgencyInfo(GdCampWithAgencyInfo campaign) {
        if (!isValidId(campaign.getAgencyUserId())) {
            return null;
        }
        return campaignAgencyInfoByAgencyUidDataLoader.get().load(campaign.getAgencyUserId());
    }

    public CompletableFuture<GdCampaignManagerInfo> getGdCampaignManagerInfo(GdCampWithManagerInfo campaign) {
        if (!isValidId(campaign.getManagerUserId())) {
            return null;
        }
        return campaignManagerInfoByManagerUidDataLoader.get().load(campaign.getManagerUserId());
    }

    public CompletableFuture<GdBrandSurveyStatus> getGdBrandSurveyStatus(GdBrandSurveyStatusCampaign campaign) {
        if (campaign.getBrandSurveyId() == null) {
            return null;
        }
        return campaignBrandSurveyStatusDataLoader.get().load(
                new GetBrandSurveyStatusKey(campaign.getId(), campaign.getBrandSurveyId())
        );
    }

    public GdCpmBudgetLimitPayload getBudgetLimits(int shard, CurrencyCode currencyCode,
                                                   GdCpmCampaignDayBudgetLimitsRequest request) {

        List<Long> campaignIdsWithoutRestart = StreamEx.of(request.getItems())
                .remove(GdCpmCampaignDayBudgetLimitsRequestItem::getIsRestarting)
                .map(GdCpmCampaignDayBudgetLimitsRequestItem::getCampaignId)
                .nonNull()
                .toList();
        Map<Long, ? extends BaseCampaign> typedCampaignsMap = campaignTypedRepository.getTypedCampaignsMap(shard,
                campaignIdsWithoutRestart);

        Map<Long, CpmCampaignWithCustomStrategy> cpmCampaignWithCustomStrategyById =
                StreamEx.of(typedCampaignsMap.values())
                        .select(CpmCampaignWithCustomStrategy.class)
                        .mapToEntry(CpmCampaignWithCustomStrategy::getId, Function.identity())
                        .toMap();

        campaignValidationService.validateCampaignBudgetLimitRequest(request, cpmCampaignWithCustomStrategyById);

        List<CpmCampaignWithCustomStrategy> cpmCampaignWithCustomStrategy = filterAndMapList(
                campaignIdsWithoutRestart,
                cpmCampaignWithCustomStrategyById::containsKey, cpmCampaignWithCustomStrategyById::get);

        List<LocalDate> localDates = StreamEx.of(request.getItems())
                .mapToEntry(GdCpmCampaignDayBudgetLimitsRequestItem::getCampaignId,
                        GdCpmCampaignDayBudgetLimitsRequestItem::getFinishDate)
                .nonNullKeys()
                .filterKeys(cpmCampaignWithCustomStrategyById::containsKey)
                .values()
                .toList();

        Map<Long, BigDecimal> minimalBudgetForNotRestartingCampaigns =
                campaignStrategyService.calculateMinimalAvailableBudgetForCpmNotRestartingStrategyWithCustomPeriod(
                        cpmCampaignWithCustomStrategy, localDates, currencyCode);

        List<GdCpmBudgetLimit> cpmBudgetLimits = StreamEx.of(request.getItems())
                .mapToEntry(item ->
                                minimalBudgetForNotRestartingCampaigns.getOrDefault(item.getCampaignId(),
                                        calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                                                item.getStartDate(), item.getFinishDate(), currencyCode)),
                        item ->
                                calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                                        item.getStartDate(), item.getFinishDate(), currencyCode))
                .mapKeyValue((min, max) -> new GdCpmBudgetLimit()
                        .withMinimalBudget(min)
                        .withMaximumBudget(max))
                .toList();

        return new GdCpmBudgetLimitPayload().withLimits(cpmBudgetLimits);
    }

    public GdCpmBudgetLimit getCurrentCpmCampaignBudgetLimits(GdCpmBannerCampaign cpmBannerCampaign) {
        if (STRATEGY_TYPES_WITH_CUSTOM_PERIOD_FOR_CALCULATING_BUDGET_LIMITS.contains(
                cpmBannerCampaign.getStrategy().getType())) {
            GdCampaignStrategyAvgCpm strategy = (GdCampaignStrategyAvgCpm) cpmBannerCampaign.getStrategy();

            BigDecimal minimalBudget = calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                    strategy.getBudget().getStart(), strategy.getBudget().getFinish(),
                    cpmBannerCampaign.getCurrency());

            BigDecimal maximumBudget = calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                    strategy.getBudget().getStart(), strategy.getBudget().getFinish(),
                    cpmBannerCampaign.getCurrency());

            return new GdCpmBudgetLimit()
                    .withMinimalBudget(minimalBudget)
                    .withMaximumBudget(maximumBudget);

        } else if (STRATEGY_TYPES_WITH_WEEK_PERIOD_FOR_CALCULATING_BUDGET_LIMITS.contains(
                cpmBannerCampaign.getStrategy().getType())) {
            LocalDate now = LocalDate.now();
            BigDecimal minimalBudget = calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                    now, now.plusWeeks(1), cpmBannerCampaign.getCurrency());

            BigDecimal maximumBudget = calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                    now, now.plusWeeks(1), cpmBannerCampaign.getCurrency());

            return new GdCpmBudgetLimit()
                    .withMinimalBudget(minimalBudget)
                    .withMaximumBudget(maximumBudget);
        }

        return null;
    }

    /**
     * Принимает на вход campaignIds. Если у всех этих кампаний расписание показа, состоящее из timeTarget и
     * timeZoneId,
     * совпадает, то оно и возвращается. Если не совпадает - возвращается дефолтное.
     *
     * @param clientId    идентификатор клиента
     * @param campaignIds ids кампаний
     * @return временной таргетинг
     */
    public GdTimeTarget getCommonTimeTarget(ClientId clientId, Set<Long> campaignIds) {
        var shard = shardHelper.getShardByClientId(clientId);
        GdTimeTarget defaultTimeTarget = defaultGdTimeTarget();

        var clientCampaignIds = campaignTypedRepository.getClientCampaignIds(shard, clientId, campaignIds);
        Map<Long, ? extends BaseCampaign> typedCampaignsMap =
                campaignTypedRepository.getTypedCampaignsMap(shard, clientCampaignIds);

        if (typedCampaignsMap.isEmpty()) {
            return defaultTimeTarget;
        }

        List<CommonCampaign> campaigns = StreamEx.of(typedCampaignsMap.values())
                .select(CommonCampaign.class)
                .toList();

        var differentTimeZoneIdsCount = StreamEx.of(campaigns)
                .map(CommonCampaign::getTimeZoneId)
                .distinct()
                .count();

        if (differentTimeZoneIdsCount > 1) {
            return defaultTimeTarget;
        }

        var differentTimeTargetsCount = StreamEx.of(campaigns)
                .map(CommonCampaign::getTimeTarget)
                .distinct()
                .count();

        if (differentTimeTargetsCount > 1) {
            return defaultTimeTarget;
        }

        TimeTarget commonTimeTarget = campaigns.get(0).getTimeTarget();
        Long commonTimeZoneId = campaigns.get(0).getTimeZoneId();

        return toGdTimeTarget(commonTimeTarget)
                .withIdTimeZone(commonTimeZoneId);
    }

    public List<GdTag> getTags(GdClientInfo clientInfo, GdCampaign campaign) {
        long campaignId = campaign.getId();
        List<CampaignTag> tags = tagRepository.getCampaignTagsWithUseCount(clientInfo.getShard(),
                singleton(campaignId));
        return mapList(tags, TagConverter::toGdTag);
    }

    /**
     * Метод возвращает источник, в котором была создана последняя РМП кампания (DIRECT, UAC, API)
     *
     * @param clientInfo информация о клиенте
     * @return источник последней РМП кампании
     */
    public GdCampaignSource getLastRmpCampaignSource(GdClientInfo clientInfo) {
        var clientId = clientInfo.getId();
        var shard = clientInfo.getShard();
        var campaignSource = campaignRepository.getLastCampaignSource(shard, clientId, CampaignsType.mobile_content);
        return GdCampaignSource.valueOf(GdiCampaignSource.fromSource(campaignSource.orElse(CampaignsSource.uac)).name());
    }

    public List<GdCpmCampaignBrandSurveyCheckResponseItem> getBrandSurveyConditions(
            ClientId clientId,
            Long operatorUid,
            GdCpmCampaignBrandSurveyCheckRequest request
    ) {
        var client = clientService.getClient(clientId);
        boolean isBrandLiftHiddenEnabled = featureService.isEnabled(operatorUid, FeatureName.BRAND_LIFT_HIDDEN);
        boolean totalBudgetWarnEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.LOW_TOTAL_BUDGET_BRANDLIFT);
        LocalDate brandSurveyBudgetDatePropVal = ppcPropertiesSupport.get(PpcPropertyNames.BRAND_SURVEY_BUDGET_DATE).get();
        var forecastThreshold = brandLiftConfig.getBranch("thresholds").getLong("target");
        List<Campaign> campaigns = campaignRepository.getCampaigns(shardHelper.getShardByClientId(clientId),
                mapList(request.getItems(), GdCpmCampaignBrandSurveyCheckData::getCid));
        var brandSurveyIds = mapAndFilterList(request.getItems(),
                GdCpmCampaignBrandSurveyCheckData::getBrandSurveyId, Objects::nonNull);
        Map<String, List<Campaign>> campBySurveyId = brandSurveyIds.isEmpty() ? emptyMap() :
                campaignRepository.getCampaignsForBrandSurveys(shardHelper.getShardByClientId(clientId), clientId, brandSurveyIds);
        Map<Long, TargetEstimation> targetEstimationMap =
                listToMap(
                        targetEstimationsService.getTargetEstimations(campaigns),
                        TargetEstimation::getCampaignId);
        return request.getItems().stream()
                .map(item -> {
                            var campaignStrategy = toCampaignStrategy(item.getBiddingStategy());
                            var brandSurveyStatus = brandSurveyConditionsService.getBrandSurveyStatus(
                                    campaignStrategy,
                                    item.getStartDate(),
                                    item.getEndDate(),
                                    item.getDayBudget(),
                                    client.getWorkCurrency(),
                                    totalBudgetWarnEnabled,
                                    brandSurveyBudgetDatePropVal,
                                    isBrandLiftHiddenEnabled,
                                    getPrevCampsForBrandlift(item.getBrandSurveyId(), item.getCid(), campBySurveyId));

                            if(Optional.ofNullable(targetEstimationMap.get(item.getCid()))
                                    .map(TargetEstimation::getTargetForecast).orElse(0L) < forecastThreshold) {
                                brandSurveyStatus.getBrandSurveyStopReasonsDaily().add(BrandSurveyStopReason.LOW_REACH);
                            }
                            return new GdCpmCampaignBrandSurveyCheckResponseItem()
                                    .withCid(nvl(item.getCid(), 0L))
                                    .withStatus(convertBrandSurveyStatus(brandSurveyStatus));
                        }
                ).collect(toList());
    }

    //Возвращает предыдущие РК для этого брендлифта
    private List<Campaign> getPrevCampsForBrandlift(String brandSurveyId, Long cid, Map<String, List<Campaign>> campBySurveyId) {
        if (brandSurveyId == null) {
            return emptyList();
        }
        List<Campaign> camps = campBySurveyId.getOrDefault(brandSurveyId, emptyList());
        return filterList(camps, it -> cid == null || !it.getId().equals(cid));
    }

    public GdMultipleCampaignsBrandSurveyCheckResponse getBrandSurveyConditions(
            ClientId clientId,
            Long operatorUid,
            GdMultipleCampaignsBrandSurveyCheckRequest request
    ) {
        boolean isBrandLiftHiddenEnabled = featureService.isEnabled(operatorUid, FeatureName.BRAND_LIFT_HIDDEN);
        boolean totalBudgetWarnEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.LOW_TOTAL_BUDGET_BRANDLIFT);

        var client = clientService.getClient(clientId);
        var shard = shardHelper.getShardByClientId(clientId);
        var forecastThreshold = brandLiftConfig.getBranch("thresholds").getLong("target");

        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, request.getCampaignIds());
        LocalDateTime createTime = campaigns.stream()
                .map(Campaign::getCreateTime)
                .filter(notNull())
                .min(LocalDateTime::compareTo)
                .orElse(null);
        LocalDate brandSurveyBudgetDatePropVal = ppcPropertiesSupport.get(PpcPropertyNames.BRAND_SURVEY_BUDGET_DATE).get();
        var brandSurveyBudgetThreshold = BigDecimal.valueOf(brandSurveyConditionsService.brandSurveyBudgetThreshold(
                client.getWorkCurrency().getCurrency(), createTime, brandSurveyBudgetDatePropVal));
        var brandSurveyBudgetThresholdDaily = BigDecimal.valueOf(brandSurveyConditionsService.brandSurveyBudgetThresholdDaily(
                client.getWorkCurrency().getCurrency(), createTime, brandSurveyBudgetDatePropVal));
        Map<Long, PricePackage> pricePackageByCampaigns = pricePackageService
                .getPricePackageByCampaigns(shard, campaigns);
        Map<Long, CampaignWithBrandLift> campaignWithBrandLiftMap =
                listToMap((List<CampaignWithBrandLift>) campaignTypedRepository.getTypedCampaigns(shard,
                mapList(campaigns, Campaign::getId)), BaseCampaign::getId);
        Map<Boolean, List<Campaign>> campaignsPartitionedByAllowedBL = campaigns.stream()
                .collect(Collectors.partitioningBy(n -> Optional.ofNullable(pricePackageByCampaigns.get(n.getId()))
                        .map(PricePackageBase::getCampaignOptions)
                        .map(campOptions -> !campOptions.getAllowBrandLift())
                        .orElse(Boolean.FALSE) ||
                        !Optional.ofNullable(campaignWithBrandLiftMap.get(n.getId()))
                                .map(CampaignWithShows::getShows).orElse(0L).equals(0L)));
        List<Campaign> allowedCampaigns = campaignsPartitionedByAllowedBL.get(Boolean.FALSE);
        List<Campaign> disallowedCampaigns = campaignsPartitionedByAllowedBL.get(Boolean.TRUE);

        Map<Long, TargetEstimation> targetEstimations = listToMap(targetEstimationsService
                .getTargetEstimations(allowedCampaigns), TargetEstimation::getCampaignId);
        var forecast = allowedCampaigns.stream()
                .map(Campaign::getId)
                .map(targetEstimations::get)
                .map(targetEstimation -> Optional.ofNullable(targetEstimation)
                        .map(TargetEstimation::getTargetForecast)
                        .orElse(0L))
                .reduce(Long::sum)
                .orElse(0L);

        var bs = allowedCampaigns.stream()
                .map(campaign -> {
                    BrandSurveyStatus brandSurveyStatus;
                    if (pricePackageByCampaigns.containsKey(campaign.getId())) {
                        brandSurveyStatus = brandSurveyConditionsService.getCpmPriceBrandSurveyStatus(
                                campaign.getStartTime(),
                                campaign.getFinishTime(),
                                pricePackageByCampaigns.get(campaign.getId()),
                                campaign.getDayBudget(),
                                campaign.getStrategy().getStrategyData().getAutoProlongation().equals(1L) ?
                                        Boolean.TRUE : Boolean.FALSE,
                                totalBudgetWarnEnabled,
                                brandSurveyBudgetDatePropVal,
                                isBrandLiftHiddenEnabled,
                                emptyList());//массовая валидация на гридах для нового БЛ учитываем только текущую кампанию
                    } else {
                        brandSurveyStatus = brandSurveyConditionsService.getBrandSurveyStatus(
                                campaign.getStrategy(),
                                campaign.getStartTime(),
                                campaign.getFinishTime(),
                                campaign.getDayBudget(),
                                client.getWorkCurrency(),
                                totalBudgetWarnEnabled,
                                brandSurveyBudgetDatePropVal,
                                isBrandLiftHiddenEnabled,
                                emptyList());//массовая валидация на гридах для нового БЛ учитываем только текущую кампанию
                    }
                    return brandSurveyStatus;
                }).reduce((bs1, bs2) -> new BrandSurveyStatus()
                        .withSumSpentByDay(bs1.getSumSpentByDay().add(bs2.getSumSpentByDay()))
                        .withSumSpentByTotalPeriod(bs1.getSumSpentByTotalPeriod().add(bs2.getSumSpentByTotalPeriod())))
                .orElse(new BrandSurveyStatus()
                        .withSumSpentByDay(BigDecimal.ZERO)
                        .withSumSpentByTotalPeriod(BigDecimal.ZERO));
        var lowBudget = bs.getSumSpentByDay().compareTo(brandSurveyBudgetThresholdDaily) < 0 ||
                bs.getSumSpentByTotalPeriod().compareTo(brandSurveyBudgetThreshold) < 0;

        return new GdMultipleCampaignsBrandSurveyCheckResponse()
                .withDayBudgetThreshold(brandSurveyBudgetThresholdDaily)
                .withTotalBudgetThreshold(brandSurveyBudgetThreshold)
                .withCampaignsWithDisabledBL(disallowedCampaigns.stream().map(Campaign::getId).collect(toList()))
                .withForecast(forecast)
                .withLowBudget(lowBudget)
                .withLowForecast(forecast < forecastThreshold);
    }

    public List<GdCpmCampaignBrandSurveyCheckResponseItem> getBrandSurveyConditions(
            ClientId clientId, GdCpmPriceCampaignBrandSurveyCheckRequest request
    ) {
        var pricePackageIds = request.getItems().stream()
                .map(GdCpmPriceCampaignBrandSurveyCheckData::getPricePackageId).collect(toList());
        var pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);
        boolean isBrandLiftHiddenEnabled = featureService.isEnabled(
                gridContextProvider.getGridContext().getOperator().getUid(), FeatureName.BRAND_LIFT_HIDDEN);
        boolean totalBudgetWarnEnabled = featureService.isEnabledForClientId(
                gridContextProvider.getGridContext().getSubjectUser().getClientId(),
                FeatureName.LOW_TOTAL_BUDGET_BRANDLIFT);
        LocalDate brandSurveyBudgetDatePropVal = ppcPropertiesSupport.get(PpcPropertyNames.BRAND_SURVEY_BUDGET_DATE).get();
        var brandSurveyIds = mapAndFilterList(request.getItems(),
                GdCpmPriceCampaignBrandSurveyCheckData::getBrandSurveyId, Objects::nonNull);
        Map<String, List<Campaign>> campBySurveyId = brandSurveyIds.isEmpty() ? emptyMap() :
                campaignRepository.getCampaignsForBrandSurveys(shardHelper.getShardByClientId(clientId), clientId, brandSurveyIds);

        return request.getItems().stream()
                .map(item -> {
                            var brandSurveyStatus = brandSurveyConditionsService.getCpmPriceBrandSurveyStatus(
                                    item.getStartDate(),
                                    item.getEndDate(),
                                    pricePackages.get(item.getPricePackageId()),
                                    item.getBudget(),
                                    item.getAutoProlongation(),
                                    totalBudgetWarnEnabled,
                                    brandSurveyBudgetDatePropVal,
                                    isBrandLiftHiddenEnabled,
                                    getPrevCampsForBrandlift(item.getBrandSurveyId(), item.getCid(), campBySurveyId));

                            return new GdCpmCampaignBrandSurveyCheckResponseItem()
                                    .withCid(item.getCid())
                                    .withStatus(convertBrandSurveyStatus(brandSurveyStatus));
                        }
                ).collect(toList());
    }

    private Map<Long, PricePackage> pricePackageMapByCampaignId(int shard, List<GdiCampaign> filteredCampaigns) {
        Set<Long> cpmPriceCampaignIds = StreamEx.of(filteredCampaigns)
                .filter(gdiCampaign -> gdiCampaign.getType().equals(CampaignType.CPM_PRICE))
                .map(GdiCampaign::getId)
                .toSet();
        if (cpmPriceCampaignIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, CpmPriceCampaign> cpmPriceCampaigns =
                (Map<Long, CpmPriceCampaign>) campaignTypedRepository.getTypedCampaignsMap(shard,
                        cpmPriceCampaignIds);
        var pricePackageIds = mapList(cpmPriceCampaigns.values(), CpmPriceCampaign::getPricePackageId);
        var pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);
        checkEachCampaignHasPricePackage(cpmPriceCampaigns.values(), pricePackages);
        return EntryStream.of(cpmPriceCampaigns)
                .mapValues(campaign -> pricePackages.get(campaign.getPricePackageId()))
                .toMap();
    }

    /**
     * Такого быть не должно, добавили чтобы разбираться с проблемами как в DIRECT-129598 и DIRECT-131012
     */
    private static void checkEachCampaignHasPricePackage(Collection<CpmPriceCampaign> cpmPriceCampaigns,
                                                         Map<Long, PricePackage> pricePackages) {
        cpmPriceCampaigns.forEach(c -> {
            if (pricePackages.get(c.getPricePackageId()) == null) {
                throw new IllegalStateException(String.format("Price package doesn't exist for cid %d statusEmpty %b " +
                                "pricePackageId %d. This should not happen, if you see this message then database is " +
                                "broken, to fix it execute `update campaigns set statusEmpty='Yes' where " +
                                "cid = %d` (there are no reasons to have cpmPrice campaigns refering nonexistent" +
                                " packages)",
                        c.getId(), c.getStatusEmpty(), c.getPricePackageId(), c.getId()));
            }
        });
    }

    public boolean getHasBannersStoppedByMonitoring(int shard, Long campaignId) {
        var bannersStoppedByMonitoring =
                bannerCommonRepository.getBannersStoppedByMonitoringByCampaignId(shard, campaignId);
        return !bannersStoppedByMonitoring.isEmpty();
    }

    @Nullable
    public BigDecimal getSpentToday(GdCampaign campaign) {
        Long exportId = campaign.getExportId();
        GdCampaignBudget budget = campaign.getStrategy().getBudget();
        if (budget != null && budget.getPeriod() == GdCampaignBudgetPeriod.DAY
                && NumberUtils.greaterThanZero(budget.getSum()) && campaign.getExportId() != 0) {
            Money spent = orderStatService.getOrdersSpentToday(List.of(exportId), false).get(exportId);
            return ifNotNull(spent, Money::bigDecimalValue);
        }
        return null;
    }

    public Optional<Long> getFirstNotArchivedTextCampaignId(int shard, ClientId clientId, long operatorUid) {
        Set<Long> campaignIdsToCheck = campaignRepository.getNonArchivedNonDeletedCampaignIdsByClientIdsAndTypes(
                shard, Set.of(clientId), Set.of(CampaignType.TEXT));
        Map<Long, AggregatedStatusCampaignData> campStatusesById = aggregatedStatusesViewService
                .getCampaignStatusesByIds(shard, campaignIdsToCheck);

        Set<Long> cidsFilteredByAggrStatus = StreamEx.of(campaignIdsToCheck)
                .filter(cid -> campStatusesById.get(cid) == null ||
                        campStatusesById.get(cid).getStatus().filter(GdSelfStatusEnum.ARCHIVED::equals).isEmpty())
                .toSet();

        CampaignSubObjectAccessValidator validator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, cidsFilteredByAggrStatus)
                .createValidator(CampaignAccessType.READ);

        return cidsFilteredByAggrStatus.stream()
                .map(validator)
                .filter(vr -> !vr.hasAnyErrors())
                .map(ValidationResult::getValue)
                .min(Comparator.naturalOrder());
    }

    /**
     * Возвращает данные о приложении для саджеста по клиенту. Ищет приложение среди последних созданных РМП-кампаний.
     *
     * @param clientInfo — гридовая информация о клиенте
     * @return — данные о приложении
     */
    public GdMobileContentSuggestInfo getLatestMobileCampaignContentInfo(GdClientInfo clientInfo) {
        var clientId = ClientId.fromLong(clientInfo.getId());
        var shard = clientInfo.getShard();
        var campaignIds = campaignRepository.getLatestCampaignIds(
                shard, clientId, Set.of(CampaignType.MOBILE_CONTENT), LATEST_CAMPAIGNS_LOOKUP_COUNT
        );
        if (campaignIds.isEmpty()) {
            return null;
        }
        var mobileAppIdsMap = campaignRepository.getCampaignMobileAppIds(shard, campaignIds);
        for (var cid : campaignIds) {
            if (mobileAppIdsMap.containsKey(cid)) {
                return mobileContentInfoRepository.getGdMobilecontentInfoForSuggest(shard,
                        mobileAppIdsMap.get(cid));
            }
        }

        return null;
    }

    public CompletableFuture<List<String>> getDomains(GdTextCampaign campaign) {
        String campaignHref = campaign.getAdditionalData() == null ? null : campaign.getAdditionalData().getHref();
        return campaignDomainsDataLoader.get().load(campaign.getId(), campaignHref);
    }

    public List<GdCampaignSourceAvailability> getAvailableCampaignSources(GdClient client) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        var representativesUids = rbacService.getClientRepresentativesUids(clientId);
        return calculateCampaignSourcesAvailability(enabledFeatures, client.getInfo(), representativesUids);
    }

    public List<GdCampaignMeasurer> getCampaignMeasurer(int shard, Long campaignId) {
        return StreamEx.of(measurersRepository.getMeasurersByCampaignIds(shard, List.of(campaignId)).values())
                .flatMap(it -> it.stream())
                .map(it -> toGdCampaignMeasurer(it))
                .toList();
    }

    public List<GdPiPage> getAllowedPages(int shard, Long campaignId) {
        var campaign = campaignTypedRepository.getTypedCampaignsMap(shard, List.of(campaignId)).get(campaignId);
        if (campaign != null && campaign instanceof CampaignWithAllowedPageIds) {
            List<Long> allowedPageIds = ((CampaignWithAllowedPageIds) campaign).getAllowedPageIds();
            if (allowedPageIds == null || allowedPageIds.isEmpty()) {
                return null;
            }
            var dbPages = Lists.newArrayList(placementDataService.getPlacementsByIds(allowedPageIds));
            // возможна ситуация, когда добавили числом pageid, а в нашу таблицу пейдж ещё не приехал
            Set<Long> dbPageIds = StreamEx.of(dbPages)
                    .map(GdPiPage::getId)
                    .toSet();
            dbPages.addAll(
                    StreamEx.of(allowedPageIds)
                            .filter(it -> !dbPageIds.contains(it))
                            .map(CampaignInfoService::missedGdPiPage)
                            .toList()
            );
            return dbPages;
        }
        return null;
    }

    /**
     * Создаёт заглушку для не найденного по id пейджу
     */
    private static GdPiPage missedGdPiPage(Long id) {
        return new GdPiPage()
                .withId(id)
                .withIsDeleted(false)
                .withIsYandexPage(false)
                .withIsTesting(false);
    }
}
