package ru.yandex.direct.grid.processing.service.campaign;

import java.time.Duration;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;

import ru.yandex.direct.abac.Attribute;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdCheckCopyCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.GdCheckCopyCampaignsPayload;
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsInterclient;
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsInterclientPayload;
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaignPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddTouchCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddUcCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignIdsList;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdTouchCampaignMutationPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcCampaignMutationPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignMetrikaCounters;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignMetrikaCountersPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBannerHrefParams;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBrandSurvey;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsDayBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsMinusKeywords;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsOrganization;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsPromotions;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsStartDate;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsTimeTargeting;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsWeeklyBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateTouchCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateUcCampaignInput;
import ru.yandex.direct.grid.processing.model.promoextension.GdUpdateCampaignsPromoExtension;
import ru.yandex.direct.grid.processing.service.attributes.AttributeResolverService;
import ru.yandex.direct.grid.processing.service.campaign.copy.CampaignCopyMutationService;
import ru.yandex.direct.grid.processing.service.campaign.touch.TouchCampaignMutationService;
import ru.yandex.direct.grid.processing.service.campaign.uc.UcCampaignMutationService;
import ru.yandex.direct.web.annotations.RequiredAttributes;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.db.PpcPropertyNames.DISABLE_OLD_UC;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class CampaignMutationGraphQlService {
    public static final String ADD_CAMPAIGNS = "addCampaigns";
    public static final String UPDATE_CAMPAIGNS = "updateCampaigns";
    public static final String DELETE_CAMPAIGNS = "deleteCampaigns";
    public static final String DELETE_BRAND_LIFT_FOR_CAMPAIGNS = "deleteBrandLiftForCampaigns";
    public static final String ADD_TOUCH_CAMPAIGN = "addTouchCampaign";
    public static final String UPDATE_TOUCH_CAMPAIGN = "updateTouchCampaign";
    public static final String ADD_UC_CAMPAIGN = "addUcCampaign";
    public static final String UPDATE_UC_CAMPAIGN = "updateUcCampaign";
    public static final String COPY_CAMPAIGNS = "copyCampaigns";
    public static final String CHECK_COPY_CAMPAIGNS = "checkCopyCampaigns";
    public static final String COPY_CAMPAIGNS_INTERCLIENT = "copyCampaignsInterclient";
    public static final String UPDATE_CAMPAIGN_METRIKA_COUNTERS = "updateCampaignMetrikaCounters";
    public static final String RESET_CAMPAIGN_FLIGHT_STATUS_APPROVE = "resetCampaignFlightStatusApprove";
    public static final String UPDATE_CAMPAIGNS_DAY_BUDJET = "updateCampaignsDayBudget";
    public static final String UPDATE_CAMPAIGNS_START_DATE = "updateCampaignsStartDate";
    public static final String UPDATE_CAMPAIGNS_TIME_TARGETING = "updateCampaignsTimeTargeting";
    public static final String UPDATE_CAMPAIGNS_MINUS_KEYWORDS = "updateCampaignsMinusKeywords";
    public static final String UPDATE_CAMPAIGNS_ORGANIZATION = "updateCampaignsOrganization";
    public static final String UPDATE_CAMPAIGNS_STRATEGY = "updateCampaignsStrategy";
    public static final String UPDATE_CAMPAIGNS_PROMOTIONS = "updateCampaignsPromotions";
    public static final String UPDATE_CAMPAIGNS_PROMO_EXTENSION = "updateCampaignsPromoExtension";
    public static final String UPDATE_CAMPAIGNS_WEEKLY_BUDGET = "updateCampaignsWeeklyBudget";
    public static final String UPDATE_CAMPAIGNS_BRAND_SURVEY = "updateCampaignsBrandSurvey";
    public static final String UPDATE_CAMPAIGNS_BANNER_HREF_PARAMS = "updateCampaignsBannerHrefParams";

    private final AttributeResolverService attributeResolverService;
    private final CampaignCopyMutationService campaignCopyMutationService;
    private final CampaignMutationService campaignMutationService;
    private final TouchCampaignMutationService touchCampaignMutationService;
    private final UcCampaignMutationService ucCampaignMutationService;

    private final PpcProperty<Boolean> oldUcDisabledProperty;

    public CampaignMutationGraphQlService(AttributeResolverService attributeResolverService,
                                          CampaignCopyMutationService campaignCopyMutationService,
                                          CampaignMutationService campaignMutationService,
                                          TouchCampaignMutationService touchCampaignMutationService,
                                          UcCampaignMutationService ucCampaignMutationService,
                                          PpcPropertiesSupport ppcPropertiesSupport) {
        this.attributeResolverService = attributeResolverService;
        this.campaignCopyMutationService = campaignCopyMutationService;
        this.campaignMutationService = campaignMutationService;
        this.touchCampaignMutationService = touchCampaignMutationService;
        this.ucCampaignMutationService = ucCampaignMutationService;

        this.oldUcDisabledProperty = ppcPropertiesSupport.get(DISABLE_OLD_UC, Duration.ofMinutes(5));
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = ADD_CAMPAIGNS)
    public GdAddCampaignPayload addCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddCampaigns input
    ) {
        return campaignMutationService.addCampaigns(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS)
    public GdUpdateCampaignPayload updateCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaigns input
    ) {
        return campaignMutationService.updateCampaigns(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = DELETE_CAMPAIGNS)
    public GdUpdateCampaignPayload deleteCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignIdsList input
    ) {
        return campaignMutationService.deleteCampaigns(context, input);
    }


    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = DELETE_BRAND_LIFT_FOR_CAMPAIGNS)
    public GdCampaignIdsList deleteBrandLiftForCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignIdsList input
    ) {
        return campaignMutationService.deleteBrandLiftForCampaigns(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = ADD_TOUCH_CAMPAIGN)
    public GdTouchCampaignMutationPayload addTouchCampaign(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddTouchCampaignInput input
    ) {
        return touchCampaignMutationService.addTouchCampaign(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_TOUCH_CAMPAIGN)
    public GdTouchCampaignMutationPayload updateTouchCampaign(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateTouchCampaignInput input
    ) {
        return touchCampaignMutationService.updateTouchCampaign(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = ADD_UC_CAMPAIGN)
    public GdUcCampaignMutationPayload addUcCampaign(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddUcCampaignInput input
    ) {
        checkState(!oldUcDisabledProperty.getOrDefault(false), "old uc disabled");

        return ucCampaignMutationService.addUcCampaign(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_UC_CAMPAIGN)
    public GdUcCampaignMutationPayload updateUcCampaign(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateUcCampaignInput input
    ) {
        return ucCampaignMutationService.updateUcCampaign(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = COPY_CAMPAIGNS)
    public GdCopyCampaignsPayload copyCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCopyCampaigns input
    ) {
        return campaignCopyMutationService.copyCampaigns(context, input);
    }

    @GraphQLNonNull
    @EnableLoggingOnValidationIssues
    @RequiredAttributes(Attribute.HAS_ACCESS_TO_COPY_PAGE)
    @GraphQLQuery(name = CHECK_COPY_CAMPAIGNS)
    public GdCheckCopyCampaignsPayload checkCopyCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCheckCopyCampaigns input
    ) {
        return campaignCopyMutationService.checkCopyCampaigns(context, input);
    }

    @GraphQLNonNull
    @EnableLoggingOnValidationIssues
    @RequiredAttributes(Attribute.HAS_ACCESS_TO_COPY_PAGE)
    @GraphQLMutation(name = COPY_CAMPAIGNS_INTERCLIENT)
    public GdCopyCampaignsInterclientPayload copyCampaignsInterclient(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCopyCampaignsInterclient input
    ) {
        return campaignCopyMutationService.copyCampaignsInterclient(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGN_METRIKA_COUNTERS)
    public GdUpdateCampaignMetrikaCountersPayload updateCampaignMetrikaCounters(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignMetrikaCounters input
    ) {
        boolean result = attributeResolverService
                .resolve(Attribute.CLIENT_HAS_FEATURE_FOR_UPDATE_CAMPAIGN_METRIKA_COUNTERS);
        checkState(result, "client does not have a feature to update campaign metrika counters");

        ClientId clientId = Objects.requireNonNull(context.getSubjectUser()).getClientId();
        return campaignMutationService.updateCampaignMetrikaCounters(clientId, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_DAY_BUDJET)
    public GdUpdateCampaignPayload updateCampaignsDayBudget(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsDayBudget input
    ) {
        return campaignMutationService.updateCampaignsDayBudget(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_WEEKLY_BUDGET)
    public GdUpdateCampaignPayload updateCampaignsWeeklyBudget(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsWeeklyBudget input
    ) {
        return campaignMutationService.updateCampaignsWeeklyBudget(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = RESET_CAMPAIGN_FLIGHT_STATUS_APPROVE)
    public GdUpdateCampaignPayload resetCampaignFlightStatusApprove(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignIdsList input
    ) {
        return campaignMutationService.resetCampaignFlightStatusApprove(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_START_DATE)
    public GdUpdateCampaignPayload updateCampaignsStartDate(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsStartDate input
    ) {
        return campaignMutationService.updateCampaignsStartDate(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_PROMO_EXTENSION)
    public GdUpdateCampaignPayload updateCampaignPromoExtension(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsPromoExtension input
    ) {
        return campaignMutationService.updateCampaignPromoExtensionChanges(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_TIME_TARGETING)
    public GdUpdateCampaignPayload updateCampaignsTimeTargeting(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsTimeTargeting input
    ) {
        return campaignMutationService.updateCampaignsTimeTargeting(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_MINUS_KEYWORDS)
    public GdUpdateCampaignPayload updateCampaignsMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsMinusKeywords input
    ) {
        return campaignMutationService.updateCampaignsMinusKeywords(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_ORGANIZATION)
    public GdUpdateCampaignPayload updateCampaignsOrganisation(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsOrganization input
    ) {
        return campaignMutationService.updateCampaignsOrganization(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_STRATEGY)
    public GdUpdateCampaignPayload updateCampaignStrategy(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsStrategy input
    ) {
        return campaignMutationService.updateCampaignsStrategy(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_PROMOTIONS)
    public GdUpdateCampaignPayload updateCampaignsPromotions(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsPromotions input
    ) {
        return campaignMutationService.updateCampaignsPromotions(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_BRAND_SURVEY)
    public GdUpdateCampaignPayload setBrandLiftForMultipleCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsAddBrandSurvey input
    ) {
        return campaignMutationService.addBrandSurvey(context, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_CAMPAIGNS_BANNER_HREF_PARAMS)
    public GdUpdateCampaignPayload updateBannerHrefParamsForMultipleCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCampaignsAddBannerHrefParams input
    ) {
        return campaignMutationService.updateBannerHrefParams(context, input);
    }

}
