package ru.yandex.direct.grid.processing.service.campaign;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.container.UpdateCampMetrikaCountersRequest;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMinusKeywords;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.SurveyStatus;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.campaign.service.CampaignBudgetReachService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.CopyCampaignService;
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsAddOperation;
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsUpdateOperation;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider;
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils;
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaignPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaignPayloadItem;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignIdsList;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdMeaningfulGoalRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignMetrikaCounters;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignMetrikaCountersPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignPayloadItem;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBannerHrefParams;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBrandSurvey;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsDayBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsMinusKeywords;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsMinusKeywordsAction;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsOrganization;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsPromotions;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsStartDate;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsTimeTargeting;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsWeeklyBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateMeaningfulGoals;
import ru.yandex.direct.grid.processing.model.common.GdMassUpdateAction;
import ru.yandex.direct.grid.processing.model.goal.GdGoal;
import ru.yandex.direct.grid.processing.model.promoextension.GdUpdateCampaignsPromoExtension;
import ru.yandex.direct.grid.processing.service.attributes.AttributeResolverService;
import ru.yandex.direct.grid.processing.service.campaign.converter.UpdateCampaignMutationConverter;
import ru.yandex.direct.grid.processing.service.campaign.converter.UpdateCampaignsStrategyConverter;
import ru.yandex.direct.grid.processing.service.campaign.converter.UpdateCampaignsWeeklyBudgetConverter;
import ru.yandex.direct.grid.processing.service.goal.GoalMutationService;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toGdUpdateCampaignMetrikaCountersPayloadWithOnlyValidationResult;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toModelChangesBannerHrefParams;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toModelChangesBrandSurvey;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toModelChangesCampaignsPromotions;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toModelChangesDayBudget;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toModelChangesOrganization;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toModelChangesStartDate;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toModelPromoExtensionChanges;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toUpdateCampMetrikaCountersRequest;
import static ru.yandex.direct.grid.processing.service.campaign.converter.AddCampaignMutationConverter.toCoreCampaigns;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UpdateCampaignMutationConverter.gdUpdateCampaignsToCoreModelChanges;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UpdateMeaningfulGoalsConverter.toCoreModelChanges;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;
import static ru.yandex.direct.validation.util.ValidationUtils.cloneValidationResultSubNodesWithIssues;

@ParametersAreNonnullByDefault
@Service
public class CampaignMutationService {
    private static final Applicability UPDATE_METRIKA_COUNTERS_APPLICABILITY = Applicability.PARTIAL;
    private static final Applicability UPDATE_PROMO_EXTENSION_APPLICABILITY = Applicability.FULL;
    private static final Applicability DEFAULT_APPLICABILITY = Applicability.PARTIAL;

    private final CampaignValidationService campaignValidationService;
    private final CampaignService campaignService;
    private final CampaignOperationService campaignOperationService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final MetrikaGoalsService metrikaGoalsService;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final AttributeResolverService attributeResolverService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final CampaignBudgetReachService campaignBudgetReachService;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final HostingsHandler hostingsHandler;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final GoalMutationService goalMutationService;
    private final GrutUacCampaignService grutUacCampaignService;
    private final GrutTransactionProvider grutTransactionProvider;
    private final int grutRetries;

    @Autowired
    public CampaignMutationService(
            CampaignValidationService campaignValidationService,
            CampaignService campaignService,
            CampaignOperationService campaignOperationService,
            CampMetrikaCountersService campMetrikaCountersService,
            MetrikaGoalsService metrikaGoalsService, SspPlatformsRepository sspPlatformsRepository,
            AttributeResolverService attributeResolverService,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            CampaignBudgetReachService campaignBudgetReachService,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            HostingsHandler hostingsHandler,
            MinusKeywordPreparingTool minusKeywordPreparingTool,
            CopyCampaignService copyCampaignService,
            GoalMutationService goalMutationService,
            GrutUacCampaignService grutUacCampaignService,
            GrutTransactionProvider grutTransactionProvider,
            @Value("${object_api.retries}") int grutRetries) {
        this.campaignValidationService = campaignValidationService;
        this.campaignService = campaignService;
        this.campaignOperationService = campaignOperationService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.metrikaGoalsService = metrikaGoalsService;
        this.sspPlatformsRepository = sspPlatformsRepository;
        this.attributeResolverService = attributeResolverService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.campaignBudgetReachService = campaignBudgetReachService;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.hostingsHandler = hostingsHandler;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.goalMutationService = goalMutationService;
        this.grutUacCampaignService = grutUacCampaignService;
        this.grutTransactionProvider = grutTransactionProvider;
        this.grutRetries = grutRetries;
    }

    GdAddCampaignPayload addCampaigns(GridGraphQLContext context, GdAddCampaigns input) {
        if (input.getCampaignAddItems().isEmpty()) {
            return new GdAddCampaignPayload()
                    .withAddedCampaigns(emptyList())
                    .withValidationResult(null);
        }

        final var clientId = context.getSubjectUser().getClientId();

        campaignValidationService.validateAddCampaigns(clientId, input);

        List<? extends BaseCampaign> campaigns = toCoreCampaigns(input, getCampaignConverterContext());

        CampaignOptions options = new CampaignOptions();
        RestrictedCampaignsAddOperation addOperation =
                campaignOperationService.createRestrictedCampaignAddOperation(campaigns, context.getOperator().getUid(),
                        UidAndClientId.of(context.getSubjectUser().getUid(), context.getSubjectUser().getClientId()),
                        options);

        MassResult<Long> addResult = addOperation.prepareAndApply();

        GdValidationResult validationResult = campaignValidationService.getValidationResult(addResult,
                path(field(GdAddCampaigns.CAMPAIGN_ADD_ITEMS)));

        List<GdAddCampaignPayloadItem> addCampaigns = getResults(addResult,
                id -> new GdAddCampaignPayloadItem().withId(id));

        return new GdAddCampaignPayload()
                .withAddedCampaigns(addCampaigns)
                .withValidationResult(validationResult);
    }

    GdUpdateCampaignPayload updateCampaigns(GridGraphQLContext context, GdUpdateCampaigns input) {
        if (input.getCampaignUpdateItems().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }

        final var clientId = context.getSubjectUser().getClientId();

        campaignValidationService.validateUpdateCampaigns(clientId, input);

        List<? extends ModelChanges<? extends BaseCampaign>> modelChanges = gdUpdateCampaignsToCoreModelChanges(input,
                getCampaignConverterContext());

        return updateCampaigns(context, modelChanges);
    }

    public GdUpdateCampaignPayload deleteCampaigns(GridGraphQLContext context, GdCampaignIdsList campaignIds) {
        campaignValidationService.validateDeleteCampaigns(campaignIds);

        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = context.getSubjectUser().getClientId();

        MassResult<Long> deleteResult = campaignService.deleteCampaigns(campaignIds.getCampaignIds(), operatorUid,
                clientId,
                Applicability.PARTIAL);

        GdValidationResult validationResult = campaignValidationService.getValidationResult(deleteResult,
                path(field(GdCampaignIdsList.CAMPAIGN_IDS)));

        List<GdUpdateCampaignPayloadItem> deletedCampaigns = getResults(deleteResult,
                id -> new GdUpdateCampaignPayloadItem().withId(id));

        return new GdUpdateCampaignPayload()
                .withUpdatedCampaigns(deletedCampaigns)
                .withValidationResult(validationResult);
    }

    public GdCampaignIdsList deleteBrandLiftForCampaigns(GridGraphQLContext context, GdCampaignIdsList campaignIds) {
        var clientId = context.getSubjectUser().getClientId();
        campaignValidationService.validateDeleteBrandLift(campaignIds, clientId);
        int shard = shardHelper.getShardByClientId(clientId);

        var validator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(
                        context.getOperator().getUid(),
                        clientId,
                        campaignIds.getCampaignIds())
                .createValidator(CampaignAccessType.READ_WRITE);

        var validCids = StreamEx.of(campaignIds.getCampaignIds())
                .map(validator)
                .filter(vr -> !vr.hasAnyErrors())
                .map(ValidationResult::getValue)
                .toList();

        var brandStatusForCampaigns = campaignBudgetReachService.getBrandStatusForCampaigns(shard, clientId, validCids);
        var cidsForBrandLiftDeletion = validCids
                .stream()
                .filter(cid -> brandStatusForCampaigns.get(cid) != null
                        && brandStatusForCampaigns.get(cid).getSurveyStatusDaily() == SurveyStatus.DRAFT)
                .collect(toList());

        campaignRepository.deleteBrandSurveyId(shard, cidsForBrandLiftDeletion);
        return new GdCampaignIdsList().withCampaignIds(cidsForBrandLiftDeletion);
    }

    private CampaignConverterContext getCampaignConverterContext() {
        return CampaignConverterContext.create(attributeResolverService, sspPlatformsRepository, hostingsHandler);
    }


    public GdUpdateCampaignPayload updateCampaignsTimeTargeting(GridGraphQLContext context,
                                                                @GraphQLNonNull GdUpdateCampaignsTimeTargeting input) {
        final var clientId = context.getSubjectUser().getClientId();

        campaignValidationService.validateUpdateTimeTargetingRequest(clientId, input);

        return updateCampaigns(context, UpdateCampaignMutationConverter.gdUpdateTimeTargetToCoreModelChanges(input));
    }

    public GdUpdateCampaignMetrikaCountersPayload updateCampaignMetrikaCounters(ClientId clientId,
                                                                                GdUpdateCampaignMetrikaCounters input) {
        campaignValidationService.validateUpdateCampaignMetrikaCountersRequest(input, clientId);

        GdValidationResult preValidationResult =
                campaignValidationService.validateUpdateCampaignMetrikaCountersAccess(clientId, input);
        if (preValidationResult != null) {
            return toGdUpdateCampaignMetrikaCountersPayloadWithOnlyValidationResult(preValidationResult);
        }

        UpdateCampMetrikaCountersRequest updateCampMetrikaCountersRequest = toUpdateCampMetrikaCountersRequest(input);
        Result<UpdateCampMetrikaCountersRequest> result =
                updateCampaignMetrikaCounters(clientId, input.getAction(), updateCampMetrikaCountersRequest);

        return toGdUpdateCampaignMetrikaCountersPayload(input.getCampaignIds(), result);
    }

    public GdUpdateCampaignPayload updateMeaningfulGoals(GridGraphQLContext context, GdUpdateMeaningfulGoals input) {
        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = requireNonNull(context.getSubjectUser()).getClientId();

        campaignValidationService.validateUpdateMeaningfulGoalsRequest(input);
        var requestedGoalIds = listToSet(input.getMeaningfulGoals(), GdMeaningfulGoalRequest::getGoalId);
        var modelChanges = toCoreModelChanges(
                input,
                input.getBindCounters()
                        ? getNewCampaignsMetrikaCounters(
                        operatorUid, clientId, input.getCampaignIds(), requestedGoalIds)
                        : Map.of(),
                input.getBindCounters() ? null : getNewMeaningfulGoalsWithOldCounters(clientId, input));
        return updateCampaigns(context, modelChanges);
    }

    /**
     * Для каждой кампании из {@code campaignIds} вычисляет новые идентификаторы счётчиков.
     * Список с новыми идентификаторами состоит из уже имеющихся на кампании + счётчики для запрошенных целей.
     * <p>
     * Метод не осуществляет никакой валидации. Считаем, что это произойдёт дальше.
     */
    private Map<Long, List<Long>> getNewCampaignsMetrikaCounters(Long operatorUid,
                                                                 ClientId clientId,
                                                                 List<Long> campaignIds,
                                                                 Set<Long> requestedGoalIds) {
        // Берём существующие счётчики на кампаниях
        var existingCountersByCids =
                campMetrikaCountersService.getCounterByCampaignIds(clientId, campaignIds);

        // Получаем доступные клиенту цели и цели со счетчиков на кампании
        // и для каждой цели из requestedGoalIds вычисляем идентификаторы счётчиков
        Set<Long> availableCounterIdsOfRequestedGoals = StreamEx.of(
                        goalMutationService.getMetrikaGoalsByAvailableAndCampaignCounters(
                                operatorUid, clientId, existingCountersByCids).getGoals())
                .filter(goal -> requestedGoalIds.contains(goal.getId()))
                .map(GdGoal::getCounterId)
                .nonNull()
                .map(Integer::longValue)
                .toSet();

        // Подмешиваем новые идентификаторы счётчиков к уже существующим на кампании
        return EntryStream.of(existingCountersByCids)
                .mapValues(ids -> {
                    Set<Long> uniqueCounterIds = new HashSet<>();
                    uniqueCounterIds.addAll(ids);
                    uniqueCounterIds.addAll(availableCounterIdsOfRequestedGoals);
                    return List.copyOf(uniqueCounterIds);
                }).toMap();
    }

    private Map<Long, List<Long>> getNewMeaningfulGoalsWithOldCounters(ClientId clientId,
                                                                       GdUpdateMeaningfulGoals input) {
        var existingCountersByCids = campMetrikaCountersService.getCounterByCampaignIds(clientId,
                input.getCampaignIds());

        Set<Long> existingCounterIds = StreamEx.ofValues(existingCountersByCids).flatMap(StreamEx::of).toSet();
        var availableCounterIds =
                campMetrikaCountersService.getAvailableAndFilterInputCounterIdsForGoals(clientId, existingCounterIds);
        var availableGoalsForCounterId = metrikaGoalsService
                .getMetrikaGoalsByCounterIds(clientId, availableCounterIds);
        var requestedGoalIds = mapList(input.getMeaningfulGoals(), GdMeaningfulGoalRequest::getGoalId);

        return EntryStream.of(existingCountersByCids)
                .mapValues(counters -> {
                    Set<Goal> availableGoalsForCampaingId = new HashSet<>();
                    for (Set<Goal> goals : EntryStream.of(availableGoalsForCounterId)
                            .filterKeys(counters::contains)
                            .toMap().values()) {
                        availableGoalsForCampaingId.addAll(goals);
                    }
                    return StreamEx.of(availableGoalsForCampaingId)
                            .map(Goal::getId)
                            .filter(requestedGoalIds::contains)
                            .toList();
                }).toMap();
    }

    GdUpdateCampaignPayload updateCampaignsOrganization(GridGraphQLContext context,
                                                        GdUpdateCampaignsOrganization input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }
        campaignValidationService.validateUpdateCampaignsOrganization(input, context.getSubjectUser().getClientId());
        var modelChanges = toModelChangesOrganization(input);
        return updateCampaigns(context, modelChanges);
    }

    GdUpdateCampaignPayload updateCampaignsStartDate(GridGraphQLContext context, GdUpdateCampaignsStartDate input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }
        campaignValidationService.validateUpdateCampaignsStartDate(input, context.getSubjectUser().getClientId());
        var modelChanges = toModelChangesStartDate(input);
        return updateCampaigns(context, modelChanges);
    }

    GdUpdateCampaignPayload updateCampaignPromoExtensionChanges(GridGraphQLContext context,
                                                                GdUpdateCampaignsPromoExtension input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }
        campaignValidationService.validateUpdateCampaignsWithPromoExtension(input,
                requireNonNull(context.getSubjectUser()).getClientId());
        var modelChanges = toModelPromoExtensionChanges(input);
        return updateCampaigns(context, modelChanges, UPDATE_PROMO_EXTENSION_APPLICABILITY);
    }

    GdUpdateCampaignPayload updateCampaignsMinusKeywords(GridGraphQLContext context,
                                                         GdUpdateCampaignsMinusKeywords input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }
        ClientId clientId = requireNonNull(context.getSubjectUser()).getClientId();
        var shard = shardHelper.getShardByClientIdStrictly(clientId);

        campaignValidationService.validateUpdateCampaignsMinusKeywords(input, context.getSubjectUser().getClientId());
        var campaigns = campaignRepository.getCampaigns(shard, input.getCampaignIds());
        var keywords = input.getMinusKeywords();

        var vr = preValidateMinusKeywords(context, campaigns, keywords);
        if (vr.hasAnyErrors()) {
            vr = cloneValidationResultSubNodesWithIssues(vr);
            return new GdUpdateCampaignPayload()
                    .withValidationResult(campaignValidationService.getValidationResult(vr,
                            path(field(GdUpdateCampaigns.CAMPAIGN_UPDATE_ITEMS))))
                    .withUpdatedCampaigns(mapList(vr.getValue(), r -> null));
        }

        List<? extends ModelChanges<? extends BaseCampaign>> modelChanges = StreamEx.of(campaigns)
                .map(campaign -> new ModelChanges<>(campaign.getId(), CampaignWithMinusKeywords.class)
                        .process(getNewMinusKeywords(campaign.getMinusKeywords(), keywords,
                                input.getAction()), CampaignWithMinusKeywords.MINUS_KEYWORDS))
                .toList();
        return updateCampaigns(context, modelChanges);
    }

    private ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> preValidateMinusKeywords(
            GridGraphQLContext context, List<Campaign> campaigns, List<String> keywords) {
        List<? extends ModelChanges<? extends BaseCampaign>> modelChanges = StreamEx.of(campaigns)
                .map(campaign -> new ModelChanges<>(campaign.getId(), CampaignWithMinusKeywords.class)
                        .process(keywords, CampaignWithMinusKeywords.MINUS_KEYWORDS))
                .toList();

        RestrictedCampaignsUpdateOperation updateOperation = createUpdateOperation(context, modelChanges,
                DEFAULT_APPLICABILITY);
        return updateOperation.preValidate();
    }

    public GdUpdateCampaignPayload updateCampaignsStrategy(GridGraphQLContext context,
                                                           GdUpdateCampaignsStrategy input) {
        campaignValidationService.validateUpdateCampaignsStrategyRequest(input, context.getSubjectUser().getClientId());
        Map<Long, List<Long>> newCounterIdsByCampaignIds = Map.of();
        if (nvl(input.getBindCounters(), false)) {
            Long goalId = input.getBiddingStrategy().getStrategyData().getGoalId();
            if (goalId != null) {
                Long operatorUid = context.getOperator().getUid();
                ClientId clientId = requireNonNull(context.getSubjectUser()).getClientId();
                newCounterIdsByCampaignIds = getNewCampaignsMetrikaCounters(operatorUid, clientId,
                        input.getCampaignIds(), Set.of(goalId));
            }
        }
        var modelChanges =
                UpdateCampaignsStrategyConverter.toCoreModelChanges(input, newCounterIdsByCampaignIds);
        return updateCampaigns(context, modelChanges);
    }

    public GdUpdateCampaignPayload updateCampaignsWeeklyBudget(GridGraphQLContext context,
                                                               GdUpdateCampaignsWeeklyBudget input) {
        campaignValidationService.validateUpdateCampaignsWeeklyBudgetRequest(input,
                context.getSubjectUser().getClientId());

        var clientId = requireNonNull(context.getSubjectUser()).getClientId();
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<Campaign> campaigns = campaignRepository.getCampaignsWithStrategy(shard, input.getCampaignIds());

        GdValidationResult preValidationResult =
                campaignValidationService.validateUpdateWeeklyBudget(campaigns, input);
        if (preValidationResult != null) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(preValidationResult);
        }

        var modelChanges =
                UpdateCampaignsWeeklyBudgetConverter.toCoreModelChanges(input, campaigns);
        var updateResult = updateCampaigns(context, modelChanges);
        var updatedCampaignIds = updateResult.getUpdatedCampaigns()
                .stream()
                .filter(Objects::nonNull)
                .map(GdUpdateCampaignPayloadItem::getId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        var updatedUcCampaigns = campaigns.stream()
                .filter(c -> AvailableCampaignSources.INSTANCE.isUC(c.getSource()))
                .map(Campaign::getId)
                .filter(updatedCampaignIds::contains)
                .collect(Collectors.toList());

        if (!updatedUcCampaigns.isEmpty()) {
            grutTransactionProvider.runInRetryableTransaction(grutRetries, null, () -> {
                grutUacCampaignService.updateWeeklyBudget(updatedUcCampaigns,
                        UacYdbUtils.INSTANCE.moneyToDb(input.getWeeklyBudget()));
                return null;
            });
        }

        return updateResult;
    }

    private RestrictedCampaignsUpdateOperation createUpdateOperation(
            GridGraphQLContext context,
            List<? extends ModelChanges<? extends BaseCampaign>> modelChanges,
            Applicability applicability) {
        var options = new CampaignOptions();
        return campaignOperationService.createRestrictedCampaignUpdateOperation(modelChanges,
                context.getOperator().getUid(),
                UidAndClientId.of(context.getSubjectUser().getUid(), context.getSubjectUser().getClientId()),
                applicability, options);
    }

    private GdUpdateCampaignPayload updateCampaigns(
            GridGraphQLContext context,
            List<? extends ModelChanges<? extends BaseCampaign>> modelChanges) {
        RestrictedCampaignsUpdateOperation updateOperation = createUpdateOperation(context, modelChanges,
                DEFAULT_APPLICABILITY);

        return performUpdateReturnPayload(updateOperation);
    }

    private GdUpdateCampaignPayload updateCampaigns(
            GridGraphQLContext context,
            List<? extends ModelChanges<? extends BaseCampaign>> modelChanges,
            Applicability applicability) {
        RestrictedCampaignsUpdateOperation updateOperation = createUpdateOperation(context, modelChanges,
                applicability);

        return performUpdateReturnPayload(updateOperation);
    }

    private GdUpdateCampaignPayload performUpdateReturnPayload(RestrictedCampaignsUpdateOperation updateOperation) {
        MassResult<Long> updateResult = updateOperation.apply();

        GdValidationResult validationResult =
                campaignValidationService.getValidationResult(updateResult,
                        path(field(GdUpdateCampaigns.CAMPAIGN_UPDATE_ITEMS)));

        List<GdUpdateCampaignPayloadItem> updatedCampaigns = getResults(updateResult,
                id -> new GdUpdateCampaignPayloadItem().withId(id));

        return new GdUpdateCampaignPayload()
                .withUpdatedCampaigns(updatedCampaigns)
                .withValidationResult(validationResult);
    }

    private GdUpdateCampaignMetrikaCountersPayload toGdUpdateCampaignMetrikaCountersPayload(
            List<Long> requestCampaignIds, Result<UpdateCampMetrikaCountersRequest> result) {
        List<Long> updatedCampaignIds = CampMetrikaCountersService.getUpdatedCampaignIdsFromResult(result);
        Set<Long> skippedCampaignIds = StreamEx.of(requestCampaignIds)
                .remove(updatedCampaignIds::contains)
                .toSet();

        GdValidationResult gdValidationResult =
                campaignValidationService.getValidationResult(result.getValidationResult(), path());

        return new GdUpdateCampaignMetrikaCountersPayload()
                .withUpdatedCampaigns(mapList(updatedCampaignIds,
                        id -> new GdUpdateCampaignPayloadItem().withId(id)))
                .withSkippedCampaignIds(skippedCampaignIds)
                .withValidationResult(gdValidationResult);
    }

    private Result<UpdateCampMetrikaCountersRequest> updateCampaignMetrikaCounters(ClientId clientId,
                                                                                   GdMassUpdateAction action,
                                                                                   UpdateCampMetrikaCountersRequest request) {
        switch (action) {
            case ADD:
                return campMetrikaCountersService
                        .addCampMetrikaCounters(clientId, request, UPDATE_METRIKA_COUNTERS_APPLICABILITY);
            case REPLACE:
                return campMetrikaCountersService
                        .replaceCampMetrikaCounters(clientId, request, UPDATE_METRIKA_COUNTERS_APPLICABILITY);
            case REMOVE:
                return campMetrikaCountersService
                        .removeCampMetrikaCounters(clientId, request, UPDATE_METRIKA_COUNTERS_APPLICABILITY);
            default:
                throw new IllegalStateException("Unexpected action: " + action);
        }
    }

    public GdUpdateCampaignPayload resetCampaignFlightStatusApprove(GridGraphQLContext context,
                                                                    GdCampaignIdsList campaignIds) {
        campaignValidationService.validateResetCampaignFlightStatusApprove(campaignIds);

        User operator = context.getOperator();
        User client = context.getSubjectUser();

        List<ModelChanges<CpmPriceCampaign>> modelChanges = StreamEx.of(campaignIds.getCampaignIds())
                .map(campaignId -> ModelChanges.build(campaignId, CpmPriceCampaign.class,
                        CpmPriceCampaign.FLIGHT_STATUS_APPROVE, PriceFlightStatusApprove.NEW))
                .toList();

        var options = new CampaignOptions();
        RestrictedCampaignsUpdateOperation updateOperation =
                campaignOperationService.createRestrictedCampaignUpdateOperation(modelChanges, operator.getUid(),
                        UidAndClientId.of(client.getUid(), client.getClientId()), options);
        MassResult<Long> updateResult = updateOperation.apply();

        GdValidationResult validationResult = campaignValidationService.getSkipPathValidationResult(updateResult,
                path(field(GdCampaignIdsList.CAMPAIGN_IDS)));

        List<GdUpdateCampaignPayloadItem> updatedCampaigns = getResults(updateResult,
                id -> new GdUpdateCampaignPayloadItem().withId(id));

        return new GdUpdateCampaignPayload()
                .withUpdatedCampaigns(updatedCampaigns)
                .withValidationResult(validationResult);
    }

    public GdUpdateCampaignPayload updateCampaignsDayBudget(GridGraphQLContext context,
                                                            GdUpdateCampaignsDayBudget input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }

        campaignValidationService.validateUpdateCampaignsDayBudget(input, context.getSubjectUser().getClientId());
        var modelChanges = toModelChangesDayBudget(input);
        return updateCampaigns(context, modelChanges);
    }

    @Nullable
    public List<String> getNewMinusKeywords(@Nullable List<String> existingMinusKeywords,
                                            @Nullable List<String> newMinusKeywords,
                                            GdUpdateCampaignsMinusKeywordsAction action) {
        existingMinusKeywords = ListUtils.emptyIfNull(existingMinusKeywords);
        newMinusKeywords = ListUtils.emptyIfNull(newMinusKeywords);
        List<String> result = new ArrayList<>();
        switch (action) {
            case ADD:
                result.addAll(newMinusKeywords);
                result.addAll(existingMinusKeywords);
                result = minusKeywordPreparingTool.removeDuplicatesAndSort(result);
                break;
            case REMOVE:
                result = minusKeywordPreparingTool.removeMinusKeywords(existingMinusKeywords, newMinusKeywords);
                break;
            case REPLACE:
                result = minusKeywordPreparingTool.replaceMinusKeywords(newMinusKeywords);
                break;
            default:
                throw new IllegalStateException("No such value: " + action);
        }
        if (result.size() == 0) {
            return null;
        }
        return result;
    }

    public GdUpdateCampaignPayload updateCampaignsPromotions(GridGraphQLContext context,
                                                             GdUpdateCampaignsPromotions input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }

        campaignValidationService.validateUpdateCampaignsPromotions(input, context.getSubjectUser().getClientId());
        var modelChanges = toModelChangesCampaignsPromotions(input);
        return updateCampaigns(context, modelChanges);
    }

    GdUpdateCampaignPayload addBrandSurvey(GridGraphQLContext context, GdUpdateCampaignsAddBrandSurvey input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }
        campaignValidationService.validateAddBrandSurveyParams(context.getSubjectUser().getClientId(), input);
        var modelChanges = toModelChangesBrandSurvey(input);
        return updateCampaigns(context, modelChanges, Applicability.FULL);
    }

    GdUpdateCampaignPayload updateBannerHrefParams(GridGraphQLContext context,
                                                   GdUpdateCampaignsAddBannerHrefParams input) {
        if (input.getCampaignIds().isEmpty()) {
            return new GdUpdateCampaignPayload()
                    .withUpdatedCampaigns(emptyList())
                    .withValidationResult(null);
        }
        campaignValidationService.validateUpdateBannerHrefParams(context.getSubjectUser().getClientId(), input);
        var modelChanges = toModelChangesBannerHrefParams(input);
        return updateCampaigns(context, modelChanges, Applicability.FULL);
    }


}
