package ru.yandex.direct.grid.processing.service.campaign;

import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusCorrect;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.util.filters.FilterProcessor;
import ru.yandex.direct.grid.core.util.filters.FilterProcessor.Builder;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdGoalStats;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaignFeature;
import ru.yandex.direct.grid.model.campaign.GdCampaignRedirectInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdInternalCampaignWithPlace;
import ru.yandex.direct.grid.model.campaign.GdMetaCampaignFilterType;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;
import ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter;
import ru.yandex.direct.grid.processing.model.campaign.GdCalcType;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignFilter;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignOrderBy;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignOrderByField;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignSourceAvailability;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendation;

import static ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService.RECOMMENDATIONS_CLICKS_EXTRACTOR;
import static ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService.RECOMMENDATIONS_COST_EXTRACTOR;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.afterOrEqual;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.beforeOrEqual;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.contains;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.containsMatching;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.equalTo;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.innerContainsMatching;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.isSubStringCU;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.numericContainsAny;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignSource;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toMetaCampaignFilterType;
import static ru.yandex.direct.grid.processing.util.StatHelper.CAMPAIGN_STATS_FILTER_PROCESSOR;
import static ru.yandex.direct.grid.processing.util.StatHelper.EMPTY_GOAL_STAT;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.BiPredicates.not;

/**
 * Класс, содержащий обработчики, используемые для работы {@link CampaignInfoService}
 * <p>
 * Обработчики вынесены отдельно, чтобы не засорять основной код
 */
@ParametersAreNonnullByDefault
public class CampaignServiceUtils {
    /**
     * Обработчик фильтров, основывающихся на полях внутреннего представления кампании
     */
    static final FilterProcessor<GdCampaignFilter, GdiCampaign> INT_CAMPAIGN_FILTER_PROCESSOR =
            getIntCampaignFilterProcessor(true);
    static final FilterProcessor<GdCampaignFilter, GdiCampaign> INT_CAMPAIGN_FILTER_PROCESSOR_WITHOUT_ARCHIVED_FILTER =
            getIntCampaignFilterProcessor(false);

    private static final Map<GdCampaignSource, String> CREATABLE_FEATURE_BY_CAMPAIGN_SOURCE = Map.of(
            GdCampaignSource.USLUGI, FeatureName.CAMPAIGN_SOURCE_USLUGI_CREATABLE.getName()
    );
    private static final Map<GdCampaignSource, Set<CurrencyCode>> AVAILABLE_CURRENCIES_BY_CAMPAIGN_SOURCE = Map.of(
            GdCampaignSource.USLUGI, Set.of(CurrencyCode.RUB)
    );
    private static final Logger LOGGER = LoggerFactory.getLogger(CampaignServiceUtils.class);
    private static final String USLUGI_EDIT_URL = "https://uslugi.yandex.ru/accounts/promotion/manage?campaign_id=";
    private static final String USLUGI_CREATE_URL = "https://uslugi.yandex.ru/accounts/promotion";
    private static final String ZEN_VIEW_URL = "https://zen.yandex.ru/profile/editor/campaigns/direct/"; // + campaignId
    private static final String ZEN_VIEW_STATS_URL =
            "https://zen.yandex.ru/profile/editor/campaigns/direct/statistics/"; // + campaignId

    public static final Set<CampaignType> CPC_CAMPAIGN_TYPES = Set.of(
            CampaignType.TEXT,
            CampaignType.PERFORMANCE,
            CampaignType.DYNAMIC,
            CampaignType.MCBANNER,
            CampaignType.MOBILE_CONTENT,
            CampaignType.CONTENT_PROMOTION
    );

    public static final Set<CampaignType> CPM_CAMPAIGN_TYPES = Set.of(
            CampaignType.CPM_BANNER,
            CampaignType.CPM_PRICE,
            CampaignType.CPM_DEALS,
            CampaignType.CPM_YNDX_FRONTPAGE
    );

    public static final Set<GdMetaCampaignFilterType> UC_CAMPAIGN_TYPES = Set.of(
            GdMetaCampaignFilterType.UC_TEXT,
            GdMetaCampaignFilterType.UC_MOBILE_CONTENT,
            GdMetaCampaignFilterType.UC_CPM_BANNER,
            GdMetaCampaignFilterType.UC_ECOM
    );

    private CampaignServiceUtils() {
    }

    private static FilterProcessor<GdCampaignFilter, GdiCampaign> getIntCampaignFilterProcessor(boolean filterArchived) {
        var builder = new Builder<GdCampaignFilter, GdiCampaign>()
                .withFilter(GdCampaignFilter::getCampaignIdIn, contains(GdiCampaign::getId))
                .withFilter(GdCampaignFilter::getCampaignIdNotIn, not(contains(GdiCampaign::getId)));
        if (filterArchived) {
            builder.withFilter(GdCampaignFilter::getArchived, equalTo(GdiCampaign::getArchived));
        }
        builder.withFilter(GdCampaignFilter::getCurrencyArchived, equalTo(GdiCampaign::getCurrencyConverted))
                .withFilter(GdCampaignFilter::getSourceIn, CampaignServiceUtils::sourceInPredicate)
                .withFilter(GdCampaignFilter::getSourceNotIn, CampaignServiceUtils::sourceNotInPredicate)
                .withFilter(GdCampaignFilter::getCampaignIdContainsAny, numericContainsAny(GdiCampaign::getId))
                .withFilter(GdCampaignFilter::getNameContains, isSubStringCU(GdiCampaign::getName))
                .withFilter(GdCampaignFilter::getNameNotContains, not(isSubStringCU(GdiCampaign::getName)))
                .withFilter(GdCampaignFilter::getNameIn,
                        containsMatching(GdiCampaign::getName, String::equalsIgnoreCase))
                .withFilter(GdCampaignFilter::getNameNotIn,
                        not(containsMatching(GdiCampaign::getName, String::equalsIgnoreCase)))
                .withFilter(GdCampaignFilter::getNameInMulti,
                        innerContainsMatching(isSubStringCU(GdiCampaign::getName)))
                .withFilter(gd -> mapSet(gd.getTypeIn(), CampaignDataConverter::toCampaignType),
                        contains(GdiCampaign::getType))
                .withFilter(GdCampaignFilter::getMetaTypeIn, CampaignServiceUtils::metaTypePredicate)
                .withFilter(GdCampaignFilter::getCalcType, CampaignServiceUtils::calcTypePredicate)
                .withFilter(gd -> mapSet(gd.getPlatformIn(), CampaignDataConverter::toCampaignsPlatform),
                        contains(GdiCampaign::getPlatform))
                .withFilter(GdCampaignFilter::getMinStartDate, beforeOrEqual(GdiCampaign::getStartDate))
                .withFilter(GdCampaignFilter::getMaxStartDate, afterOrEqual(GdiCampaign::getStartDate))
                .withFilter(GdCampaignFilter::getMinFinishDate, beforeOrEqual(GdiCampaign::getFinishDate))
                .withFilter(GdCampaignFilter::getMaxFinishDate, afterOrEqual(GdiCampaign::getFinishDate))
                .withFilter(GdCampaignFilter::getHasBidModifiers, equalTo(GdiCampaign::getHasBidModifiers))
                .withFilter(GdCampaignFilter::getHasBroadmatch, equalTo(GdiCampaign::getHasBroadMatch))
                .withFilter(GdCampaignFilter::getHasMetrika,
                        (hm, c) -> hm == CollectionUtils.isNotEmpty(c.getMetrikaCounters()))
                .withFilter(GdCampaignFilter::getHasSiteMonitoring, equalTo(GdiCampaign::getHasSiteMonitoring))
                .withFilter(GdCampaignFilter::getFavorite, equalTo(GdiCampaign::getFavorite))
                .withFilter(GdCampaignFilter::getInternalAdPlaceIdIn,
                        contains(GdiCampaign::getInternalAdPlaceId))
                .withFilter(GdCampaignFilter::getUac,
                        (uac, c) -> uac == (AvailableCampaignSources.INSTANCE.isUC(GdiCampaignSource.toSource(c.getSource()))))
                .withFilter(GdCampaignFilter::getDescriptionContains, isSubStringCU(GdiCampaign::getDescription))
                .withFilter(GdCampaignFilter::getDescriptionNotContains,
                        not(isSubStringCU(GdiCampaign::getDescription)));
        return builder.build();
    }

    private static boolean calcTypePredicate(GdCalcType calcType, GdiCampaign campaign) {
        return calcType == GdCalcType.CPM && CampaignTypeKinds.CPM.contains(campaign.getType())
                || calcType == GdCalcType.CPC && !CampaignTypeKinds.CPM.contains(campaign.getType())
                || isBothCpcAndCpmCampaign(campaign);
    }

    static boolean metaTypePredicate(Set<GdMetaCampaignFilterType> filterTypes, GdiCampaign campaign) {
        GdMetaCampaignFilterType metaType = toMetaCampaignFilterType(campaign);
        if (filterTypes.contains(GdMetaCampaignFilterType.UC)) {
            return UC_CAMPAIGN_TYPES.contains(metaType);
        } else {
            return filterTypes.contains(metaType);
        }
    }

    static boolean sourceInPredicate(Set<GdCampaignSource> sources, GdiCampaign campaign) {
        return sources.contains(toGdCampaignSource(campaign.getSource()));
    }

    static boolean sourceNotInPredicate(Set<GdCampaignSource> sources, GdiCampaign campaign) {
        return !sourceInPredicate(sources, campaign);
    }

    // кампании Дзена могут крутиться за клики и просмотры
    private static boolean isBothCpcAndCpmCampaign(GdiCampaign campaign) {
        return campaign.getSource().equals(GdiCampaignSource.ZEN);
    }

    /**
     * Обработчик фильтров по внешнему представлению кампании
     */
    static final FilterProcessor<GdCampaignFilter, GdCampaign> CAMPAIGN_FILTER_PROCESSOR =
            getCampaignFilterProcessor(true);

    static final FilterProcessor<GdCampaignFilter, GdCampaign> CAMPAIGN_FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS =
            getCampaignFilterProcessor(false);

    private static FilterProcessor<GdCampaignFilter, GdCampaign> getCampaignFilterProcessor(boolean enableFilterByStatus) {
        var builder = new Builder<GdCampaignFilter, GdCampaign>()
                .withFilter(GdCampaignFilter::getStrategyTypeIn, contains(c -> c.getFlatStrategy().getType()))
                .withFilter(GdCampaignFilter::getCampaignStrategyTypeIn,
                        contains(c -> c.getStrategy().getStrategyType()));
        if (enableFilterByStatus) {
            builder.withFilter(GdCampaignFilter::getStatusIn, contains(c -> c.getStatus().getPrimaryStatus()));
        }
        builder.withFilter(GdCampaignFilter::getStats, CAMPAIGN_STATS_FILTER_PROCESSOR)
                .withFilter(GdCampaignFilter::getServicedStateIn, contains(c -> c.getAccess().getServicedState()))
                .withFilter(GdCampaignFilter::getHasCompletedMediaplan,
                        equalTo(c -> c.getFeatures().contains(GdCampaignFeature.HAS_COMPLETED_MEDIAPLAN)));
        return builder.build();
    }

    private static final Function<GdCampaign, String> CAMPAIGN_EMAIL_EXTRACTOR =
            campaign -> campaign.getNotification().getEmailSettings().getEmail();

    private static final Function<GdCampaign, Long> CAMPAIGN_INTERNAL_AD_PLACE_ID_EXTRACTOR =
            campaign -> {
                if (!(campaign instanceof GdInternalCampaignWithPlace)) {
                    return null;
                }

                return ((GdInternalCampaignWithPlace) campaign).getPlaceId();
            };

    /**
     * Мапы методов для сортировки
     */
    private static final Map<GdCampaignOrderByField, Function<GdCampaign, ? extends Comparable>> CAMP_COMPARE_FUNC =
            ImmutableMap.<GdCampaignOrderByField, Function<GdCampaign, ? extends Comparable>>builder()
                    .put(GdCampaignOrderByField.ID, GdCampaign::getId)
                    .put(GdCampaignOrderByField.NAME, GdCampaign::getName)
                    .put(GdCampaignOrderByField.TYPE, GdCampaign::getType)
                    .put(GdCampaignOrderByField.EMAIL, CAMPAIGN_EMAIL_EXTRACTOR)
                    .put(GdCampaignOrderByField.SUM, GdCampaign::getSum)
                    .put(GdCampaignOrderByField.SUM_REST, GdCampaign::getSumRest)
                    .put(GdCampaignOrderByField.START_DATE, GdCampaign::getStartDate)
                    .put(GdCampaignOrderByField.END_DATE, GdCampaign::getEndDate)
                    .put(GdCampaignOrderByField.STAT_COST, stat(GdEntityStats::getCost))
                    .put(GdCampaignOrderByField.STAT_COST_WITH_TAX, stat(GdEntityStats::getCostWithTax))
                    .put(GdCampaignOrderByField.STAT_SHOWS, stat(GdEntityStats::getShows))
                    .put(GdCampaignOrderByField.STAT_CLICKS, stat(GdEntityStats::getClicks))
                    .put(GdCampaignOrderByField.STAT_CTR, stat(GdEntityStats::getCtr))
                    .put(GdCampaignOrderByField.STAT_REVENUE, stat(GdEntityStats::getRevenue))
                    .put(GdCampaignOrderByField.STAT_GOALS, stat(GdEntityStats::getGoals))
                    .put(GdCampaignOrderByField.STAT_CPM_PRICE, stat(GdEntityStats::getCpmPrice))
                    .put(GdCampaignOrderByField.STAT_PROFITABILITY, stat(GdEntityStats::getProfitability))
                    .put(GdCampaignOrderByField.STAT_CRR, stat(GdEntityStats::getCrr))
                    .put(GdCampaignOrderByField.STAT_AVG_DEPTH, stat(GdEntityStats::getAvgDepth))
                    .put(GdCampaignOrderByField.STAT_AVG_GOAL_COST, stat(GdEntityStats::getAvgGoalCost))
                    .put(GdCampaignOrderByField.STAT_AVG_CLICK_COST, stat(GdEntityStats::getAvgClickCost))
                    .put(GdCampaignOrderByField.STAT_AVG_SHOW_POSITION, stat(GdEntityStats::getAvgShowPosition))
                    .put(GdCampaignOrderByField.STAT_AVG_CLICK_POSITION, stat(GdEntityStats::getAvgClickPosition))
                    .put(GdCampaignOrderByField.STAT_BOUNCE_RATE, stat(GdEntityStats::getBounceRate))
                    .put(GdCampaignOrderByField.STAT_CONVERSION_RATE, stat(GdEntityStats::getConversionRate))
                    .put(GdCampaignOrderByField.RECOMMENDATIONS_CLICKS,
                            recommendations(RECOMMENDATIONS_CLICKS_EXTRACTOR))
                    .put(GdCampaignOrderByField.RECOMMENDATIONS_COST, recommendations(RECOMMENDATIONS_COST_EXTRACTOR))
                    .put(GdCampaignOrderByField.INTERNAL_AD_PLACE_ID, CAMPAIGN_INTERNAL_AD_PLACE_ID_EXTRACTOR)
                    .put(GdCampaignOrderByField.DESCRIPTION, GdCampaign::getDescription)
                    .put(GdCampaignOrderByField.STATUS, c -> c.getAggregatedStatusInfo() == null ?
                            0 : c.getAggregatedStatusInfo().getStatus().getSortOrderValue())
                    .build();

    private static final Map<GdCampaignOrderByField, Function<GdGoalStats, ? extends Comparable>> GOAL_COMPARE_FUNC =
            ImmutableMap.<GdCampaignOrderByField, Function<GdGoalStats, ? extends Comparable>>builder()
                    .put(GdCampaignOrderByField.STAT_GOALS, GdGoalStats::getGoals)
                    .put(GdCampaignOrderByField.STAT_CONVERSION_RATE, GdGoalStats::getConversionRate)
                    .put(GdCampaignOrderByField.STAT_COST_PER_ACTION, GdGoalStats::getCostPerAction)
                    .build();

    static Comparator<GdCampaign> getComparator(List<GdCampaignOrderBy> orderBy) {

        if (CollectionUtils.isEmpty(orderBy)) {
            return (a, b) -> 0;
        }

        Comparator<GdCampaign> comparator = null;
        for (GdCampaignOrderBy item : orderBy) {

            Function<GdCampaign, ? extends Comparable> func;
            if (item.getParams() == null) {
                func = CAMP_COMPARE_FUNC.get(item.getField());

                if (func == null) {
                    throw new IllegalArgumentException(String.format("No comparator for %s", item.getField()));
                }
            } else {
                Function<GdGoalStats, ? extends Comparable> goalFunc = GOAL_COMPARE_FUNC.get(item.getField());
                if (goalFunc == null) {
                    throw new IllegalArgumentException(String.format("No goal comparator for %s", item.getField()));
                }

                func = (Function<GdCampaign, Comparable>) campaign -> {

                    for (GdGoalStats goalStat : campaign.getGoalStats()) {
                        if (Objects.equals(goalStat.getGoalId(), item.getParams().getGoalId())) {
                            return goalFunc.apply(goalStat);
                        }
                    }

                    //Если цель не найдена, то все значения для сортировки считаем нулем, а не null
                    return goalFunc.apply(EMPTY_GOAL_STAT);
                };
            }

            Comparator<Comparable> naturalComparator = Comparator.nullsLast(Comparator.<Comparable>naturalOrder());
            Comparator<GdCampaign> currentComparator
                    = (a, b) -> naturalComparator.compare(func.apply(a), func.apply(b));

            currentComparator = item.getOrder() == Order.ASC ? currentComparator : currentComparator.reversed();

            if (comparator == null) {
                comparator = currentComparator;
            } else {
                comparator = comparator.thenComparing(currentComparator);
            }
        }
        if (comparator == null) {
            throw new IllegalArgumentException("Empty list of comparators");
        }

        return comparator;
    }

    private static <T> Function<GdCampaign, T> stat(Function<GdEntityStats, T> extractor) {
        return c -> c.getStats() == null ? null : extractor.apply(c.getStats());
    }

    private static <T> Function<GdCampaign, T> recommendations(Function<List<GdRecommendation>, T> extractor) {
        return c -> c.getRecommendations() == null ? null : extractor.apply(c.getRecommendations());
    }

    public static boolean isWallet(GdiBaseCampaign campaign) {
        return campaign.getType() == CampaignType.WALLET;
    }

    public static boolean isCpmPriceCampaignReadyToStart(GdiBaseCampaign campaign) {
        return isApprovedCpmPriceCampaign(campaign)
                && campaign.getFlightStatusCorrect() == PriceFlightStatusCorrect.YES
                && campaign.getShowing();
    }

    public static boolean isApprovedCpmPriceCampaign(GdiBaseCampaign campaign) {
        return campaign.getType() == CampaignType.CPM_PRICE
                && campaign.getFlightStatusApprove() == PriceFlightStatusApprove.YES;
    }

    public static GdCampaignRedirectInfo getRedirectInfo(GdCampaign campaign) {
        switch (campaign.getSource()) {
            //Пока нет ссылки на кабинет, оставляю заглушки для того, чтобы можно было протестировать.
            case USLUGI:
                return new GdCampaignRedirectInfo()
                        .withEditUrl(USLUGI_EDIT_URL + campaign.getId())
                        .withViewUrl(USLUGI_EDIT_URL + campaign.getId());
            case EDA:
                return new GdCampaignRedirectInfo()
                        .withEditUrl("https://vendor.eda.yandex/promotion/places")
                        .withViewUrl("https://vendor.eda.yandex/promotion/places");
            case ZEN:
                return new GdCampaignRedirectInfo()
                        .withViewUrl(ZEN_VIEW_URL + campaign.getId())
                        .withViewStatsUrl(ZEN_VIEW_STATS_URL + campaign.getId());
            default:
                return null;
        }
    }

    public static List<GdCampaignSourceAvailability> calculateCampaignSourcesAvailability(Set<String> enabledFeatures,
                                                                                          GdClientInfo clientInfo,
                                                                                          Collection<Long> representativesUids) {
        Set<GdCampaignSource> allowedCreateRedirectSources = clientInfo.getAgencyClientId() == null ?
                Set.of() : EntryStream.of(CREATABLE_FEATURE_BY_CAMPAIGN_SOURCE)
                .filterValues(enabledFeatures::contains)
                .filterKeys(source -> !AVAILABLE_CURRENCIES_BY_CAMPAIGN_SOURCE.containsKey(source) ||
                        AVAILABLE_CURRENCIES_BY_CAMPAIGN_SOURCE.get(source).contains(clientInfo.getWorkCurrency())
                )
                .keys()
                .toSet();
        return StreamEx.of(allowedCreateRedirectSources)
                .map(source -> new GdCampaignSourceAvailability()
                        .withSource(source)
                        .withVisible(true) // выпилить, когда фронт его выпилит в DIRECT-143172
                        .withEditable(true) // выпилить, когда фронт его выпилит в DIRECT-143172
                        .withCreateRedirectUrl(createRedirectUrl(
                                allowedCreateRedirectSources, source, clientInfo, representativesUids
                        ))
                ).toList();
    }

    @Nullable
    private static String createRedirectUrl(Set<GdCampaignSource> allowedSources, GdCampaignSource source,
                                            GdClientInfo clientInfo, Collection<Long> representativesUids) {
        // мало кейсов для switch
        if (allowedSources.contains(source) && source == GdCampaignSource.USLUGI) {
            return createUslugiRedirectUrl(clientInfo.getChiefUser().getLogin(), representativesUids);
        }
        return null;
    }

    @Nullable
    private static String createUslugiRedirectUrl(String chiefLogin, Collection<Long> representativesUids) {
        try {
            var uriBuilder = new URIBuilder(USLUGI_CREATE_URL);
            uriBuilder.addParameter("client_login", chiefLogin);
            for (var uid : representativesUids) {
                uriBuilder.addParameter("client_puid", uid.toString());
            }
            return uriBuilder.build().toString();
        } catch (URISyntaxException e) {
            LOGGER.error("Couldn't construct a URL", e);
            return null;
        }
    }
}
