package ru.yandex.direct.grid.processing.service.campaign;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.container.UpdateCampMetrikaCountersRequest;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStub;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrganizationAndPhone;
import ru.yandex.direct.core.entity.campaign.model.ContentPromotionCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmYndxFrontpageCampaign;
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign;
import ru.yandex.direct.core.entity.campaign.model.InternalAutobudgetCampaign;
import ru.yandex.direct.core.entity.campaign.model.InternalDistribCampaign;
import ru.yandex.direct.core.entity.campaign.model.InternalFreeCampaign;
import ru.yandex.direct.core.entity.campaign.model.McBannerCampaign;
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign;
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects;
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.timetarget.GdHolidaysSettings;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignDayBudgetLimitsRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignDayBudgetLimitsRequestItem;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaignUnion;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddContentPromotionCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCpmYndxFrontpageCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddDynamicCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddInternalAutobudgetCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddInternalDistribCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddInternalFreeCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddMcBannerCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddMobileContentCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddSmartCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddTextCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignIdsList;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdMeaningfulGoalRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignMetrikaCounters;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignUnion;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaigns;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBannerHrefParams;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsAddBrandSurvey;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsDayBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsMinusKeywords;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsOrganization;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsPromotions;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsStartDate;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsTimeTargeting;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCampaignsWeeklyBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateContentPromotionCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCpmYndxFrontpageCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateDynamicCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateInternalAutobudgetCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateInternalDistribCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateInternalFreeCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateMcBannerCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateMeaningfulGoals;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateMobileContentCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateSmartCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateTextCampaign;
import ru.yandex.direct.grid.processing.model.common.GdMassUpdateAction;
import ru.yandex.direct.grid.processing.model.promoextension.GdUpdateCampaignsPromoExtension;
import ru.yandex.direct.grid.processing.service.campaign.uc.CampaignValidationServiceKotlinHelperKt;
import ru.yandex.direct.grid.processing.service.validation.GridDefectIds;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.ListUtils;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static org.joda.time.DateTimeConstants.DAYS_PER_WEEK;
import static org.joda.time.DateTimeConstants.HOURS_PER_DAY;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_RATE_CORRECTION;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_RATE_CORRECTION;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_SELECTED_HOURS_PER_WEEK;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_SELECTED_WORKING_HOURS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.metrikaCounterIsUnavailable;
import static ru.yandex.direct.feature.FeatureName.METRIKA_COUNTERS_ACCESS_VALIDATION_ON_SAVE_CAMPAIGN_ENABLED;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.invalidUnion;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResultIfErrors;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.checkFieldsContainNonNullValue;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_AUTOBUDGET_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_CONTENT_PROMOTION_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_DISTRIB_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_DYNAMIC_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_FREE_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_MCBANNER_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_MOBILE_CONTENT_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_SMART_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.ADD_AND_UPDATE_TEXT_CAMPAIGN_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.CampaignPathConverters.UPDATE_CAMPAIGN_METRIKA_COUNTERS_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.TimePathConverters.TIME_INTERVAL_PATH_CONVERTER;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validIntegerId;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.lessThan;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class CampaignValidationService {

    private static final Logger logger = LoggerFactory.getLogger(CampaignValidationService.class);

    public static final int MAX_CAMPAIGNS_COUNT_PER_UPDATE = 100;
    public static final int MAX_CAMPAIGNS_STRATEGY_COUNT_PER_UPDATE = 10;
    public static final int MAX_CAMPAIGNS_ORGANIZATION_COUNT_PER_UPDATE = 10;
    public static final Set<GdCampaignType> ALLOWED_TYPES_TO_UPDATE_STRATEGIES = Set.of(
            GdCampaignType.TEXT,
            GdCampaignType.DYNAMIC,
            GdCampaignType.MOBILE_CONTENT,
            GdCampaignType.MCBANNER,
            GdCampaignType.PERFORMANCE
    );

    //типы автостратегий где есть недельный бюджет и его можно отключить
    private final static Set<StrategyName> STRATEGIES_WITH_OPTIONAL_WEEKLY_BUDGET = Set.of(
            StrategyName.AUTOBUDGET_AVG_CLICK,
            StrategyName.AUTOBUDGET_AVG_CPA,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER,
            StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP,
            StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER,
            StrategyName.AUTOBUDGET_AVG_CPI,
            StrategyName.AUTOBUDGET_ROI,
            StrategyName.AUTOBUDGET_CRR);

    private static final Path UPDATE_CAMPAIGN_METRIKA_COUNTERS_PATH =
            path(field(GdUpdateCampaignMetrikaCounters.METRIKA_COUNTERS));
    private final GridValidationService gridValidationService;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final PathNodeConverterProvider skipPathNodeConverterProvider;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final FeatureService featureService;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;

    public CampaignValidationService(GridValidationService gridValidationService,
                                     CampMetrikaCountersService campMetrikaCountersService,
                                     FeatureService featureService, CampaignRepository campaignRepository,
                                     ShardHelper shardHelper) {
        this.gridValidationService = gridValidationService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.featureService = featureService;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(TextCampaign.class, ADD_AND_UPDATE_TEXT_CAMPAIGN_PATH_CONVERTER)
                .register(McBannerCampaign.class, ADD_AND_UPDATE_MCBANNER_CAMPAIGN_PATH_CONVERTER)
                .register(DynamicCampaign.class, ADD_AND_UPDATE_DYNAMIC_CAMPAIGN_PATH_CONVERTER)
                .register(SmartCampaign.class, ADD_AND_UPDATE_SMART_CAMPAIGN_PATH_CONVERTER)
                .register(MobileContentCampaign.class, ADD_AND_UPDATE_MOBILE_CONTENT_CAMPAIGN_PATH_CONVERTER)
                .register(ContentPromotionCampaign.class, ADD_AND_UPDATE_CONTENT_PROMOTION_CAMPAIGN_PATH_CONVERTER)
                .register(InternalAutobudgetCampaign.class, ADD_AND_UPDATE_AUTOBUDGET_CAMPAIGN_PATH_CONVERTER)
                .register(InternalDistribCampaign.class, ADD_AND_UPDATE_DISTRIB_CAMPAIGN_PATH_CONVERTER)
                .register(InternalFreeCampaign.class, ADD_AND_UPDATE_FREE_CAMPAIGN_PATH_CONVERTER)
                .register(TimeInterval.class, TIME_INTERVAL_PATH_CONVERTER)
                .register(UpdateCampMetrikaCountersRequest.class, UPDATE_CAMPAIGN_METRIKA_COUNTERS_PATH_CONVERTER)
                .build();
        SkipByDefaultMappingPathNodeConverter skipPathNodeConverter =
                SkipByDefaultMappingPathNodeConverter.emptyConverter();
        this.skipPathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(CampaignStub.class, skipPathNodeConverter)
                .register(TextCampaign.class, skipPathNodeConverter)
                .register(McBannerCampaign.class, skipPathNodeConverter)
                .register(DynamicCampaign.class, skipPathNodeConverter)
                .register(SmartCampaign.class, skipPathNodeConverter)
                .register(MobileContentCampaign.class, skipPathNodeConverter)
                .register(ContentPromotionCampaign.class, skipPathNodeConverter)
                .register(CpmPriceCampaign.class, skipPathNodeConverter)
                .register(CpmYndxFrontpageCampaign.class, skipPathNodeConverter)
                .register(CpmBannerCampaign.class, skipPathNodeConverter)
                .register(InternalAutobudgetCampaign.class, skipPathNodeConverter)
                .register(InternalDistribCampaign.class, skipPathNodeConverter)
                .register(InternalFreeCampaign.class, skipPathNodeConverter)
                .build();
    }

    /**
     * Проверяем, что объект GdUpdateCampaignUnion -- является "настоящим union", что означает,
     * что из всех полей, не пустое должно быть одно и только одно
     */
    private static Constraint<GdUpdateCampaignUnion, Defect> validateUpdateUnion() {
        return fromPredicate(union -> StreamEx.of(GdUpdateCampaignUnion.allModelProperties())
                        .map(x -> x.getRaw(union))
                        .nonNull()
                        .count() == 1,
                invalidUnion());
    }

    private static Constraint<GdAddCampaignUnion, Defect> validateAddUnion() {
        return fromPredicate(union -> StreamEx.of(GdAddCampaignUnion.allModelProperties())
                        .map(x -> x.getRaw(union))
                        .nonNull()
                        .count() == 1,
                invalidUnion());
    }

    private static boolean hasInvalidHoursNumberPerDay(List<List<Integer>> timeBoard) {
        return timeBoard.stream().anyMatch(day -> day.size() != HOURS_PER_DAY);
    }

    private static boolean hasInvalidCoefs(List<List<Integer>> timeBoard) {
        return timeBoard.stream().flatMap(List::stream)
                .anyMatch(coef -> coef < MIN_RATE_CORRECTION || coef > MAX_RATE_CORRECTION);
    }

    private static boolean hasLessThan40WorkingHoursSelected(List<List<Integer>> timeBoard) {
        return timeBoard
                .stream()
                .limit(5)
                .flatMap(List::stream)
                .filter(coef -> coef > MIN_RATE_CORRECTION)
                .count() < MIN_SELECTED_WORKING_HOURS;
    }

    private static boolean hasLessThen8HoursPerWeekSelected(List<List<Integer>> timeBoard) {
        return timeBoard
                .stream()
                .flatMap(List::stream)
                .filter(coef -> coef > MIN_RATE_CORRECTION)
                .count() < MIN_SELECTED_HOURS_PER_WEEK;
    }

    private static Constraint<List<List<Integer>>, Defect> validateTimeBoard(boolean isCampaignNewMinDaysLimitEnabled) {
        return fromPredicate(timeBoard -> timeBoard.size() == DAYS_PER_WEEK
                        && !hasInvalidHoursNumberPerDay(timeBoard)
                        && !hasInvalidCoefs(timeBoard)
                        && (isCampaignNewMinDaysLimitEnabled ?
                        !hasLessThen8HoursPerWeekSelected(timeBoard) :
                        !hasLessThan40WorkingHoursSelected(timeBoard)),
                new Defect<>(GridDefectIds.TimeTarget.INVALID_TIME_BOARD_FORMAT));
    }

    public static final Validator<GdHolidaysSettings, Defect> HOLIDAYS_SETTINGS_VALIDATOR =
            settings -> {
                ModelItemValidationBuilder<GdHolidaysSettings> vb = ModelItemValidationBuilder.of(settings);
                if (settings.getIsShow()) {
                    // startHour - zero-base, endHour — нет. Пара (0, 1) означает показы с 00.00 до 00.59.
                    // Значения не могут быть равны.
                    vb.item(GdHolidaysSettings.START_HOUR).check(notNull()).check(inRange(0, HOURS_PER_DAY - 1));
                    vb.item(GdHolidaysSettings.END_HOUR).check(notNull()).check(inRange(1, HOURS_PER_DAY));
                    vb.item(GdHolidaysSettings.RATE_CORRECTIONS)
                            .check(inRange(MIN_RATE_CORRECTION, MAX_RATE_CORRECTION), When.notNull());
                    if (settings.getStartHour() != null && settings.getEndHour() != null) {
                        vb.item(GdHolidaysSettings.START_HOUR).check(lessThan(settings.getEndHour()));
                    }
                }
                return vb.getResult();
            };

    private Validator<GdTimeTarget, Defect> getTimeTargetValidator(ClientId clientId) {
        final var isCampaignNewMinDaysLimitEnabled = this.featureService.isEnabledForClientId(clientId,
                FeatureName.CAMPAIGN_NEW_MIN_DAYS_LIMIT);

        return timeTarget -> {
            ModelItemValidationBuilder<GdTimeTarget> vb = ModelItemValidationBuilder.of(timeTarget);
            vb.item(GdTimeTarget.TIME_BOARD).check(validateTimeBoard(isCampaignNewMinDaysLimitEnabled), When.notNull());
            vb.item(GdTimeTarget.ID_TIME_ZONE).check(validId(), When.notNull());
            vb.item(GdTimeTarget.USE_WORKING_WEEKENDS).check(notNull());
            vb.item(GdTimeTarget.ENABLED_HOLIDAYS_MODE).check(notNull());
            vb.item(GdTimeTarget.HOLIDAYS_SETTINGS).checkBy(HOLIDAYS_SETTINGS_VALIDATOR,
                    When.isTrue(timeTarget.getEnabledHolidaysMode() != null && timeTarget.getEnabledHolidaysMode()));
            return vb.getResult();
        };
    }

    private Validator<GdUpdateTextCampaign, Defect> getUpdateTextCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateTextCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateTextCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateMcBannerCampaign, Defect> getUpdateMcbannerCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateMcBannerCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateMcBannerCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateDynamicCampaign, Defect> getUpdateDynamicCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateDynamicCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateDynamicCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateSmartCampaign, Defect> getUpdateSmartCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateSmartCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateSmartCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateMobileContentCampaign, Defect> getUpdateMobileContentCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateMobileContentCampaign> vb =
                    ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateMobileContentCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateContentPromotionCampaign, Defect> getUpdateContentPromotionCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateContentPromotionCampaign> vb =
                    ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateContentPromotionCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateInternalAutobudgetCampaign, Defect> getUpdateInternalAutobudgetCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateInternalAutobudgetCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateInternalAutobudgetCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateInternalDistribCampaign, Defect> getUpdateInternalDistribCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateInternalDistribCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateInternalDistribCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateInternalFreeCampaign, Defect> getUpdateInternalFreeCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateInternalFreeCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateInternalFreeCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCpmYndxFrontpageCampaign, Defect> getUpdateCpmYndxFrontpageCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdUpdateCpmYndxFrontpageCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdUpdateCpmYndxFrontpageCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddTextCampaign, Defect> getAddTextCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddTextCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddTextCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddMcBannerCampaign, Defect> getAddMcbannerCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddMcBannerCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddMcBannerCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddDynamicCampaign, Defect> getAddDynamicCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddDynamicCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddDynamicCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddSmartCampaign, Defect> getAddSmartCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddSmartCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddSmartCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddMobileContentCampaign, Defect> getAddMobileContentCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddMobileContentCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddMobileContentCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddContentPromotionCampaign, Defect> getAddContentPromotionCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddContentPromotionCampaign> vb =
                    ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddContentPromotionCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddInternalAutobudgetCampaign, Defect> getAddInternalAutobudgetCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddInternalAutobudgetCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddInternalAutobudgetCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddInternalDistribCampaign, Defect> getAddInternalDistribCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddInternalDistribCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddInternalDistribCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddInternalFreeCampaign, Defect> getAddInternalFreeCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddInternalFreeCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddInternalFreeCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddCpmYndxFrontpageCampaign, Defect> getAddCpmYndxFrontpageCampaignValidator(ClientId clientId) {
        return campaign -> {
            ModelItemValidationBuilder<GdAddCpmYndxFrontpageCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(GdAddCpmYndxFrontpageCampaign.TIME_TARGET).checkBy(this.getTimeTargetValidator(clientId),
                    When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaignUnion, Defect> getUpdateCampaignUnionValidator(ClientId clientId) {
        return req -> {
            ModelItemValidationBuilder<GdUpdateCampaignUnion> vb = ModelItemValidationBuilder.of(req);
            vb.check(validateUpdateUnion());
            vb.item(GdUpdateCampaignUnion.TEXT_CAMPAIGN).checkBy(this.getUpdateTextCampaignValidator(clientId),
                    When.notNull());
            vb.item(GdUpdateCampaignUnion.MC_BANNER_CAMPAIGN).checkBy(this.getUpdateMcbannerCampaignValidator(clientId), When.notNull());
            vb.item(GdUpdateCampaignUnion.DYNAMIC_CAMPAIGN).checkBy(this.getUpdateDynamicCampaignValidator(clientId),
                    When.notNull());
            vb.item(GdUpdateCampaignUnion.SMART_CAMPAIGN).checkBy(this.getUpdateSmartCampaignValidator(clientId),
                    When.notNull());
            vb.item(GdUpdateCampaignUnion.MOBILE_CONTENT_CAMPAIGN)
                    .checkBy(this.getUpdateMobileContentCampaignValidator(clientId), When.notNull());
            vb.item(GdUpdateCampaignUnion.CONTENT_PROMOTION_CAMPAIGN)
                    .checkBy(this.getUpdateContentPromotionCampaignValidator(clientId), When.notNull());
            vb.item(GdUpdateCampaignUnion.INTERNAL_AUTOBUDGET_CAMPAIGN)
                    .checkBy(this.getUpdateInternalAutobudgetCampaignValidator(clientId), When.notNull());
            vb.item(GdUpdateCampaignUnion.INTERNAL_DISTRIB_CAMPAIGN)
                    .checkBy(this.getUpdateInternalDistribCampaignValidator(clientId), When.notNull());
            vb.item(GdUpdateCampaignUnion.INTERNAL_FREE_CAMPAIGN)
                    .checkBy(this.getUpdateInternalFreeCampaignValidator(clientId), When.notNull());
            vb.item(GdUpdateCampaignUnion.CPM_YNDX_FRONTPAGE_CAMPAIGN)
                    .checkBy(this.getUpdateCpmYndxFrontpageCampaignValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdAddCampaignUnion, Defect> getAddCampaignUnionValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdAddCampaignUnion, Defect> vb = ModelItemValidationBuilder.of(req);
            vb.check(validateAddUnion());
            vb.item(req.getTextCampaign(), GdAddCampaignUnion.TEXT_CAMPAIGN.name())
                    .checkBy(this.getAddTextCampaignValidator(clientId), When.notNull());
            vb.item(req.getMcBannerCampaign(), GdAddCampaignUnion.MC_BANNER_CAMPAIGN.name())
                    .checkBy(this.getAddMcbannerCampaignValidator(clientId), When.notNull());
            vb.item(req.getDynamicCampaign(), GdAddCampaignUnion.DYNAMIC_CAMPAIGN.name())
                    .checkBy(this.getAddDynamicCampaignValidator(clientId), When.notNull());
            vb.item(req.getSmartCampaign(), GdAddCampaignUnion.SMART_CAMPAIGN.name())
                    .checkBy(this.getAddSmartCampaignValidator(clientId), When.notNull());
            vb.item(req.getMobileContentCampaign(), GdAddCampaignUnion.MOBILE_CONTENT_CAMPAIGN.name())
                    .checkBy(this.getAddMobileContentCampaignValidator(clientId), When.notNull());
            vb.item(req.getContentPromotionCampaign(), GdAddCampaignUnion.CONTENT_PROMOTION_CAMPAIGN.name())
                    .checkBy(this.getAddContentPromotionCampaignValidator(clientId), When.notNull());
            vb.item(req.getInternalAutobudgetCampaign(), GdAddCampaignUnion.INTERNAL_AUTOBUDGET_CAMPAIGN.name())
                    .checkBy(this.getAddInternalAutobudgetCampaignValidator(clientId), When.notNull());
            vb.item(req.getInternalDistribCampaign(), GdAddCampaignUnion.INTERNAL_DISTRIB_CAMPAIGN.name())
                    .checkBy(this.getAddInternalDistribCampaignValidator(clientId), When.notNull());
            vb.item(req.getInternalFreeCampaign(), GdAddCampaignUnion.INTERNAL_FREE_CAMPAIGN.name())
                    .checkBy(this.getAddInternalFreeCampaignValidator(clientId), When.notNull());
            vb.item(req.getCpmYndxFrontpageCampaign(), GdAddCampaignUnion.CPM_YNDX_FRONTPAGE_CAMPAIGN.name())
                    .checkBy(this.getAddCpmYndxFrontpageCampaignValidator(clientId), When.notNull());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaigns, Defect> getUpdateCampaignsValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaigns, Defect> lvb = ModelItemValidationBuilder.of(req);
            lvb.list(req.getCampaignUpdateItems(), GdUpdateCampaigns.CAMPAIGN_UPDATE_ITEMS.name())
                    .checkEachBy(this.getUpdateCampaignUnionValidator(clientId));
            return lvb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsStartDate, Defect> updateCampaignsStartDateValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaignsStartDate, Defect> vb = ModelItemValidationBuilder.of(req);
            vb.list(req.getCampaignIds(), GdUpdateCampaignsStartDate.CAMPAIGN_IDS.name())
                    .check(notEmptyCollection())
                    .check(maxListSize(MAX_CAMPAIGNS_COUNT_PER_UPDATE))
                    .checkEach(validId());
            vb.check(checkFieldsContainNonNullValue(ImmutableList.of(GdUpdateCampaignsStartDate.START_DATE,
                    GdUpdateCampaignsStartDate.FINISH_DATE)));
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsDayBudget, Defect> updateCampaignsDayBudgetValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaignsDayBudget, Defect> vb = ModelItemValidationBuilder.of(req);
            vb.list(req.getCampaignIds(), GdUpdateCampaignsDayBudget.CAMPAIGN_IDS.name())
                    .check(notEmptyCollection())
                    .check(maxListSize(MAX_CAMPAIGNS_COUNT_PER_UPDATE))
                    .checkEach(unique())
                    .checkEach(validId());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsMinusKeywords, Defect> updateCampaignsMinusKeywordsValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaignsMinusKeywords, Defect> vb = ModelItemValidationBuilder.of(req);
            vb.list(req.getCampaignIds(), GdUpdateCampaignsMinusKeywords.CAMPAIGN_IDS.name())
                    .check(notEmptyCollection())
                    .check(maxListSize(MAX_CAMPAIGNS_COUNT_PER_UPDATE))
                    .checkEach(validId());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsOrganization, Defect> updateCampaignsOrganizationValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaignsOrganization, Defect> vb = ModelItemValidationBuilder.of(req);
            vb.list(req.getCampaignIds(), GdUpdateCampaignsOrganization.CAMPAIGN_IDS.name())
                    .check(maxListSize(MAX_CAMPAIGNS_ORGANIZATION_COUNT_PER_UPDATE))
                    .check(notEmptyCollection())
                    .checkEach(validId());
            vb.item(req.getDefaultPermalinkId(),
                            CampaignWithOrganizationAndPhone.DEFAULT_PERMALINK_ID.name())
                    .check(validId(), When.notNull());
            vb.item(req.getDefaultChainId(),
                            CampaignWithOrganizationAndPhone.DEFAULT_CHAIN_ID.name())
                    .check(validId(), When.notNull());
            vb.item(req.getDefaultTrackingPhoneId(),
                            CampaignWithOrganizationAndPhone.DEFAULT_TRACKING_PHONE_ID.name())
                    .check(validId(), When.notNull());
            return vb.getResult();
        };
    }

    private static Validator<GdCpmCampaignDayBudgetLimitsRequest, Defect> budgetLimitRequestValidator(
            Map<Long, CpmCampaignWithCustomStrategy> actualCampaignById, LocalDate now) {
        return req -> {
            var lvb = ModelItemValidationBuilder.of(req);
            lvb.list(GdCpmCampaignDayBudgetLimitsRequest.ITEMS)
                    .checkEachBy(budgetLimitItemRequestValidator(actualCampaignById, now));

            return lvb.getResult();
        };
    }

    private static Validator<GdCpmCampaignDayBudgetLimitsRequestItem, Defect> budgetLimitItemRequestValidator(
            Map<Long, CpmCampaignWithCustomStrategy> actualCampaignById, LocalDate now) {
        return req -> {
            var vb = ModelItemValidationBuilder.of(req);
            vb.item(GdCpmCampaignDayBudgetLimitsRequestItem.CAMPAIGN_ID)
                    .check(validId(), When.isFalse(req.getIsRestarting()))
                    .check(inSet(actualCampaignById.keySet()), When.isFalse(req.getIsRestarting()));
            vb.item(GdCpmCampaignDayBudgetLimitsRequestItem.FINISH_DATE)
                    .check(isNotBeforeThan(now), When.isFalse(req.getIsRestarting()));

            if (actualCampaignById.containsKey(req.getCampaignId())) {
                LocalDate actualStart =
                        actualCampaignById.get(req.getCampaignId()).getStrategy().getStrategyData().getStart();

                vb.item(GdCpmCampaignDayBudgetLimitsRequestItem.START_DATE)
                        .check(isEqual(actualStart, CommonDefects.invalidValue()));
            }

            return vb.getResult();
        };
    }

    private Validator<GdAddCampaigns, Defect> getAddCampaignsValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdAddCampaigns, Defect> lvb = ModelItemValidationBuilder.of(req);
            lvb.list(req.getCampaignAddItems(), GdAddCampaigns.CAMPAIGN_ADD_ITEMS.name())
                    .checkEachBy(this.getAddCampaignUnionValidator(clientId));
            return lvb.getResult();
        };
    }

    private static final Validator<GdCampaignIdsList, Defect> CAMPAIGN_IDS_VALIDATOR =
            req -> {
                ItemValidationBuilder<GdCampaignIdsList, Defect> lvb = ModelItemValidationBuilder.of(req);
                lvb.list(req.getCampaignIds(), GdCampaignIdsList.CAMPAIGN_IDS.name())
                        .check(notEmptyCollection())
                        .checkEach(validId());
                return lvb.getResult();
            };

    private Validator<GdCampaignIdsList, Defect> deleteBrandLifValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdCampaignIdsList, Defect> lvb =
                    ModelItemValidationBuilder.of(req);
            lvb.list(req.getCampaignIds(), GdCampaignIdsList.CAMPAIGN_IDS.name())
                    .check(notEmptyCollection())
                    .checkEach(validId());
            return lvb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsAddBannerHrefParams, Defect> updateBannerHrefParamsValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaignsAddBannerHrefParams, Defect> lvb =
                    ModelItemValidationBuilder.of(req);
            lvb.list(req.getCampaignIds(), GdUpdateCampaignsAddBannerHrefParams.CAMPAIGN_IDS.name())
                    .check(notEmptyCollection())
                    .checkEach(validId());
            return lvb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsAddBrandSurvey, Defect> updateAddBrandSurveyValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaignsAddBrandSurvey, Defect> lvb = ModelItemValidationBuilder.of(req);
            lvb.list(req.getCampaignIds(), GdUpdateCampaignsAddBrandSurvey.CAMPAIGN_IDS.name())
                    .check(notEmptyCollection())
                    .checkEach(validId());
            return lvb.getResult();
        };
    }

    private Validator<GdUpdateCampaignMetrikaCounters, Defect> updateCampaignMetrikaCountersValidator(ClientId clientId) {
        return req -> {
            ModelItemValidationBuilder<GdUpdateCampaignMetrikaCounters> lvb = ModelItemValidationBuilder.of(req);
            lvb.list(GdUpdateCampaignMetrikaCounters.CAMPAIGN_IDS)
                    .check(notEmptyCollection())
                    .check(maxListSize(MAX_CAMPAIGNS_COUNT_PER_UPDATE))
                    .checkEach(validId());

            lvb.list(GdUpdateCampaignMetrikaCounters.METRIKA_COUNTERS)
                    .check(notEmptyCollection())
                    .checkEach(validIntegerId());
            return lvb.getResult();
        };
    }

    private static final Validator<GdMeaningfulGoalRequest, Defect> MEANINGFUL_GOAL_REQUEST_VALIDATOR =
            goal -> {
                ModelItemValidationBuilder<GdMeaningfulGoalRequest> vb = ModelItemValidationBuilder.of(goal);
                vb.item(GdMeaningfulGoalRequest.GOAL_ID)
                        .check(validId());
                return vb.getResult();
            };

    private static final Validator<GdUpdateMeaningfulGoals, Defect> UPDATE_MEANINGFUL_GOALS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdUpdateMeaningfulGoals> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdUpdateMeaningfulGoals.CAMPAIGN_IDS)
                        .check(notEmptyCollection())
                        .check(maxListSize(MAX_CAMPAIGNS_COUNT_PER_UPDATE))
                        .checkEach(unique())
                        .checkEach(validId());
                vb.list(GdUpdateMeaningfulGoals.MEANINGFUL_GOALS)
                        .checkEachBy(MEANINGFUL_GOAL_REQUEST_VALIDATOR);
                return vb.getResult();
            };

    private Validator<GdUpdateCampaignsTimeTargeting, Defect> getUpdateTimeTargetingValidator(ClientId clientId) {
        return req -> {
            ModelItemValidationBuilder<GdUpdateCampaignsTimeTargeting> vb = ModelItemValidationBuilder.of(req);

            vb.list(GdUpdateCampaignsTimeTargeting.CAMPAIGN_IDS)
                    .check(notEmptyCollection())
                    .check(maxListSize(MAX_CAMPAIGNS_COUNT_PER_UPDATE))
                    .checkEach(unique())
                    .checkEach(validId());

            vb.item(GdUpdateCampaignsTimeTargeting.TIME_TARGET)
                    .checkBy(this.getTimeTargetValidator(clientId));
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsStrategy, Defect> updateCampaignsStrategyValidator(ClientId clientId) {
        return req -> {
            ModelItemValidationBuilder<GdUpdateCampaignsStrategy> vb = ModelItemValidationBuilder.of(req);

            vb.list(GdUpdateCampaignsStrategy.CAMPAIGN_IDS)
                    .check(notEmptyCollection())
                    .checkEach(unique())
                    .check(maxListSize(MAX_CAMPAIGNS_STRATEGY_COUNT_PER_UPDATE))
                    .checkEach(validId());
            vb.item(GdUpdateCampaignsStrategy.TYPE)
                    .check(fromPredicate(ALLOWED_TYPES_TO_UPDATE_STRATEGIES::contains, campaignTypeNotSupported()));
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsPromotions, Defect> updateCampaignsPromotionsValidator(ClientId clientId) {
        return req -> {
            ItemValidationBuilder<GdUpdateCampaignsPromotions, Defect> vb = ModelItemValidationBuilder.of(req);
            vb.list(req.getCampaignIds(), GdUpdateCampaignsPromotions.CAMPAIGN_IDS.name())
                    .check(notEmptyCollection())
                    .check(maxListSize(MAX_CAMPAIGNS_COUNT_PER_UPDATE))
                    .checkEach(unique())
                    .checkEach(validId());
            return vb.getResult();
        };
    }

    private Validator<GdUpdateCampaignsWeeklyBudget, Defect> updateCampaignsWeeklyBudgetValidator(ClientId clientId) {
        return req -> {
            ModelItemValidationBuilder<GdUpdateCampaignsWeeklyBudget> vb = ModelItemValidationBuilder.of(req);

            vb.list(GdUpdateCampaignsWeeklyBudget.CAMPAIGN_IDS)
                    .check(notEmptyCollection())
                    .checkEach(unique())
                    .check(maxListSize(MAX_CAMPAIGNS_STRATEGY_COUNT_PER_UPDATE))
                    .checkEach(validId());
            return vb.getResult();
        };
    }

    void validateUpdateCampaigns(ClientId clientId, GdUpdateCampaigns updateCampaigns) {
        gridValidationService.applyValidator(this.getUpdateCampaignsValidator(clientId), updateCampaigns, false);
    }

    void validateUpdateCampaignsStartDate(GdUpdateCampaignsStartDate updateCampaignsStartDate,
                                          ClientId clientId) {
        gridValidationService.applyValidator(this.updateCampaignsStartDateValidator(clientId),
                updateCampaignsStartDate, false);
    }

    void validateUpdateCampaignsWithPromoExtension(GdUpdateCampaignsPromoExtension updateCampaignsWithPromoExtension,
                                                   ClientId clientId) {
        CampaignValidationServiceKotlinHelperKt.validateUpdateCampaignsWithPromoExtension(
                gridValidationService,
                updateCampaignsWithPromoExtension,
                campaignRepository.getCampaignsTypeMap(
                        shardHelper.getShardByClientId(clientId),
                        updateCampaignsWithPromoExtension.getCampaignIds())
        );
    }

    public void validateUpdateCampaignsDayBudget(GdUpdateCampaignsDayBudget updateCampaignsDayBudget,
                                                 ClientId clientId) {
        gridValidationService.applyValidator(this.updateCampaignsDayBudgetValidator(clientId),
                updateCampaignsDayBudget, false);
    }

    void validateUpdateCampaignsMinusKeywords(GdUpdateCampaignsMinusKeywords updateCampaignsMinusKeywords,
                                              ClientId clientId) {
        gridValidationService.applyValidator(this.updateCampaignsMinusKeywordsValidator(clientId),
                updateCampaignsMinusKeywords, false);
    }

    void validateUpdateCampaignsOrganization(GdUpdateCampaignsOrganization updateCampaignsOrganization,
                                             ClientId clientId) {
        gridValidationService.applyValidator(this.updateCampaignsOrganizationValidator(clientId),
                updateCampaignsOrganization,
                false);
    }

    void validateAddCampaigns(ClientId clientId, GdAddCampaigns updateCampaigns) {
        gridValidationService.applyValidator(this.getAddCampaignsValidator(clientId), updateCampaigns, false);
    }

    void validateDeleteCampaigns(GdCampaignIdsList updateCampaigns) {
        gridValidationService.applyValidator(CAMPAIGN_IDS_VALIDATOR, updateCampaigns, false);
    }

    void validateDeleteBrandLift(GdCampaignIdsList updateCampaigns, ClientId clientId) {
        gridValidationService.applyValidator(this.deleteBrandLifValidator(clientId), updateCampaigns, false);
    }

    void validateCampaignBudgetLimitRequest(GdCpmCampaignDayBudgetLimitsRequest campaignDayBudgetLimitsRequest,
                                            Map<Long, CpmCampaignWithCustomStrategy> actualCampaignById) {
        LocalDate now = LocalDate.now();
        gridValidationService.applyValidator(budgetLimitRequestValidator(actualCampaignById, now),
                campaignDayBudgetLimitsRequest, false);
    }

    public void validateUpdateCampaignMetrikaCountersRequest(GdUpdateCampaignMetrikaCounters updateMetrikaCounters,
                                                             ClientId clientId) {
        gridValidationService.applyValidator(this.updateCampaignMetrikaCountersValidator(clientId),
                updateMetrikaCounters, false);
    }

    public void validateUpdateMeaningfulGoalsRequest(GdUpdateMeaningfulGoals input) {
        gridValidationService.applyValidator(UPDATE_MEANINGFUL_GOALS_VALIDATOR, input, false);
    }

    public void validateUpdateTimeTargetingRequest(ClientId clientId, GdUpdateCampaignsTimeTargeting input) {
        gridValidationService.applyValidator(this.getUpdateTimeTargetingValidator(clientId), input, false);
    }

    public void validateUpdateCampaignsStrategyRequest(GdUpdateCampaignsStrategy input, ClientId clientId) {
        gridValidationService.applyValidator(this.updateCampaignsStrategyValidator(clientId), input, false);
    }

    public void validateUpdateCampaignsWeeklyBudgetRequest(GdUpdateCampaignsWeeklyBudget input,
                                                           ClientId clientId) {
        gridValidationService.applyValidator(this.updateCampaignsWeeklyBudgetValidator(clientId), input, false);
    }

    public void validateUpdateCampaignsPromotions(GdUpdateCampaignsPromotions updateCampaignsPromotions,
                                                  ClientId clientId) {
        gridValidationService.applyValidator(
                this.updateCampaignsPromotionsValidator(clientId), updateCampaignsPromotions, false);
    }

    public void validateAddBrandSurveyParams(ClientId clientId,
                                             GdUpdateCampaignsAddBrandSurvey addBrandSurvey) {
        gridValidationService.applyValidator(this.updateAddBrandSurveyValidator(clientId), addBrandSurvey, false);
    }

    public void validateUpdateBannerHrefParams(ClientId clientId,
                                               GdUpdateCampaignsAddBannerHrefParams updateBannerHrefParams) {
        gridValidationService.applyValidator(
                this.updateBannerHrefParamsValidator(clientId), updateBannerHrefParams, false);
    }

    @Nullable
    public GdValidationResult validateUpdateCampaignMetrikaCountersAccess(ClientId clientId,
                                                                          GdUpdateCampaignMetrikaCounters input) {
        // если удаляем счетчики или если фича на проверку не включена, то валидировать доступ на счетчики не надо
        if (input.getAction() == GdMassUpdateAction.REMOVE || !featureService
                .isEnabledForClientId(clientId, METRIKA_COUNTERS_ACCESS_VALIDATION_ON_SAVE_CAMPAIGN_ENABLED)) {
            return null;
        }

        ListValidationBuilder<Integer, Defect> vb = ListValidationBuilder.of(input.getMetrikaCounters());
        Set<Integer> availableCounterIds;
        try {
            var counterIds = mapList(input.getMetrikaCounters(), Integer::longValue);
            Set<Long> availableMetrikaCounterIds =
                    campMetrikaCountersService.getAvailableCounterIdsByClientId(clientId, counterIds);
            availableCounterIds = ListUtils.longToIntegerSet(availableMetrikaCounterIds);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            vb.check(metrikaReturnsResultWithErrors(), When.notNull());
            return buildGridValidationResultIfErrors(vb.getResult(), UPDATE_CAMPAIGN_METRIKA_COUNTERS_PATH,
                    pathNodeConverterProvider);
        }

        vb.checkEach(inSet(availableCounterIds), metrikaCounterIsUnavailable());

        return buildGridValidationResultIfErrors(vb.getResult(), UPDATE_CAMPAIGN_METRIKA_COUNTERS_PATH,
                pathNodeConverterProvider);
    }

    @Nullable
    public GdValidationResult validateUpdateWeeklyBudget(List<Campaign> campaigns,
                                                         GdUpdateCampaignsWeeklyBudget input) {
        Map<Long, Campaign> campaignsById = listToMap(campaigns, Campaign::getId);
        ListValidationBuilder<Long, Defect> vb = ListValidationBuilder.of(input.getCampaignIds());
        var vr = vb.checkEach(checkUpdateWeeklyBudgetAvailability(campaignsById),
                        When.isValid())
                .checkEach(checkDisableWeeklyBudgetAvailability(campaignsById),
                        When.isTrue(NumberUtils.isNullOrZero(input.getWeeklyBudget())))
                .getResult();

        return buildGridValidationResultIfErrors(vr,
                path(field(GdUpdateCampaignsWeeklyBudget.CAMPAIGN_IDS)),
                pathNodeConverterProvider);
    }

    private static Constraint<Long, Defect> checkUpdateWeeklyBudgetAvailability(Map<Long, Campaign> campaignsById) {
        return id -> {
            var campaign = (Campaign) campaignsById.get(id);
            if (campaign == null) {
                return CampaignDefects.campaignNotFound();
            } else if (!campaign.getAutobudget()) {
                return StrategyDefects.weekBudgetIsNotSupported();
            } else if (isEditWeeklyBudgetForbidden(campaign.getAutobudget(), campaign.getType(),
                    CampaignMetatype.ECOM == campaign.getMetatype())) {
                return CampaignDefects.campaignTypeNotSupported();
            }

            return null;
        };
    }

    private static Constraint<Long, Defect> checkDisableWeeklyBudgetAvailability(Map<Long, Campaign> campaignsById) {
        return id -> {
            var campaign = (Campaign) campaignsById.get(id);
            if (isDisableWeeklyBudgetForbidden(campaign.getSource(), campaign.getStrategy().getStrategyName())) {
                return StrategyDefects.weekBudgetIsMandatory();
            }

            return null;
        };
    }

    public void validateResetCampaignFlightStatusApprove(GdCampaignIdsList campaignIds) {
        gridValidationService.applyValidator(CAMPAIGN_IDS_VALIDATOR, campaignIds, false);
    }

    @Nullable
    public GdValidationResult getSkipPathValidationResult(MassResult<Long> result, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(result.getValidationResult(), path, skipPathNodeConverterProvider);
    }

    @Nullable
    public GdValidationResult getValidationResult(MassResult<Long> result, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(result.getValidationResult(), path, pathNodeConverterProvider);
    }

    @Nullable
    public GdValidationResult getValidationResult(ValidationResult<?, Defect> validationResult, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(validationResult, path, pathNodeConverterProvider);
    }

    /**
     * запрещено ли редактирование недельного бюджета с гридов
     */
    public static boolean isEditWeeklyBudgetForbidden(boolean isAutobudget, CampaignType campaignType,
                                                      boolean isGoodsCampaign) {
        return !isAutobudget || // недельный бюджет это свойство стратегии автобюджета
                isGoodsCampaign || //для товарных кампаний запрещено
                campaignType == CampaignType.CPM_BANNER ||   //для любых охватных запрещено
                campaignType == CampaignType.CPM_PRICE ||
                campaignType == CampaignType.CPM_DEALS ||
                campaignType == CampaignType.CPM_YNDX_FRONTPAGE;
    }

    /**
     * Запрещено ли полностью отключать настройку недельного бюджета
     */
    public static boolean isDisableWeeklyBudgetForbidden(@Nullable CampaignSource source,
                                                         @Nullable StrategyName strategyName) {
        return AvailableCampaignSources.INSTANCE.isUC(source) || strategyName == null ||
                !STRATEGIES_WITH_OPTIONAL_WEEKLY_BUDGET.contains(strategyName);
    }
}
